﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {
    const int MemoryHeapSize = 32 * 1024 * 1024;

    nnt::atk::util::FsCommonSetup g_FsSetup;
    nnt::atk::util::AtkCommonSetup g_AtkSetup;

    char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;

    const unsigned int TestData[] = { SEQ_MARIOKART, SE_YOSHI, STRM_MARIOKART };
}

void LoadData()
{
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
}

void UpdateAndWaitForStrmPrepare(nn::atk::SoundHandle& soundHandle)
{
    EXPECT_TRUE(nn::atk::SoundSystem::GetVoiceCount() == 0);
    EXPECT_FALSE(soundHandle.IsPrepared());

    // ストリームサウンドの場合は、サウンドアーカイブプレイヤーで再生・再生準備をしても、すぐに再生準備が完了しない。
    // 一旦サウンドアーカイブプレイヤーを Update() して再生準備のコマンドを送り、再生準備が完了するのを待つ必要がある。
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

    const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(2000);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

    for (auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
        nn::atk::SoundSystem::VoiceCommandProcess(6);
        nn::os::SleepThread(WaitTime);
        if (soundHandle.IsPrepared())
        {
            break;
        }
    }
    // VoiceCommand が積まれているため発音数が切り替わる
    EXPECT_TRUE(nn::atk::SoundSystem::GetVoiceCount() != 0);
    EXPECT_TRUE(soundHandle.IsPrepared()) << "Preparing stream sound timed out.";
}

void UpdateAndWaitForStrmStart(nn::atk::StreamSoundHandle& streamSoundHandle)
{
    // 実際に再生が行われていることをパラメータ上で確認できるまで待つ
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

    const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(2000);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

    for (auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
        nn::atk::SoundSystem::VoiceCommandProcess(6);
        nn::os::SleepThread(WaitTime);
        if (streamSoundHandle.GetPlaySamplePosition() != 0)
        {
            break;
        }
    }
    EXPECT_TRUE(streamSoundHandle.GetPlaySamplePosition() != 0) << "Start stream sound timed out.";
}

TEST(VoiceCommandProcess, StartStopTest)
{
    for (auto& testData: TestData)
    {
        // Start は直後から SoundHandle に反映される
        const nn::TimeSpan TestTime = nn::TimeSpan::FromMilliSeconds(200);
        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();
        g_AtkSetup.Initialize(g_Allocator);
        LoadData();

        // テスト開始
        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

        EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, testData).IsSuccess());
        if (testData == STRM_MARIOKART)
        {
            UpdateAndWaitForStrmPrepare(soundHandle);
        }

        EXPECT_TRUE(soundHandle.IsAttachedSound());
        EXPECT_TRUE(soundHandle.IsPrepared());
        soundArchivePlayer.Update();

        soundHandle.StartPrepared();
        nn::os::SleepThread(WaitTime);

        // VoiceCommand 処理が行われなければ再生処理は行われない
        // STRM_MARIOKART はすでに VoiceCommand 処理を行っているのでスキップ
        if (testData != STRM_MARIOKART)
        {
            for (nn::TimeSpan elapsed = 0; elapsed <= TestTime; elapsed += WaitTime)
            {
                soundArchivePlayer.Update();
                nn::os::SleepThread(WaitTime);
                EXPECT_TRUE(nn::atk::SoundSystem::GetVoiceCount() == 0);
            }

            // VoiceCommand が積まれた瞬間に発音数が切り替わる
            ASSERT_EQ(0, nnt::atk::util::GetVoiceCommandListCount());
            nn::atk::SoundSystem::VoiceCommandProcess(1);
            EXPECT_TRUE(nn::atk::SoundSystem::GetVoiceCount() != 0);
        }

        soundHandle.Stop(0);
        EXPECT_TRUE(nn::atk::SoundSystem::GetVoiceCount() != 0);

        // VoiceCommand 処理が行われなければ停止処理は行われない
        for (nn::TimeSpan elapsed = 0; elapsed <= TestTime; elapsed += WaitTime)
        {
            soundArchivePlayer.Update();
            nn::os::SleepThread(WaitTime);
            EXPECT_TRUE(nn::atk::SoundSystem::GetVoiceCount() != 0);
        }

        const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(10000);
        bool isSoundStopped = false;
        // VoiceCommand 処理により停止処理が行われたのち、音源のリリース処理が完了すれば完全に停止する
        // テスト中一番長いリリースは、 SEQ_MARIOKART のバンク BANK_BGM が使う INST_006 で、
        // Release 値が 98 (最大 2109 ms 相当のリリース時間) となっている。
        for (nn::TimeSpan elapsed = 0; elapsed <= TimeOut; elapsed += WaitTime)
        {
            nn::atk::SoundSystem::VoiceCommandProcess(6);
            soundArchivePlayer.Update();
            nn::os::SleepThread(WaitTime);
            if (nn::atk::SoundSystem::GetVoiceCount() == 0)
            {
                isSoundStopped = true;
                break;
            }
        }
        EXPECT_TRUE(isSoundStopped) << "Stop sound timed out.";

        // 終了処理
        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();

        g_Allocator.Finalize();
    }
}

TEST(VoiceCommandProcess, StreamLackOfCommand)
{
    // Start は直後から SoundHandle に反映される
    const nn::TimeSpan TestTime = nn::TimeSpan::FromMilliSeconds(5000);

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    UpdateAndWaitForStrmPrepare(soundHandle);
    UpdateAndWaitForStrmStart(streamSoundHandle);

    int64_t startStreamSoundPosition = streamSoundHandle.GetPlaySamplePosition();

    // Voice Command を枯渇させ、 Loading Delay を意図的に起こす
    nn::os::SleepThread(TestTime);
    // Loading Delay が atk のパラメータに反映されるタイミングは VoiceCommandProcess 直後
    ASSERT_EQ(0, nnt::atk::util::GetVoiceCommandListCount());
    nn::atk::SoundSystem::VoiceCommandProcess(1);
    EXPECT_TRUE(streamSoundHandle.IsSuspendByLoadingDelay());

    // Loading Delay による停止が起きたとき、再生位置が再生直後に取得した値より進み、
    // 再生が完了したブロックの一番最後のバッファの終端位置にあることを確認する
    EXPECT_GT(streamSoundHandle.GetPlaySamplePosition(), startStreamSoundPosition);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(VoiceCommandProcess, StreamLackOfCommandWithLoadingDataCompleted)
{
    // Start は直後から SoundHandle に反映される
    const nn::TimeSpan TestTime = nn::TimeSpan::FromMilliSeconds(20000);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(1200);

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART).IsSuccess());
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    UpdateAndWaitForStrmPrepare(soundHandle);
    UpdateAndWaitForStrmStart(streamSoundHandle);

    bool isSuspended = false;

    // Voice Command を枯渇させ、 Loading Delay を意図的に起こす
    for (nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTime; elapsed += WaitTime)
    {
        nn::atk::SoundSystem::VoiceCommandProcess(1);

        nn::os::SleepThread(WaitTime);

        if (streamSoundHandle.IsSuspendByLoadingDelay())
        {
            isSuspended = true;
            break;
        }
    }

    EXPECT_TRUE(isSuspended) << "Stream sound is not suspended.";

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(VoiceCommandProcess, UpdatePeriodChange)
{
    const int VoiceCommandUpdatePeriod[] = { 5, 16, 33 };
    const int VoiceCommandNumber[] = { 1, 4, 7 };
    const int SoundArchivePlayerUpdatePeriod[] = { 5, 16, 33 };

    for (const int voiceCommandUpdatePeriod : VoiceCommandUpdatePeriod)
    {
        for (const int voiceCommandNumber : VoiceCommandNumber)
        {
            for (const int soundArchivePlayerUpdatePeriod : SoundArchivePlayerUpdatePeriod)
            {
                g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
                g_FsSetup.Initialize();
                g_AtkSetup.Initialize(g_Allocator);
                LoadData();

                NN_LOG("VoiceCommandUpdate = %d [ms], VoiceCommandNumber = %d, SoundArchivePlayerUpdate = %d [ms]\n"
                        , voiceCommandUpdatePeriod
                        , voiceCommandNumber
                        , soundArchivePlayerUpdatePeriod);

                nn::atk::SoundHandle seqMarioKartHandle;
                nn::atk::SoundHandle seYoshiHandle;
                nn::atk::SoundHandle strmMarioKartHandle;
                nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

                EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
                NN_LOG("Start SEQ_MARIOKART.\n");
                EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
                NN_LOG("Start SE_YOSHI.\n");
                EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
                NN_LOG("Start STRM_MARIOKART.\n");

                const int TestTimeMilliSeconds = 5000;
                const nn::TimeSpan TestTime = nn::TimeSpan::FromMilliSeconds(TestTimeMilliSeconds);
                const int WaitTimeMilliSeconds = 1;
                const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(WaitTimeMilliSeconds);

                for (nn::TimeSpan elapsed = 0; elapsed <= TestTime; elapsed += WaitTime)
                {
                    if (elapsed.GetMilliSeconds() % soundArchivePlayerUpdatePeriod == 0)
                    {
                        soundArchivePlayer.Update();
                    }
                    if (elapsed.GetMilliSeconds() % voiceCommandUpdatePeriod == 0)
                    {
                        nn::atk::SoundSystem::VoiceCommandProcess(voiceCommandNumber);
                    }

                    // SE_YOSHI は最後まで回したら再度再生
                    if (!seYoshiHandle.IsAttachedSound())
                    {
                        seYoshiHandle.Stop(0);
                        EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
                    }
                    nn::os::SleepThread(WaitTime);
                }

                g_AtkSetup.Finalize(g_Allocator);
                g_FsSetup.Finalize();
                g_Allocator.Finalize();
            }
        }
    }
}

void ExhaustDriverCommandBuffer() NN_NOEXCEPT
{
    // その場でコマンドプッシュを行う SetAuxBusVolume を連打して、バッファを枯渇直前にする
    while ( nn::atk::SoundSystem::GetAllocatableDriverCommandSize() > sizeof(nn::atk::detail::DriverCommandAuxBusVolume) )
    {
        nn::atk::SoundSystem::SetAuxBusVolume(nn::atk::AuxBus_A, 1.0f, 0);
    }
}

void DumpDriverCommandBuffer() NN_NOEXCEPT
{
    NN_LOG("DriverCommand cnt:%4d, allocated:%8zd, allocatable:%8zd, total:%8zd\n"
        , nn::atk::SoundSystem::GetAllocatedDriverCommandCount()
        , nn::atk::SoundSystem::GetAllocatedDriverCommandBufferSize()
        , nn::atk::SoundSystem::GetAllocatableDriverCommandSize()
        , nn::atk::SoundSystem::GetDriverCommandBufferSize()
    );
}

TEST(VoiceCommandProcess, ExhaustionOfCommandBuffer)
{
    // ドライバーコマンドバッファ枯渇時に問題なく動作するかを確認するテスト
    const nn::TimeSpan TestTime = nn::TimeSpan::FromMinutes(2);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    // STRM 再生の開始ゲームフレーム、フレーム間隔
    const int beginningFrameToPlayStrm = 2;
    const int intervalFrameToPlayStrm  = 600;
    int frameCountToPlayStrm           = 0;
    nn::atk::SoundHandle strmSoundHandle;
    // WSD 再生の開始ゲームフレーム、フレーム間隔
    const int beginningFrameToPlayWsd  = 5;
    const int intervalFrameToPlayWsd   = 300;
    int frameCountToPlayWsd            = 0;
    nn::atk::SoundHandle wsdSoundHandle;
    // SEQ 再生の開始ゲームフレーム、フレーム間隔
    const int beginningFrameToPlaySeq  = 3;
    const int intervalFrameToPlaySeq   = 300;
    int frameCountToPlaySeq            = 0;
    nn::atk::SoundHandle seqSoundHandle;

    // レンダラをサスペンドし、コマンドバッファを枯渇させる
    nn::atk::SoundSystem::SuspendAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
    ExhaustDriverCommandBuffer();
    DumpDriverCommandBuffer();
    bool isRendererSuspended = true;

    // 定期的にレンダラをサスペンド・レジュームしながら、バッファ枯渇させつつコマンド発行する
    int frameCount = 0;
    const int intervalFrameToSuspendRenderer  = 30;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    for (nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTime; elapsed += WaitTime)
    {
        if ( frameCount % intervalFrameToSuspendRenderer ==  0 )
        {
            if ( isRendererSuspended )
            {
                nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
            }
            else
            {
                nn::atk::SoundSystem::SuspendAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
                ExhaustDriverCommandBuffer();
            }
            isRendererSuspended = !isRendererSuspended;
        }
        if ( frameCount >= beginningFrameToPlayStrm )
        {
            if ( frameCountToPlayStrm % intervalFrameToPlayStrm == 0 )
            {
                strmSoundHandle.Stop(0);
                bool result = soundArchivePlayer.StartSound(&strmSoundHandle, STRM_MARIOKART).IsSuccess();
                NN_LOG("Start STRM_MARIOKART.[%s]\n", result ? "Success" : "Failure");
            }
            frameCountToPlayStrm++;
        }
        if ( frameCount >= beginningFrameToPlayWsd )
        {
            if ( frameCountToPlayWsd % intervalFrameToPlayWsd == 0 )
            {
                wsdSoundHandle.Stop(0);
                bool result = soundArchivePlayer.StartSound(&wsdSoundHandle, SE_YOSHI).IsSuccess();
                NN_LOG("Start SE_YOSHI.[%s]\n", result ? "Success" : "Failure");
            }
            frameCountToPlayWsd++;
        }
        if ( frameCount >= beginningFrameToPlaySeq )
        {
            if ( frameCountToPlaySeq % intervalFrameToPlaySeq == 0 )
            {
                seqSoundHandle.Stop(0);
                bool result = soundArchivePlayer.StartSound(&seqSoundHandle, SEQ_MARIOKART).IsSuccess();
                NN_LOG("Start SEQ_MARIOKART.[%s]\n", result ? "Success" : "Failure");
            }
            frameCountToPlaySeq++;
        }

        nn::atk::SoundSystem::VoiceCommandProcess(4);
        soundArchivePlayer.Update();
        frameCount++;
        nn::os::SleepThread(WaitTime);
    }

    // レンダラを停止したままだと SoundArchivePlayer の終了処理でボイスコマンドのフラッシュ待ちから返らなくなる
    if ( isRendererSuspended )
    {
        nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
