﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <memory>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>

namespace {

const int MemoryHeapSize     = 32 * 1024 * 1024;
const int MemoryPoolHeapSize = 32 * 1024 * 1024;

// フェード検査用の 1 フレームの待ち時間
// const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(33);
// const int64_t WaitTimeMsec = WaitTime.GetMilliSeconds();
// nn::os::TimerEventType      g_TimerEvent;

nnt::atk::util::FsCommonSetup   g_FsSetup;

class SoundSystemInitializeTester : public ::testing::Test
{
public:
    nn::mem::StandardAllocator& GetAllocator()
    {
        return m_Allocator;
    }

protected:
    virtual void SetUp() NN_OVERRIDE
    {
        m_HeapMemoryPtr = std::shared_ptr<void>(malloc(MemoryHeapSize), free);
        nnt::atk::util::OnPreAtkTest();
        m_Allocator.Initialize(m_HeapMemoryPtr.get(), MemoryHeapSize);
        m_FsSetup.Initialize();
    }
    virtual void TearDown() NN_OVERRIDE
    {
        m_FsSetup.Finalize();
        m_Allocator.Finalize();
    }

private:
    nnt::atk::util::FsCommonSetup   m_FsSetup;
    std::shared_ptr<void> m_HeapMemoryPtr;
    nn::mem::StandardAllocator  m_Allocator;
};

class SoundSystemSetup : public nnt::atk::util::AtkCommonSetup
{
public:
    void InitializeTestSoundSystem(nn::mem::StandardAllocator& allocator)
    {
        nnt::atk::util::AtkCommonSetup::InitializeSoundSystem(allocator);
    }

    void InitializeTestSoundSystem(nn::atk::SoundSystem::SoundSystemParam param, nn::mem::StandardAllocator& allocator)
    {
        nnt::atk::util::AtkCommonSetup::InitializeSoundSystem(param, allocator);
    }

    void FinalizeTestSoundSystem(nn::mem::StandardAllocator& allocator)
    {
        nnt::atk::util::AtkCommonSetup::FinalizeSoundSystem(allocator);
    }
};

SoundSystemSetup g_AtkSetup;

static char                                    g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator                     g_Allocator;
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN static char g_PoolHeapMemory[MemoryPoolHeapSize];
nn::mem::StandardAllocator                     g_AllocatorForMemoryPool;

void WaitForProcessCommand() NN_NOEXCEPT
{
    uint32_t command = 0;
    nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    // command = 0 の場合は無効値なので処理しない
    while (!nnt::atk::util::SendDummyCommand(&command) || command == 0)
    {
        nnt::atk::util::FlushCommand(false);
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
        nn::os::SleepThread(WaitTime);
    }

    // DriverCommand の Flush 直後に nnt::atk::util::IsCommandFinished() を呼ぶと、
    // true が返ってくることがあり十分な時間待てないことがあるため一回 SleepThread で待つ。
    // TODO: Flush 直後に IsCommandFinished を呼ぶと true が返ってくることがある原因を調査する
    do
    {
        nnt::atk::util::FlushCommand(false);
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
        nn::os::SleepThread(WaitTime);
    } while (!nnt::atk::util::IsCommandFinished(command));
}

void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT
{
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    archivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
    nn::os::SleepThread(WaitTime);
}

float GetClampedSubMixVolume(float volume)
{
    if (volume > nn::audio::SubMixType::GetVolumeMax())
    {
        return nn::audio::SubMixType::GetVolumeMax();
    }
    else if (volume < nn::audio::SubMixType::GetVolumeMin())
    {
        return nn::audio::SubMixType::GetVolumeMin();
    }
    return volume;
}

#if !defined(NN_SDK_BUILD_RELEASE)
void SetAuxBusAndWait(nn::atk::AuxBus bus, float volume, nn::TimeSpan fadeTimes)
{
    nn::atk::SoundSystem::SetAuxBusVolume(bus, volume, fadeTimes);
    WaitForProcessCommand();
}

void SetAuxBusAndWait(nn::atk::AuxBus bus, float volume, nn::TimeSpan fadeTimes, int subMixIndex)
{
    nn::atk::SoundSystem::SetAuxBusVolume(bus, volume, fadeTimes, subMixIndex);
    WaitForProcessCommand();
}
#endif

class EffectAuxTest : public nn::atk::EffectAux
{
public:
    virtual bool Initialize() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG("EffectAuxTest: Initialize().\n");
        return true;
    }
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG("EffectAuxTest: Finalize().\n");
    }
protected:
    virtual void UpdateSamples(int32_t* pSamples, const UpdateSamplesArg& arg) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pSamples);
        NN_UNUSED(arg);
    }
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SoundSystem, InitializeDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam param;
    size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
    void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);

    EXPECT_DEATH_IF_SUPPORTED(nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>(nullptr),
        memSizeForSoundSystem), ".*");
    EXPECT_DEATH_IF_SUPPORTED(nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>(pMemoryForSoundSystem),
        memSizeForSoundSystem - 1), ".*");

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundSystem, InitializeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);
    EXPECT_TRUE(nn::atk::SoundSystem::IsInitialized());
    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

// NwRenderer のコードが残っていてもビルドエラーにならないことを確認するテストです。
TEST(SoundSystem, InitializeWithNwRenderer)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());
    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableNwRenderer = true;
    param.nwVoiceSynthesizeBufferCount = static_cast<uint32_t>(nn::atk::SoundSystem::GetSynthesizeBufferSize());
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);
    EXPECT_TRUE(nn::atk::SoundSystem::IsInitialized());

    EXPECT_FALSE(nn::atk::SoundSystem::detail_IsInitializedNwRenderer());

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());


    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( SoundSystem, InitializeWithUnusedEffectChannelMuting )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableUnusedEffectChannelMuting = true;
    const size_t bufferSize = nn::atk::SoundSystem::GetRequiredMemSize( param );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::atk::SoundSystem::WorkMemoryAlignSize );

    nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>(buffer), bufferSize );
    EXPECT_TRUE( nn::atk::SoundSystem::IsInitialized() );

    nn::atk::SoundSystem::Finalize();
    EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

    g_Allocator.Free( buffer );

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( SoundSystem, InitializeWithInitializeBufferSet )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    // enableCircularBufferSinkBufferManagement する場合のテスト
    {
        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCircularBufferSink = true;
        param.enableCircularBufferSinkBufferManagement = true;
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
        void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );


        nn::Result result;
        nn::atk::SoundSystem::InitializeBufferSet bufferSet;
        bufferSet.workMem = reinterpret_cast<uintptr_t>(pMemoryForSoundSystem);
        bufferSet.workMemSize = memSizeForSoundSystem;

        bool isSucceed = nn::atk::SoundSystem::Initialize(
            &result,
            param,
            bufferSet );

        EXPECT_TRUE( isSucceed );
        NNT_EXPECT_RESULT_SUCCESS( result );

        nn::atk::SoundSystem::Finalize();
        g_Allocator.Free( pMemoryForSoundSystem );

        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );
    }

    // enableCircularBufferSinkBufferManagement しない場合のテスト
    {
        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCircularBufferSink = true;
        param.enableCircularBufferSinkBufferManagement = false;
        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
        void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
        size_t memSizeForCircularBufferSinkBuffer = nn::atk::SoundSystem::GetRequiredMemSizeForCircularBufferSink( param );
        void* pMemoryForCircularBufferSinkBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, memSizeForSoundSystem );

        nn::Result result;
        nn::atk::SoundSystem::InitializeBufferSet bufferSet;
        bufferSet.workMem = reinterpret_cast<uintptr_t>(pMemoryForSoundSystem);
        bufferSet.workMemSize = memSizeForSoundSystem;
        bufferSet.circularBufferSinkMem = reinterpret_cast<uintptr_t>(pMemoryForCircularBufferSinkBuffer);
        bufferSet.circularBufferSinkMemSize = memSizeForCircularBufferSinkBuffer;

        bool isSucceed = nn::atk::SoundSystem::Initialize(
            &result,
            param,
            bufferSet );

        EXPECT_TRUE( isSucceed );
        NNT_EXPECT_RESULT_SUCCESS( result );

        nn::atk::SoundSystem::Finalize();
        g_Allocator.Free( pMemoryForCircularBufferSinkBuffer );
        g_Allocator.Free( pMemoryForSoundSystem );

        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );
    }

    // enableCircularBufferSinkBufferManagement しない場合かつ enableMemoryPoolManagement もしない場合のテスト
    {
        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCircularBufferSink = true;
        param.enableCircularBufferSinkBufferManagement = false;
        param.enableMemoryPoolManagement = false;

        size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
        void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
        size_t memSizeForMemoryPoolBuffer = nn::atk::SoundSystem::GetRequiredMemSizeForMemoryPool( param );
        void* pMemoryForMemoryPoolBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, memSizeForMemoryPoolBuffer );
        size_t memSizeForCircularBufferSinkBuffer = nn::atk::SoundSystem::GetRequiredMemSizeForCircularBufferSink( param );
        void* pMemoryForCircularBufferSinkBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, memSizeForCircularBufferSinkBuffer );

        nn::Result result;
        nn::atk::SoundSystem::InitializeBufferSet bufferSet;
        bufferSet.workMem = reinterpret_cast<uintptr_t>(pMemoryForSoundSystem);
        bufferSet.workMemSize = memSizeForSoundSystem;
        bufferSet.memoryPoolMem = reinterpret_cast<uintptr_t>(pMemoryForMemoryPoolBuffer);
        bufferSet.memoryPoolMemSize = memSizeForMemoryPoolBuffer;
        bufferSet.circularBufferSinkMem = reinterpret_cast<uintptr_t>(pMemoryForCircularBufferSinkBuffer);
        bufferSet.circularBufferSinkMemSize = memSizeForCircularBufferSinkBuffer;

        bool isSucceed = nn::atk::SoundSystem::Initialize(
            &result,
            param,
            bufferSet );

        EXPECT_TRUE( isSucceed );
        NNT_EXPECT_RESULT_SUCCESS( result );

        nn::atk::SoundSystem::Finalize();
        g_Allocator.Free( pMemoryForCircularBufferSinkBuffer );
        g_Allocator.Free( pMemoryForMemoryPoolBuffer );
        g_Allocator.Free( pMemoryForSoundSystem );

        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, InitializeResultTest)
{
    nnt::atk::util::OnPreAtkTest();
    const int MaxInitializeCount = 2; // 二重初期化の場合も ResultSuccess になる事を確認
    std::size_t memSizeForSoundSystem[MaxInitializeCount];
    void* pMemoryForSoundSystem[MaxInitializeCount];

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // 初期化した場合の返り値と Result をテスト
    for ( auto i = 0; i < MaxInitializeCount; i++ )
    {
        nn::atk::SoundSystem::SoundSystemParam param;
        nn::Result result;

        memSizeForSoundSystem[i] = nn::atk::SoundSystem::GetRequiredMemSize(param);
        pMemoryForSoundSystem[i] = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForSoundSystem[i], nn::atk::SoundSystem::WorkMemoryAlignSize);
        bool isSucceed = nn::atk::SoundSystem::Initialize(
            &result,
            param,
            reinterpret_cast<uintptr_t>(pMemoryForSoundSystem[i]),
            memSizeForSoundSystem[i]);

        EXPECT_TRUE(isSucceed);
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    nn::atk::SoundSystem::Finalize();
    for ( auto i = 0; i < MaxInitializeCount; i++ )
    {
        g_Allocator.Free(pMemoryForSoundSystem[i]);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, InitializeResultWithPoolTest)
{
    nnt::atk::util::OnPreAtkTest();
    const int MaxInitializeCount = 2; // 二重初期化の場合も ResultSuccess になる事を確認
    std::size_t memSizeForSoundSystem[MaxInitializeCount];
    void* pMemoryForSoundSystem[MaxInitializeCount];
    std::size_t memSizeForSoundSystemMemoryPool[MaxInitializeCount];
    void* pMemoryForSoundSystemMemoryPool[MaxInitializeCount];

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_AllocatorForMemoryPool.Initialize(g_PoolHeapMemory, MemoryPoolHeapSize);
    g_FsSetup.Initialize();

    // メモリプール手動管理時、初期化した場合の返り値と Result をテスト
    for ( auto i = 0; i < MaxInitializeCount; i++ )
    {
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableMemoryPoolManagement = false;
        nn::Result result;

        memSizeForSoundSystem[i] = nn::atk::SoundSystem::GetRequiredMemSize(param);
        pMemoryForSoundSystem[i] = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForSoundSystem[i], nn::atk::SoundSystem::WorkMemoryAlignSize);
        memSizeForSoundSystemMemoryPool[i] = nn::atk::SoundSystem::GetRequiredMemSizeForMemoryPool(param);
        pMemoryForSoundSystemMemoryPool[i] = nnt::atk::util::AllocateUninitializedMemory( g_AllocatorForMemoryPool, memSizeForSoundSystemMemoryPool[i], nn::audio::BufferAlignSize );
        bool isSucceed = nn::atk::SoundSystem::Initialize(
            &result,
            param,
            reinterpret_cast<uintptr_t>(pMemoryForSoundSystem[i]),
            memSizeForSoundSystem[i],
            reinterpret_cast<uintptr_t>(pMemoryForSoundSystemMemoryPool[i]),
            memSizeForSoundSystemMemoryPool[i]);

        EXPECT_TRUE(isSucceed);
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    nn::audio::MemoryPoolType pool;
    nn::atk::SoundSystem::AttachMemoryPool( &pool, g_PoolHeapMemory, sizeof(g_PoolHeapMemory) );
    nn::atk::SoundSystem::DetachMemoryPool( &pool );

    nn::atk::SoundSystem::Finalize();
    for ( auto i = 0; i < MaxInitializeCount; i++ )
    {
        g_AllocatorForMemoryPool.Free(pMemoryForSoundSystemMemoryPool[i]);
        g_Allocator.Free(pMemoryForSoundSystem[i]);
    }

    g_FsSetup.Finalize();
    g_AllocatorForMemoryPool.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, InitializeWithPoolTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableMemoryPoolManagement = false;

    //  buffer, memPoolBuffer のバッファを 32 だけ大きめに確保します ( 32 は適当な数字です )
    //  これにより、「アラインがズレているテスト」でアサートしなかったときにメモリ破壊を起こさないようにしています。
    const size_t memSize = nn::atk::SoundSystem::GetRequiredMemSize( param );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  memSize + 32, nn::atk::SoundSystem::WorkMemoryAlignSize );
    const size_t memPoolSize = nn::atk::SoundSystem::GetRequiredMemSizeForMemoryPool( param );
    const size_t alignedMemPoolSize = nn::util::align_up( memPoolSize, nn::audio::MemoryPoolType::SizeGranularity );
    void* memPoolBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  alignedMemPoolSize + 32, nn::audio::MemoryPoolType::AddressAlignment );

#if !defined(NN_SDK_BUILD_RELEASE)
    // nullptr テスト
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( nullptr ), memSize, reinterpret_cast<uintptr_t>( memPoolBuffer ), memPoolSize ), ".*" );
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), memSize, reinterpret_cast<uintptr_t>( nullptr ), memPoolSize ), ".*" );

    // サイズが小さい
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), memSize - 1, reinterpret_cast<uintptr_t>( memPoolBuffer ), memPoolSize ), ".*" );
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), memSize, reinterpret_cast<uintptr_t>( memPoolBuffer ), memPoolSize - 1 ), ".*" );

    // アラインがズレている
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ) + 1, memSize, reinterpret_cast<uintptr_t>( memPoolBuffer ), memPoolSize ), ".*" );
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), memSize, reinterpret_cast<uintptr_t>( memPoolBuffer ) + 1, memPoolSize ), ".*" );
#endif

    EXPECT_TRUE( nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), memSize, reinterpret_cast<uintptr_t>( memPoolBuffer ), memPoolSize ) );
    EXPECT_TRUE( nn::atk::SoundSystem::IsInitialized() );

    nn::audio::MemoryPoolType memPool;
    nn::atk::SoundSystem::AttachMemoryPool( &memPool, memPoolBuffer, alignedMemPoolSize );
    nn::atk::SoundSystem::DetachMemoryPool( &memPool );

    nn::atk::SoundSystem::Finalize();
    EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

    g_Allocator.Free( memPoolBuffer );
    g_Allocator.Free( buffer );

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, InitializeTestWithUserThreadRendering)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // enableUserThreadRendering = true
    {
        EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableUserThreadRendering = true;
        g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);
        EXPECT_TRUE(nn::atk::SoundSystem::IsInitialized());

        for(int i = 0; i < 5; ++i)
        {
            nn::atk::SoundSystem::ExecuteRendering();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }

        g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
        EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());
    }

    // enableUserThreadRendering = true, enableProfiler = true (SIGLO-60731 のレグレッションテスト)
    {
        EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableUserThreadRendering = true;
        param.enableProfiler = true;
        param.enableCircularBufferSink = true;
        g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);
        EXPECT_TRUE(nn::atk::SoundSystem::IsInitialized());

        nn::atk::ProfileReader profileReader;
        nn::atk::SoundSystem::RegisterProfileReader(profileReader);

        for(int i = 0; i < 5; ++i)
        {
            nn::atk::SoundSystem::ExecuteRendering();
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
        }

        g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
        EXPECT_FALSE(nn::atk::SoundSystem::IsInitialized());
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST_F(SoundSystemInitializeTester, QueueCountAdjustDeathTest)
{
    const std::pair<int, int> TestCondition[] =
    {
        {0, nn::atk::SoundSystem::SoundSystemParam::DefaultTaskThreadCommandQueueCount},
        {nn::atk::SoundSystem::SoundSystemParam::DefaultSoundThreadCommandQueueCount, 0},
    };
    for( const auto& condition : TestCondition )
    {
        nn::atk::SoundSystem::SoundSystemParam param;
        param.soundThreadCommandQueueCount = condition.first;
        param.taskThreadCommandQueueCount = condition.second;
        EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::GetRequiredMemSize(param) , ".*");
    }
}
#endif

TEST_F(SoundSystemInitializeTester, QueueCountAdjustTest)
{
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t prevMemSizeForSoundSystem = 0;
    for (int soundThreadCueueCount = 1; soundThreadCueueCount < param.soundThreadCommandQueueCount * 2; ++soundThreadCueueCount)
    {
        param.soundThreadCommandQueueCount = soundThreadCueueCount;
        std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        EXPECT_GT(memSizeForSoundSystem, prevMemSizeForSoundSystem);

        void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory(GetAllocator(), memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);
        EXPECT_TRUE( nn::atk::SoundSystem::Initialize(
            param,
            reinterpret_cast<uintptr_t>(pMemoryForSoundSystem),
            memSizeForSoundSystem) );

        nn::atk::SoundSystem::Finalize();
        GetAllocator().Free(pMemoryForSoundSystem);
        prevMemSizeForSoundSystem = memSizeForSoundSystem;
    }

    prevMemSizeForSoundSystem = 0;
    for (int taskThreadCueueCount = 1; taskThreadCueueCount < param.soundThreadCommandQueueCount * 2; ++taskThreadCueueCount)
    {
        param.taskThreadCommandQueueCount = taskThreadCueueCount;
        std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize(param);
        EXPECT_GT(memSizeForSoundSystem, prevMemSizeForSoundSystem);

        void* pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory(GetAllocator(), memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize);
        EXPECT_TRUE( nn::atk::SoundSystem::Initialize(
            param,
            reinterpret_cast<uintptr_t>(pMemoryForSoundSystem),
            memSizeForSoundSystem) );

        nn::atk::SoundSystem::Finalize();
        GetAllocator().Free(pMemoryForSoundSystem);
        prevMemSizeForSoundSystem = memSizeForSoundSystem;
    }
}

TEST(SoundSystem, CircularBufferSinkStateTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    EXPECT_EQ( nn::atk::CircularBufferSinkState_Invalid, nn::atk::SoundSystem::GetCircularBufferSinkState() );

    // ReadCircularSink 用のバッファ確保
    const size_t waveBufferSize = 48000;  //  波形用バッファサイズです。大きさは適当です。
    void* waveBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, waveBufferSize);

    // CircularBufferSink を有効にしない状態でのテスト
    {
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCircularBufferSink = false;
        const size_t bufferSize = nn::atk::SoundSystem::GetRequiredMemSize( param );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, bufferSize, nn::atk::SoundSystem::WorkMemoryAlignSize );
        nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), bufferSize );

        EXPECT_EQ( nn::atk::CircularBufferSinkState_Invalid, nn::atk::SoundSystem::GetCircularBufferSinkState() );
        EXPECT_EQ( 0u, nn::atk::SoundSystem::ReadCircularBufferSink( waveBuffer, waveBufferSize ) );

        nn::atk::SoundSystem::Finalize();

        EXPECT_EQ( nn::atk::CircularBufferSinkState_Invalid, nn::atk::SoundSystem::GetCircularBufferSinkState() );

        g_Allocator.Free( buffer );
    }

    // CircularBufferSink を有効にした状態でのテスト
    {
        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCircularBufferSink = true;
        const size_t bufferSize = nn::atk::SoundSystem::GetRequiredMemSize( param );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, bufferSize, nn::atk::SoundSystem::WorkMemoryAlignSize );
        nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), bufferSize );

        EXPECT_EQ( nn::atk::CircularBufferSinkState_Started, nn::atk::SoundSystem::GetCircularBufferSinkState() );

        nn::atk::SoundSystem::StopCircularBufferSink();
        EXPECT_EQ( nn::atk::CircularBufferSinkState_Stopped, nn::atk::SoundSystem::GetCircularBufferSinkState() );
        EXPECT_EQ( 0u, nn::atk::SoundSystem::ReadCircularBufferSink( waveBuffer, waveBufferSize ) );

        nn::atk::SoundSystem::StartCircularBufferSink();
        EXPECT_EQ( nn::atk::CircularBufferSinkState_Started, nn::atk::SoundSystem::GetCircularBufferSinkState() );

        nn::atk::SoundSystem::Finalize();

        EXPECT_EQ( nn::atk::CircularBufferSinkState_Invalid, nn::atk::SoundSystem::GetCircularBufferSinkState() );

        g_Allocator.Free( buffer );
    }

    g_Allocator.Free( waveBuffer );

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, ReadCircularBufferSinkReturnValueTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    bool isCircularBufferSinkEnabled[] = { false, true };
    for(const auto& isEnabled : isCircularBufferSinkEnabled )
    {
        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

        nn::atk::SoundSystem::SoundSystemParam param;
        param.enableCircularBufferSink = isEnabled;
        const size_t bufferSize = nn::atk::SoundSystem::GetRequiredMemSize( param );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, bufferSize, nn::atk::SoundSystem::WorkMemoryAlignSize );

        nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( buffer ), bufferSize );
        EXPECT_TRUE( nn::atk::SoundSystem::IsInitialized() );

        const size_t waveBufferSize = 48000;  //  波形用バッファサイズです。大きさは適当です。
        void* waveBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, waveBufferSize);

        //  タイミングによって isEnabled = true の場合でも ReadCircularBufferSink が 0 を返すことがあるため
        //  30 回 ReadCircularBufferSink を行って返り値の確認をするようにします。 ( 30 回の回数に根拠はありません )
        const int loopCount = 30;
        bool isReturnedOnly0= true;
        for(int i = 0; i < loopCount; i++)
        {
            WaitForProcessCommand();
            if( nn::atk::SoundSystem::ReadCircularBufferSink( waveBuffer, waveBufferSize ) > 0u )
            {
                isReturnedOnly0 = false;
                break;
            }
        }

        if( isEnabled )
        {
            EXPECT_EQ( isReturnedOnly0, false );
        }
        else
        {
            EXPECT_EQ( isReturnedOnly0, true );
        }

        nn::atk::SoundSystem::Finalize();
        EXPECT_FALSE( nn::atk::SoundSystem::IsInitialized() );

        g_Allocator.Free( waveBuffer );
        g_Allocator.Free( buffer );
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, OutputModeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode::OutputMode_Monaural);
    EXPECT_EQ(nn::atk::OutputMode::OutputMode_Monaural, nn::atk::SoundSystem::GetOutputMode());
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode::OutputMode_Stereo);
    EXPECT_EQ(nn::atk::OutputMode::OutputMode_Stereo, nn::atk::SoundSystem::GetOutputMode());
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode::OutputMode_Surround);
    EXPECT_EQ(nn::atk::OutputMode::OutputMode_Surround, nn::atk::SoundSystem::GetOutputMode());

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, OutputDeviceFlagTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    for(uint8_t i = 0; i < nn::atk::OutputDeviceIndex::OutputDeviceIndex_Count; i++)
    {
        nn::atk::SoundSystem::SetOutputDeviceFlag( nn::atk::OutputLineIndex::OutputLineIndex_User0, i);
        EXPECT_EQ(i, nn::atk::SoundSystem::GetOutputDeviceFlag(nn::atk::OutputLineIndex::OutputLineIndex_User0));
    }

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

/*
void FadeOutCheck(void *arg)
{
    NN_UNUSED(arg);

    const int64_t FadeTimeMsec = 1000;

    nn::atk::SoundSystem::SetMasterVolume(1.0f, static_cast<int>(FadeTimeMsec));
    g_SoundArchivePlayer.Update();
    for(int64_t elapsedMsec = 0; elapsedMsec <= FadeTimeMsec; elapsedMsec += WaitTimeMsec)
    {
        nn::os::WaitTimerEvent(&g_TimerEvent);
        float testVolume = nn::atk::SoundSystem::GetMasterVolume();
        g_SoundArchivePlayer.Update();
        // +-10ms 以内(オーディオフレーム 2 回分) に収まると想定しているが、うまくいかず +-1フレーム分としている
        EXPECT_LE(testVolume, 1.0f * (elapsedMsec + WaitTimeMsec) / FadeTimeMsec) << elapsedMsec;
        EXPECT_GE(testVolume, 1.0f * (elapsedMsec - WaitTimeMsec) / FadeTimeMsec) << elapsedMsec;
    }

    nn::os::StopTimerEvent(&g_TimerEvent);
}
*/

TEST(SoundSystem, MasterVolumeChangeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    EXPECT_EQ(1.0f, nn::atk::SoundSystem::GetMasterVolume());
    nn::atk::SoundSystem::SetMasterVolume(-1.0f, 0);
    WaitForProcessCommand();
    EXPECT_EQ(0.0f, nn::atk::SoundSystem::GetMasterVolume());
    nn::atk::SoundSystem::SetMasterVolume(0.0f, 0);
    WaitForProcessCommand();
    EXPECT_EQ(0.0f, nn::atk::SoundSystem::GetMasterVolume());
    nn::atk::SoundSystem::SetMasterVolume(2.0f, 0);
    WaitForProcessCommand();
    EXPECT_EQ(2.0f, nn::atk::SoundSystem::GetMasterVolume());
    nn::atk::SoundSystem::SetMasterVolume(3.0f, 0);
    WaitForProcessCommand();
    EXPECT_EQ(3.0f, nn::atk::SoundSystem::GetMasterVolume());

    nn::atk::SoundSystem::SetMasterVolume(0.0f, 0);
    WaitForProcessCommand();
    EXPECT_EQ(0.0f, nn::atk::SoundSystem::GetMasterVolume());

    // フェードインチェック(時折失敗するためコメントアウト)
    /*
    InitializeAtk();
    nn::os::ThreadType fadeOutCheckThread;
    nn::os::InitializeTimerEvent(&g_TimerEvent, nn::os::EventClearMode_AutoClear);
    const size_t threadStackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char threadStack[threadStackSize];
    NN_ABORT_UNLESS(nn::os::CreateThread(&fadeOutCheckThread, FadeOutCheck, NULL, threadStack, threadStackSize, nn::os::DefaultThreadPriority).IsSuccess(), "cannot create thread.");

    nn::os::StartThread(&fadeOutCheckThread);
    nn::os::StartPeriodicTimerEvent(&g_TimerEvent, nn::TimeSpan::FromMilliSeconds(0), WaitTime);
    nn::os::WaitThread(&fadeOutCheckThread);
    nn::os::DestroyThread(&fadeOutCheckThread);
    nn::os::FinalizeTimerEvent(&g_TimerEvent);
    FinalizeAtk();
    */

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}


#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SoundSystem, AuxBusVolumeChangeDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    // Invalid AuxBus
    EXPECT_DEATH_IF_SUPPORTED(SetAuxBusAndWait(
        static_cast<nn::atk::AuxBus>(-1),
        0.0f,
        nn::TimeSpan::FromMilliSeconds(0)),".*");
    EXPECT_DEATH_IF_SUPPORTED(SetAuxBusAndWait(
        nn::atk::AuxBus_Count,
        0.0f,
        nn::TimeSpan::FromMilliSeconds(0)), ".*");

    // 追加の AuxBus を有効にしないままボリューム値を設定しようとしている
    EXPECT_DEATH_IF_SUPPORTED(SetAuxBusAndWait(
        nn::atk::AuxBus_A,
        0.0f,
        nn::TimeSpan::FromMilliSeconds(0),
        1), ".*");

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();

    // 追加の AuxBus を利用した DEATH テスト
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam additionalParam;
    additionalParam.enableAdditionalEffectBus = true;
    g_AtkSetup.InitializeTestSoundSystem(additionalParam, g_Allocator);

    // Invalid SubMix Settings
    EXPECT_DEATH_IF_SUPPORTED(SetAuxBusAndWait(
        nn::atk::AuxBus_A,
        0.0f,
        nn::TimeSpan::FromMilliSeconds(0),
        -1), ".*");
    // 現状、追加の AuxBus には AuxBus_C は存在しない
    EXPECT_DEATH_IF_SUPPORTED(SetAuxBusAndWait(
        nn::atk::AuxBus_C,
        0.0f,
        nn::TimeSpan::FromMilliSeconds(0),
        1), ".*");

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundSystem, AuxBusVolumeChangeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.InitializeTestSoundSystem(g_Allocator);

    const nn::TimeSpan FadeTimes[] = { nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(100) };
    const int LoopCountMax = 100;
    const float VolumeSettings[] = { -1.0f, 0.0f, 3.0f };

    for (nn::TimeSpan fadeTime : FadeTimes)
    {
        const float DefaultVolume = 1.0f;

        // メインの AuxBus のテスト
        for (int i = 0; i < nn::atk::AuxBus_Count; i++)
        {
            nn::atk::AuxBus bus = static_cast<nn::atk::AuxBus>(i);

            // デフォルト値テスト
            EXPECT_EQ(DefaultVolume, nn::atk::SoundSystem::GetAuxBusVolume(bus));

            float prevVolume = DefaultVolume;
            for (float testVolume : VolumeSettings)
            {
                nn::atk::SoundSystem::SetAuxBusVolume(bus, testVolume, fadeTime);
                int loopCount = 0;

                for (loopCount = 0; loopCount < LoopCountMax; ++loopCount)
                {
                    WaitForProcessCommand();
                    float clampedTestVolume = GetClampedSubMixVolume(testVolume);
                    float actualVolume = nn::atk::SoundSystem::GetAuxBusVolume(bus);
                    if (actualVolume == clampedTestVolume)
                    {
                        break;
                    }
                    else
                    {
                        if (clampedTestVolume > prevVolume)
                        {
                            EXPECT_GE(actualVolume, prevVolume);
                            EXPECT_LE(actualVolume, clampedTestVolume);
                        }
                        else
                        {
                            EXPECT_GE(actualVolume, clampedTestVolume);
                            EXPECT_LE(actualVolume, prevVolume);
                        }
                    }
                }
                prevVolume = nn::atk::SoundSystem::GetAuxBusVolume(bus);
                EXPECT_NE(loopCount, LoopCountMax) << "Invalid loop count.";
            }

            // 再度ループが回った際にデフォルト値テストに失敗しないよう、デフォルトボリューム値に戻しておく
            nn::atk::SoundSystem::SetAuxBusVolume(bus, DefaultVolume, fadeTime);
            // SetAuxBusVolume の設定が反映されるのを待つ
            WaitForProcessCommand();
        }
    }

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, AuxBusVolumeChangeTestForAdditionalEffect)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableAdditionalEffectBus = true;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    const nn::TimeSpan FadeTimes[] = { nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(100) };
    const int LoopCountMax = 100;
    const float VolumeSettings[] = { -1.0f, 0.0f, 3.0f };

    for (nn::TimeSpan fadeTime : FadeTimes)
    {
        const float DefaultVolume = 1.0f;

        // メインの AuxBus のテスト
        for (int i = 0; i < nn::atk::AuxBus_Count; i++)
        {
            nn::atk::AuxBus bus = static_cast<nn::atk::AuxBus>(i);

            // デフォルト値テスト
            EXPECT_EQ(DefaultVolume, nn::atk::SoundSystem::GetAuxBusVolume(bus));

            float prevVolume = DefaultVolume;
            for (float testVolume : VolumeSettings)
            {
                nn::atk::SoundSystem::SetAuxBusVolume(bus, testVolume, fadeTime);
                int loopCount = 0;

                for(loopCount = 0; loopCount < LoopCountMax; ++loopCount)
                {
                    WaitForProcessCommand();
                    float clampedTestVolume = GetClampedSubMixVolume(testVolume);
                    float actualVolume = nn::atk::SoundSystem::GetAuxBusVolume(bus);
                    if (actualVolume == clampedTestVolume)
                    {
                        break;
                    }
                    else
                    {
                        if (clampedTestVolume > prevVolume)
                        {
                            EXPECT_GE(actualVolume, prevVolume);
                            EXPECT_LE(actualVolume, clampedTestVolume);
                        }
                        else
                        {
                            EXPECT_GE(actualVolume, clampedTestVolume);
                            EXPECT_LE(actualVolume, prevVolume);
                        }
                    }
                }
                prevVolume = nn::atk::SoundSystem::GetAuxBusVolume(bus);
                EXPECT_NE(loopCount, LoopCountMax) << "Invalid loop count.";
            }

            // 再度ループが回った際にデフォルト値テストに失敗しないよう、デフォルトボリューム値に戻しておく
            nn::atk::SoundSystem::SetAuxBusVolume(bus, DefaultVolume, fadeTime);
            // SetAuxBusVolume の設定が反映されるのを待つ
            WaitForProcessCommand();
        }

        // Additional AuxBus のテスト
        // Additional AuxBus は AuxBus_A, AuxBus_B のみしかないため、
        // 範囲が Main AuxBus とは異なる
        for (int i = 0; i < nn::atk::AuxBus_C; i++)
        {
            nn::atk::AuxBus bus = static_cast<nn::atk::AuxBus>(i);

            // デフォルト値テスト
            EXPECT_EQ(DefaultVolume, nn::atk::SoundSystem::GetAuxBusVolume(bus, 1));

            float prevVolume = DefaultVolume;

            for (float testVolume : VolumeSettings)
            {
                nn::atk::SoundSystem::SetAuxBusVolume(bus, testVolume, fadeTime, 1);
                int loopCount = 0;

                for (loopCount = 0; loopCount < LoopCountMax; ++loopCount)
                {
                    WaitForProcessCommand();
                    float clampedTestVolume = GetClampedSubMixVolume(testVolume);
                    float actualVolume = nn::atk::SoundSystem::GetAuxBusVolume(bus, 1);

                    if (actualVolume == clampedTestVolume)
                    {
                        break;
                    }
                    else
                    {
                        if (clampedTestVolume > prevVolume)
                        {
                            EXPECT_GE(actualVolume, prevVolume);
                            EXPECT_LE(actualVolume, clampedTestVolume);
                        }
                        else
                        {
                            EXPECT_GE(actualVolume, clampedTestVolume);
                            EXPECT_LE(actualVolume, prevVolume);
                        }
                    }
                }
                prevVolume = nn::atk::SoundSystem::GetAuxBusVolume(bus, 1);
                EXPECT_NE(loopCount, LoopCountMax) << "Invalid loop count.";
            }

            // 再度ループが回った際にデフォルト値テストに失敗しないよう、デフォルトボリューム値に戻しておく
            nn::atk::SoundSystem::SetAuxBusVolume(bus, DefaultVolume, fadeTime, 1);
            // SetAuxBusVolume の設定が反映されるのを待つ
            WaitForProcessCommand();
        }
    }

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, VoiceCountTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    // 音源データの読み込み
    g_AtkSetup.LoadData(WSD_SNARE, "WSD_SNARE");

    EXPECT_GE(nn::atk::SoundSystem::VoiceCountMax, nn::atk::SoundSystem::GetVoiceCountMax());
    EXPECT_EQ(0, nn::atk::SoundSystem::GetVoiceCount());

    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, WSD_SNARE).IsSuccess());
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  PrepareSound() の結果を反映させます
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
    //  PrepareSound() でボイス数が増えていないことを確認
    EXPECT_EQ(0, nn::atk::SoundSystem::GetVoiceCount());

    soundHandle.StartPrepared();
    soundArchivePlayer.Update();

    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(2000);

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommand 版ではボイス数が反映されない
    const nn::TimeSpan WaitTimeToUpdateVoiceCount = nn::TimeSpan::FromMilliSeconds(20);
    nn::os::SleepThread(WaitTimeToUpdateVoiceCount);
    EXPECT_EQ(0, nn::atk::SoundSystem::GetVoiceCount()) << "Processed voice commands";
#endif

    // SoundArchivePlayer の Update 後に、ボイス数が SoundSystem に反映されるのを待つ
    bool isChangeVoiceCount = false;
    for(nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut ; elapsed += WaitTime)
    {
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
        int voiceCount = nn::atk::SoundSystem::GetVoiceCount();
        if(voiceCount != 0)
        {
            isChangeVoiceCount = true;
            EXPECT_EQ(1, voiceCount) << "Invalid voice count";
            break;
        }
    }
    EXPECT_TRUE(isChangeVoiceCount) << "Time out";

    g_AtkSetup.Finalize(g_Allocator);

    // VoiceCountMax の数を変更させて初期化させた場合のテスト
    nn::atk::SoundSystem::SoundSystemParam param;
    const int VoiceCountMaxMin = 1;
    param.voiceCountMax = VoiceCountMaxMin;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);
    EXPECT_TRUE(nn::atk::SoundSystem::IsInitialized());
    EXPECT_EQ(VoiceCountMaxMin, nn::atk::SoundSystem::GetVoiceCountMax());

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundSystem, SuspendResumeAudioRendererTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    // 音源データの読み込み
    g_AtkSetup.LoadData(WSD_SNARE, "WSD_SNARE");
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");

    // TODO: フェードアウト対応時にテスト項目を追加する
    const unsigned int TestData[] = { WSD_SNARE, SEQ_MARIOKART, STRM_MARIOKART };

    for (unsigned int testData : TestData)
    {
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;

        // 再生準備
        soundArchivePlayer.PrepareSound(&soundHandle, testData);
        UpdateAndWait(soundArchivePlayer);

        while (!soundHandle.IsPrepared())
        {
            UpdateAndWait(soundArchivePlayer);
        }

        soundArchivePlayer.StartSound(&soundHandle, testData);
        UpdateAndWait(soundArchivePlayer);

        nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);
        nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
        nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);

        nn::atk::SoundSystem::SuspendAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
        const nn::TimeSpan TestStopTime = nn::TimeSpan::FromMilliSeconds(500);

        int64_t previousPosition = 0;
        if (sequenceSoundHandle.IsAttachedSound())
        {
            previousPosition = static_cast<int64_t>(sequenceSoundHandle.GetTick());
        }
        if (streamSoundHandle.IsAttachedSound())
        {
            previousPosition = streamSoundHandle.GetPlaySamplePosition();
        }
        if (waveSoundHandle.IsAttachedSound())
        {
            previousPosition = waveSoundHandle.GetPlaySamplePosition();
        }

        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

        // SuspendAudioRenderer 後に再生位置が動いていないか確認するテスト
        const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(5000);
        int positionStoppedCount = 0;
        const int ExpectedPositionStoppedCount = 10;
        for (nn::TimeSpan elapsed = 0; elapsed < TestStopTime; elapsed += WaitTime)
        {
            if (sequenceSoundHandle.IsAttachedSound())
            {
                if (sequenceSoundHandle.GetTick() == static_cast<uint32_t>(previousPosition))
                {
                    positionStoppedCount++;
                }
                else
                {
                    positionStoppedCount = 0;
                }
                previousPosition = static_cast<int64_t>(sequenceSoundHandle.GetTick());
            }
            if (streamSoundHandle.IsAttachedSound())
            {
                if (streamSoundHandle.GetPlaySamplePosition() == previousPosition)
                {
                    positionStoppedCount++;
                }
                else
                {
                    positionStoppedCount = 0;
                }
                previousPosition = streamSoundHandle.GetPlaySamplePosition();
            }
            if (waveSoundHandle.IsAttachedSound())
            {
                if (waveSoundHandle.GetPlaySamplePosition() == previousPosition)
                {
                    positionStoppedCount++;
                }
                else
                {
                    positionStoppedCount = 0;
                }
                previousPosition = waveSoundHandle.GetPlaySamplePosition();
            }

            if (positionStoppedCount == ExpectedPositionStoppedCount)
            {
                break;
            }

            UpdateAndWait(soundArchivePlayer);
        }
        EXPECT_EQ(ExpectedPositionStoppedCount, positionStoppedCount);

        nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
        // DriverCommand 処理が回っていることを WaitForProcessCommand で確認する
        WaitForProcessCommand();

        // SuspendAudioRenderer 後に再生位置が動いていないか確認するテスト
        bool isPositionMoved = false;
        for (nn::TimeSpan elapsed = 0; elapsed < TimeOut; elapsed += WaitTime)
        {
            if (sequenceSoundHandle.IsAttachedSound())
            {
                if (sequenceSoundHandle.GetTick() != static_cast<uint32_t>(previousPosition))
                {
                    isPositionMoved = true;
                    break;
                }
            }
            if (streamSoundHandle.IsAttachedSound())
            {
                if (streamSoundHandle.GetPlaySamplePosition() != previousPosition)
                {
                    isPositionMoved = true;
                    break;
                }
            }
            if (waveSoundHandle.IsAttachedSound())
            {
                if (waveSoundHandle.GetPlaySamplePosition() != previousPosition)
                {
                    isPositionMoved = true;
                    break;
                }
            }

            UpdateAndWait(soundArchivePlayer);
        }
        EXPECT_TRUE(isPositionMoved);

        soundHandle.Stop(0);
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
    }
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} //NOLINT(impl/function_size)

TEST(SoundSystem, MultiSuspendResumeAudioRendererTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    // Suspend していない状態で Resume を呼んでも落ちないことを確認
    nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));

    // 多重に Suspend しても落ちないことを確認
    nn::atk::SoundSystem::SuspendAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
    nn::atk::SoundSystem::SuspendAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));

    // 現状は Resume したままだと終了処理ができなくなるので Suspend と同じ回数 Resume を呼ぶ
    nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
    nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#if !defined(NN_SDK_BUILD_RELEASE)
void AppendEffectAndUpdateRenderer(nn::atk::AuxBus bus, nn::atk::EffectAux* pEffect, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    pEffect->SetEnabled(true);
    nn::atk::SoundSystem::AppendEffect(bus, pEffect, buffer, bufferSize);
    nnt::atk::util::WaitForProcessCommand();
    nn::atk::detail::driver::HardwareManager::GetInstance().RequestUpdateAudioRenderer();
}
TEST(SoundSystem, MemoryPoolCheckDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nnt::atk::util::AtkCommonSetup::InitializeParam parameter;
    parameter.GetSoundSystemParam().enableMemoryPoolAttachCheck = true; // メモリプールアタッチのチェックを有効化
    g_AtkSetup.Initialize(parameter, g_Allocator);

    EffectAuxTest aux;
    size_t auxBufferSize = nn::util::align_up(nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&aux), nn::audio::MemoryPoolType::SizeGranularity);
    void* auxBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, auxBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
    NN_ABORT_UNLESS_NOT_NULL(auxBuffer);

    // EffectAux のバッファをメモリプールアタッチせずにアペンドする
    EXPECT_DEATH_IF_SUPPORTED(AppendEffectAndUpdateRenderer(nn::atk::AuxBus_A, &aux, auxBuffer, auxBufferSize), ".*");

    g_Allocator.Free(auxBuffer);
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundSystem, RenderingTimeLimitTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    const int renderingTimeLimitPercent = 50;
    nn::atk::SoundSystem::SetAudioRendererRenderingTimeLimit(renderingTimeLimitPercent);
    EXPECT_EQ(renderingTimeLimitPercent, nn::atk::SoundSystem::GetAudioRendererRenderingTimeLimit());

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SoundSystem, RenderingTimeLimitDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    nn::atk::SoundSystem::SoundSystemParam param;
    g_AtkSetup.InitializeTestSoundSystem(param, g_Allocator);

    EXPECT_DEATH_IF_SUPPORTED(nn::atk::SoundSystem::SetAudioRendererRenderingTimeLimit(110), ".*");
    EXPECT_DEATH_IF_SUPPORTED(nn::atk::SoundSystem::SetAudioRendererRenderingTimeLimit(0), ".*");

    g_AtkSetup.FinalizeTestSoundSystem(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
