﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>
#include <nn/mem.h>

namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup  g_AtkSetup;

const auto WaitTime = nn::TimeSpan::FromMilliSeconds(16);

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
//  UpdateAndWaitForUpdateAndWaitForWaveStart 関数で VoiceCommand の確認ができているか
bool g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForWaveStart = false;
#endif

void UpdateAndWaitForWaveStart(nn::atk::WaveSoundHandle& waveSoundHandle)
{
    const auto TimeOut = nn::TimeSpan::FromMilliSeconds(5000);
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    soundArchivePlayer.Update();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    if( g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForWaveStart == false )
    {
        //  VoiceCommandProcess を呼ばなければ再生されないことを確認
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));
        EXPECT_EQ(0, waveSoundHandle.GetPlaySamplePosition()) << "Wave sound was started without voice commands.";

        //  2 回目のチェックはしなくてもいい
        g_IsCheckedVoiceCommandInFunctionOfUpdateAndWaitForWaveStart = true;
    }
#endif

    // 実際に再生が行われていることをパラメータ上で確認できるまで待つ
    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
        if(waveSoundHandle.GetPlaySamplePosition() != 0)
        {
            break;
        }
    }
    EXPECT_NE(0, waveSoundHandle.GetPlaySamplePosition()) << "Start wave sound timed out.";
}

}

TEST(WaveSoundHandle, PlaySamplePositionTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SE_WIHAHO, "SE_WIHAHO");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SE_WIHAHO).IsSuccess());
    nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);

    int64_t currentPosition = waveSoundHandle.GetPlaySamplePosition();
    EXPECT_EQ(0, currentPosition);
    waveSoundHandle.StartPrepared();
    UpdateAndWaitForWaveStart(waveSoundHandle);
    int64_t previousPosition = currentPosition;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  VoiceCommandProcess を呼ばないと進まないことを確認
    const nn::TimeSpan TestTime = nn::TimeSpan::FromMilliSeconds(200);
    const int64_t stopPosition = waveSoundHandle.GetPlaySamplePosition();
    for(nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TestTime; elapsed += WaitTime)
    {
        soundArchivePlayer.Update();
        nn::os::SleepThread(WaitTime);

        currentPosition = waveSoundHandle.GetPlaySamplePosition();
        EXPECT_EQ(currentPosition, stopPosition);
    }
#endif

    uint32_t commandTag = 0;

    while(!nnt::atk::util::SendDummyCommand(&commandTag))
    {
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    bool isCommandSend = true;
    int testCount = 0;
    const int TestCountMax = 10;

    while(testCount < TestCountMax)
    {
        // 音源の再生が終了していた場合はそのまま終了
        if(!soundHandle.IsAttachedSound())
        {
            break;
        }

        if(isCommandSend)
        {
            if(nnt::atk::util::IsCommandFinished(commandTag))
            {
                currentPosition = waveSoundHandle.GetPlaySamplePosition();
                // SE_YOSHI は短いため、音源再生が終端まで行われた場合は検査を終了
                if(currentPosition == 0 && previousPosition != 0)
                {
                    break;
                }

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                //  vcmd 版では進まないこともあるため >= で判定します。(SIGLO-52881)
                EXPECT_GE(currentPosition, previousPosition);
#else
                EXPECT_GT(currentPosition, previousPosition);
#endif

                previousPosition = currentPosition;
                ++testCount;
                isCommandSend = nnt::atk::util::SendDummyCommand(&commandTag);
            }
        }
        else
        {
            isCommandSend = nnt::atk::util::SendDummyCommand(&commandTag);
        }

        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(WaveSoundHandle, LoopSoundPlaySamplePositionTest)
{
    nnt::atk::util::OnPreAtkTest();
    //  SIGLO-36012 の回帰テストです
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(WSD_ORGAN, "WSD_ORGAN");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE( soundArchivePlayer.StartSound(&soundHandle, WSD_ORGAN).IsSuccess() );
    nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);
    UpdateAndWaitForWaveStart(waveSoundHandle);

    nn::atk::WaveSoundDataInfo data;
    waveSoundHandle.ReadWaveSoundDataInfo( &data );

    //  WSD が TestLoopCountMax 回ループするまで繰り返します
    const int TestLoopCountMax = 5;
    bool isInLoop = false;
    for(int loop = 0; loop < TestLoopCountMax; loop++)
    {
        nn::TimeSpan spendTime = nn::TimeSpan::FromMilliSeconds( 0 );
        const nn::TimeSpan Timeout = nn::TimeSpan::FromMilliSeconds( 1000 );

        while( spendTime < Timeout )  //  不具合でループしない場合は無限ループに陥るため、Timeout で抜けるようにしています
        {
            const int64_t previousPosition = waveSoundHandle.GetPlaySamplePosition();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
            nn::os::SleepThread( WaitTime );

            const int64_t currentPosition = waveSoundHandle.GetPlaySamplePosition();

            //  ループ範囲に入るまでは data.loopStart <= currentPosition の確認を行いません
            if( !isInLoop )
            {
                isInLoop = data.loopStart <= currentPosition;
            }
            if( isInLoop )
            {
                EXPECT_LE( data.loopStart, currentPosition );
            }
            EXPECT_LE( currentPosition, data.loopEnd );

            if( previousPosition > currentPosition )
            {
                break;
            }
            spendTime += WaitTime;
        }

        EXPECT_LT( spendTime, Timeout ) << "It was time out to wait 1 loop.";
    }
    EXPECT_EQ( isInLoop, true ) << "It was not into loop.";

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(WaveSoundHandle, WaveSoundDataInfoTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SE_WIHAHO, "SE_WIHAHO");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::WaveSoundDataInfo info;

    nn::atk::WaveSoundHandle invalidWaveSoundHandle(&soundHandle);
    EXPECT_FALSE(invalidWaveSoundHandle.ReadWaveSoundDataInfo(&info));

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SE_WIHAHO).IsSuccess());
    nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);
    EXPECT_TRUE(waveSoundHandle.ReadWaveSoundDataInfo(&info));

    EXPECT_NE(0, info.loopEnd);
    // テストデータである SE_WIHAHO は 44100 Hz のサンプルレート
    EXPECT_EQ(44100, info.sampleRate);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(WaveSoundHandle, InvalidMixVolumeChannelTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.LoadData(SE_WIHAHO, "SE_WIHAHO");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    EXPECT_TRUE(soundArchivePlayer.PrepareSound(&soundHandle, SE_WIHAHO).IsSuccess());
    nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);

    // 5.1ch 音量設定のチェック
    nn::atk::MixVolume mixVolume;
    EXPECT_DEATH_IF_SUPPORTED(waveSoundHandle.SetMixVolume(-1, mixVolume), ".*");
    EXPECT_DEATH_IF_SUPPORTED(waveSoundHandle.SetMixVolume(nn::atk::WaveChannelMax, mixVolume), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
