﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk/fnd/basis/atkfnd_WorkBufferAllocator.h>
#include <nn/atk/fnd/basis/atkfnd_WorkBufferSizeCalculator.h>
#include <nnt.h>
#include <nn/mem/mem_StandardAllocator.h>

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(BufferAllocation, WorkBufferAllocatorDeathTest)
{
    // 不正なコンストラクタに関するテスト
    {
        const int TempSize = 8 * 1024;
        NN_ALIGNAS(4096) char temp[TempSize];

        // テストで使うポインタのアライメントを 1024 とする
        const size_t TestPtrAlignment = 1024;
        void* ValidPtr = reinterpret_cast<char*>(temp) + 1024;
        const size_t ValidSize = TempSize - TestPtrAlignment;

        // nullptr の代入
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(nullptr, ValidSize, 1), ".*");
        // 2 のべき乗でないアライメントの代入
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(ValidPtr, ValidSize, 0), ".*");
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(ValidPtr, ValidSize, 3), ".*");
        // ValidPtr が、要求したアライメントに沿っていない
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(ValidPtr, ValidSize, TestPtrAlignment * 2), ".*");
    }

    // 不正なバッファアロケーション
    {
        const int TempSize = 12 * 1024;
        NN_ALIGNAS(4096 * 2) char temp[TempSize];

        // テストで使うポインタのアライメントを 4096 とする
        const size_t BufferAlignment = 4096;
        void* buffer = reinterpret_cast<char*>(temp) + 4096;
        const size_t BufferSize = TempSize - BufferAlignment; // (= 8192)

        {
            nn::atk::detail::fnd::WorkBufferAllocator invalidAllocator(buffer, BufferSize);
            // 確保サイズオーバー
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(BufferSize + 1), ".*");
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(invalidAllocator.GetFreeSize() + 1), ".*");
        }

        {
            nn::atk::detail::fnd::WorkBufferAllocator invalidAllocator(buffer, 10);
            invalidAllocator.Allocate(2, 1);
            // アライメント調整をした結果のサイズオーバー
            // (バッファサイズ: 10byte, 確保サイズ: 12byte)
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(4, 8), ".*");
        }

        {
            nn::atk::detail::fnd::WorkBufferAllocator invalidAllocator(buffer, BufferAlignment * 2);
            invalidAllocator.Allocate(1, BufferAlignment, 2);
            // アライメント調整をした結果のサイズオーバー
            // (バッファサイズ: BufferAlignment * 2, 確保サイズ: BufferAlignment * 2 + 1)
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(1, BufferAlignment), ".*");
        }

        {
            nn::atk::detail::fnd::WorkBufferAllocator invalidAllocator(buffer, 20);
            // アライメント調整をした結果のサイズオーバー
            // (バッファサイズ: 20byte, 確保サイズ: 36byte)
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(4, 8, 5), ".*");
            // 不正な count での Allocate
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(4, 2, 0), ".*");
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(4, 2, -1), ".*");
        }

        {
            nn::atk::detail::fnd::WorkBufferAllocator invalidAllocator(buffer, BufferSize);
            // 不正な count での Allocate
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate<int>(0), ".*");
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate<int>(-1), ".*");
        }

        {
            nn::atk::detail::fnd::WorkBufferAllocator invalidAllocator(buffer, BufferSize);
            // アロケーション時のアライメントが 2 のべき乗でない
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(4, 0), ".*");
            EXPECT_DEATH_IF_SUPPORTED(invalidAllocator.Allocate(4, 3), ".*");
        }
    }
}
#endif

TEST(BufferAllocation, WorkBufferAllocatorTest)
{
    const int TempSize = 12 * 1024;
    NN_ALIGNAS(4096 * 2) char temp[TempSize];

    // テストで使うポインタのアライメントを 4096 とする
    const size_t BufferAlignment = 4096;
    void* buffer = reinterpret_cast<char*>(temp) + 4096;
    const size_t BufferSize = TempSize - BufferAlignment; // (= 8192)

    {
        nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(buffer, BufferSize, BufferAlignment);
        // 初期状態のチェック
        EXPECT_EQ(bufferAllocator.GetSize(), bufferAllocator.GetFreeSize());
        EXPECT_EQ(0u, bufferAllocator.GetAllocSize());
        EXPECT_EQ(BufferSize, bufferAllocator.GetFreeSize());

        // 初期状態からの確保
        auto allocatedSize = bufferAllocator.GetAllocSize();
        size_t testAllocationSize = 1;
        size_t testAlignment = BufferAlignment;
        size_t expectedSize = nn::util::align_up(allocatedSize, testAlignment) + testAllocationSize;
        void* testBuffer = bufferAllocator.Allocate(1, testAlignment);
        EXPECT_TRUE(nn::util::is_aligned(reinterpret_cast<uintptr_t>(testBuffer), testAlignment));
        EXPECT_EQ(BufferSize, bufferAllocator.GetSize());
        EXPECT_EQ(expectedSize, bufferAllocator.GetAllocSize());
        EXPECT_EQ(BufferSize - expectedSize, bufferAllocator.GetFreeSize());

        // サイズのみを引数に取るバッファ確保
        allocatedSize = bufferAllocator.GetAllocSize();
        testAllocationSize = 4;
        expectedSize = allocatedSize + testAllocationSize;
        testBuffer = bufferAllocator.Allocate(testAllocationSize);
        EXPECT_EQ(expectedSize, bufferAllocator.GetAllocSize());
        EXPECT_EQ(BufferSize - expectedSize, bufferAllocator.GetFreeSize());

        // サイズとアライメントを引数に取るバッファ確保
        allocatedSize = bufferAllocator.GetAllocSize();
        testAllocationSize = 1;
        testAlignment = 4;
        expectedSize = nn::util::align_up(allocatedSize, testAlignment) + testAllocationSize;
        testBuffer = bufferAllocator.Allocate(testAllocationSize, testAlignment);
        EXPECT_TRUE(nn::util::is_aligned(reinterpret_cast<uintptr_t>(testBuffer), testAlignment));
        EXPECT_EQ(expectedSize, bufferAllocator.GetAllocSize());
        EXPECT_EQ(BufferSize - expectedSize, bufferAllocator.GetFreeSize());

        // サイズ、アライメントと個数を引数に取るバッファ確保
        allocatedSize = bufferAllocator.GetAllocSize();
        testAllocationSize = 4;
        testAlignment = 8;
        int testCount = 5;
        expectedSize = allocatedSize;
        for(int i = 0; i < testCount; ++i)
        {
            expectedSize = nn::util::align_up(expectedSize, testAlignment) + testAllocationSize;
        }
        testBuffer = bufferAllocator.Allocate(testAllocationSize, testAlignment, testCount);
        EXPECT_TRUE(nn::util::is_aligned(reinterpret_cast<uintptr_t>(testBuffer), testAlignment));
        EXPECT_EQ(expectedSize, bufferAllocator.GetAllocSize());
        EXPECT_EQ(BufferSize - expectedSize, bufferAllocator.GetFreeSize());

        // WorkBufferAllocator が残りすべてのバッファを確保
        testBuffer = bufferAllocator.Allocate(bufferAllocator.GetFreeSize());
        EXPECT_EQ(BufferSize, bufferAllocator.GetAllocSize());
        EXPECT_EQ(0u, bufferAllocator.GetFreeSize());
        EXPECT_EQ(BufferSize, bufferAllocator.GetSize());
    }

    // テンプレートを用いる Allocate のチェック
    {
        nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(buffer, BufferSize, BufferAlignment);

        bufferAllocator.Allocate(1);
        float* testBuffer = bufferAllocator.Allocate<float>();
        EXPECT_TRUE(nn::util::is_aligned(reinterpret_cast<uintptr_t>(testBuffer), NN_ALIGNOF(float)));
        const size_t ExpectSize = nn::util::align_up(1, NN_ALIGNOF(float)) + sizeof(float);
        EXPECT_EQ(ExpectSize, bufferAllocator.GetAllocSize());
    }

    // テンプレート + count 引数を用いる Allocate のチェック
    {
        nn::atk::detail::fnd::WorkBufferAllocator bufferAllocator(buffer, BufferSize, BufferAlignment);

        bufferAllocator.Allocate(1);
        float* testBuffer = bufferAllocator.Allocate<float>(4);
        EXPECT_TRUE(nn::util::is_aligned(reinterpret_cast<uintptr_t>(testBuffer), NN_ALIGNOF(float)));
        const size_t ExpectSize = nn::util::align_up(1, NN_ALIGNOF(float)) + sizeof(float) * 4;
        EXPECT_EQ(ExpectSize, bufferAllocator.GetAllocSize());
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(BufferAllocation, WorkBufferSizeCalculatorDeathTest)
{
    nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
    // アライメントが 2 のべき乗でない
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add(2, 0), ".*");
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add(2, 3), ".*");
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add(2, 12, 5), ".*");

    // 不正な個数での Add
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add(4, 4, 0), ".*");
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add(4, 4, -1), ".*");
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add<float>(0), ".*");
    EXPECT_DEATH_IF_SUPPORTED(calculator.Add<float>(-1), ".*");

}
#endif

namespace
{

enum AllocationMode
{
    AllocationMode_Size,
    AllocationMode_SizeWithAlignment,
    AllocationMode_Full
};

struct WorkBufferSizeCalculatorTestCondition
{
    AllocationMode mode;
    size_t expect;
    size_t size;
    size_t alignment;
    int count;
};

}

void CheckRequiredBufferSize(const WorkBufferSizeCalculatorTestCondition* const pCondition, int count, const char* const label)
{
    NN_ABORT_UNLESS_NOT_NULL(pCondition);
    NN_ABORT_UNLESS_GREATER_EQUAL(count, 0);

    nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
    EXPECT_EQ(0u, calculator.GetSize());
    for(auto i = 0; i < count; ++i)
    {
        const WorkBufferSizeCalculatorTestCondition& condition = pCondition[i];
        if(condition.mode == AllocationMode_Size)
        {
            calculator.Add(condition.size);
        }
        else if(pCondition[i].mode == AllocationMode_SizeWithAlignment)
        {
            calculator.Add(condition.size, condition.alignment);
        }
        else if(pCondition[i].mode == AllocationMode_Full)
        {
            calculator.Add(condition.size, condition.alignment, condition.count);
        }

        EXPECT_EQ(condition.expect, calculator.GetSize()) << label << " failed.";
    }
}

TEST(BufferAllocation, WorkBufferSizeCalculatorTest)
{
    // WorkBufferSizeCalculatorTestCondition は (どのように Add するか, 期待する値, 要求メモリ量, 要求アライメント, 要求個数) の順にパラメーターが並んでいます
    {
        // サイズのみを引数に取るバッファ要求量計算
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_Size, 2, 2, 0, 0},
            {AllocationMode_Size, 6, 4, 0, 0},
            {AllocationMode_Size, 12, 6, 0, 0},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size)");
    }

    {
        // サイズ、アライメントを引数に取るバッファ要求量計算(ケース 1)
        // 今まで Add したアラインよりも大きなアラインで Add されないケース
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 16, 16, 64, 0},
            {AllocationMode_SizeWithAlignment, 48, 16, 32, 0},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size and alignment, case 1)");
    }

    {
        // サイズ、アライメントを引数に取るバッファ要求量計算(ケース 2)
        // 今まで Add したアラインよりも大きなアラインで Add されることが 1 回あるケース
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 2, 2, 1, 0},
            {AllocationMode_SizeWithAlignment, 6, 1, 4, 0},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size and alignment, case 2)");
    }

    {
        // サイズ、アライメントを引数に取るバッファ要求量計算(ケース 3)
        // 今まで Add したアラインよりも大きなアラインで Add されることが 1 回、そうでないのが 1 回あるケース
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 1, 1, 8, 0},
            {AllocationMode_SizeWithAlignment, 6, 2, 4, 0},
            {AllocationMode_SizeWithAlignment, 17, 1, 16, 0},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size and alignment, case 3)");
    }

    {
        // サイズ、アライメントを引数に取るバッファ要求量計算(ケース 4)
        // 今まで Add したアラインよりも大きなアラインで Add されることが 1 回、そうでないのが 1 回あるケース
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 1, 1, 8, 0},
            {AllocationMode_SizeWithAlignment, 10, 6, 4, 0},
            {AllocationMode_SizeWithAlignment, 25, 1, 16, 0},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size and alignment, case 4)");
    }

    {
        // サイズ、アライメントを引数に取るバッファアロケーションテスト（ケース 5）
        // 今まで Add したアラインよりも大きなアラインで Add されることが 2 回あるケース
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 3, 3, 2, 0},
            {AllocationMode_SizeWithAlignment, 12, 6, 4, 0},
            {AllocationMode_SizeWithAlignment, 27, 1, 16, 0},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size and alignment, case 5)");
    }

    {
        // サイズ、アライメント、個数を引数に取るバッファ要求量計算
        const WorkBufferSizeCalculatorTestCondition conditions[] =
        {
            {AllocationMode_Full, 28, 4, 8, 4},
            {AllocationMode_Full, 42, 6, 4, 2},
        };
        CheckRequiredBufferSize(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer size calculation test(size, alignment, and count)");
    }

    {
        // テンプレートを用いるバッファ要求量計算のチェック
        nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
        size_t testSize = 1;
        size_t expectSize = testSize;
        calculator.Add(testSize);
        EXPECT_EQ(expectSize, calculator.GetSize());

        testSize = sizeof(int);
        expectSize = nn::util::align_up(expectSize, NN_ALIGNOF(int)) + testSize;
        calculator.Add<int>();
        EXPECT_EQ(expectSize, calculator.GetSize()) << "Buffer size calculation test(template) failed.";
    }

    {
        // テンプレート、個数を用いるバッファ要求量計算のチェック
        nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
        size_t testSize = 1;
        size_t expectSize = testSize;
        calculator.Add(testSize);
        EXPECT_EQ(expectSize, calculator.GetSize());

        testSize = sizeof(double);
        auto testCount = 4;
        for(auto i = 0; i < testCount; ++i)
        {
            expectSize = nn::util::align_up(expectSize, NN_ALIGNOF(double)) + testSize;
        }
        calculator.Add<double>(testCount);
        EXPECT_EQ(expectSize, calculator.GetSize()) << "Buffer size calculation Test(template and count) failed.";
    }
}

namespace
{

struct AllocationTestCondition
{
    AllocationMode mode;
    size_t size;
    size_t alignment;
    int count;
};

}

void CheckBufferAllocation(const AllocationTestCondition* const pCondition, int count, const char* const label)
{
    NN_ABORT_UNLESS_NOT_NULL(pCondition);
    NN_ABORT_UNLESS_GREATER_EQUAL(count, 0);

    nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
    size_t firstAlignment = 0;
    size_t maxAlignment = 0;
    for(auto i = 0; i < count; ++i)
    {
        const AllocationTestCondition& condition = pCondition[i];
        if(condition.mode == AllocationMode_Size)
        {
            calculator.Add(condition.size);
        }
        else if(condition.mode == AllocationMode_SizeWithAlignment)
        {
            calculator.Add(condition.size, condition.alignment);
        }
        else if(condition.mode == AllocationMode_Full)
        {
            calculator.Add(condition.size, condition.alignment, condition.count);
        }

        if(firstAlignment == 0)
        {
            firstAlignment = condition.alignment != 0 ? condition.alignment : 1;
            maxAlignment = firstAlignment;
        }
        if(maxAlignment < condition.alignment)
        {
            maxAlignment = condition.alignment;
        }
    }

    // テストで実行するために、大きなアライメントとサイズを事前に確保しておく
    NN_ALIGNAS(4096 * 2) char Temp[1024 * 16];
    bool isEqualBufferSize = false;
    for(auto offset = firstAlignment; offset <= maxAlignment; offset += firstAlignment)
    {
        void* buffer = reinterpret_cast<char*>(Temp) + offset;
        size_t bufferSize = calculator.GetSize();

        nn::atk::detail::fnd::WorkBufferAllocator allocator(buffer, bufferSize, firstAlignment);

        for(auto i = 0; i < count; ++i)
        {
            const AllocationTestCondition& condition = pCondition[i];
            if(condition.mode == AllocationMode_Size)
            {
                allocator.Allocate(condition.size);
            }
            else if(condition.mode == AllocationMode_SizeWithAlignment)
            {
                allocator.Allocate(condition.size, condition.alignment);
            }
            else if(condition.mode == AllocationMode_Full)
            {
                allocator.Allocate(condition.size, condition.alignment, condition.count);
            }
        }

        if(!isEqualBufferSize && allocator.GetFreeSize() == 0u)
        {
            isEqualBufferSize = true;
        }
    }

    EXPECT_TRUE(isEqualBufferSize) << label << " failed.";
}

TEST(BufferAllocation, BufferAllocationConbinationTest)
{
    // AllocationTestCondition は (どのように Add するか, 要求メモリ量, 要求アライメント, 要求個数) の順にパラメーターが並んでいます
    {
        // サイズのみを引数に取るバッファアロケーションテスト
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_Size, 2, 0, 0},
            {AllocationMode_Size, 4, 0, 0},
            {AllocationMode_Size, 6, 0, 0},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size)");
    }

    {
        // サイズ、アライメントを引数に取るバッファアロケーションテスト（ケース 1）
        // 今まで Add したアラインよりも大きなアラインで Add されないケース
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 16, 64, 0},
            {AllocationMode_SizeWithAlignment, 16, 32, 0},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size and alignment, case 1)");
    }

    {
        // サイズ、アライメントを引数に取るバッファアロケーションテスト（ケース 2）
        // 今まで Add したアラインよりも大きなアラインで Add されることが 1 回あるケース
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 2, 1, 0},
            {AllocationMode_SizeWithAlignment, 1, 4, 0},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size and alignment, case 2)");
    }

    {
        // サイズ、アライメントを引数に取るバッファアロケーションテスト（ケース 3）
        // 今まで Add したアラインよりも大きなアラインで Add されることが 1 回、そうでないのが 1 回あるケース
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 1, 8, 0},
            {AllocationMode_SizeWithAlignment, 2, 4, 0},
            {AllocationMode_SizeWithAlignment, 1, 16, 0},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size and alignment, case 3)");
    }

    {
        // サイズ、アライメントを引数に取るバッファアロケーションテスト（ケース 4）
        // 今まで Add したアラインよりも大きなアラインで Add されることが 1 回、そうでないのが 1 回あるケース
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 1, 8, 0},
            {AllocationMode_SizeWithAlignment, 6, 4, 0},
            {AllocationMode_SizeWithAlignment, 1, 16, 0},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size and alignment, case 4)");
    }

    {
        // サイズ、アライメントを引数に取るバッファアロケーションテスト（ケース 5）
        // 今まで Add したアラインよりも大きなアラインで Add されることが 2 回あるケース
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_SizeWithAlignment, 3, 2, 0},
            {AllocationMode_SizeWithAlignment, 6, 4, 0},
            {AllocationMode_SizeWithAlignment, 1, 16, 0},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size and alignment, case 5)");
    }

    {
        // サイズ、アライメント、個数を引数に取るバッファアロケーションテスト
        const AllocationTestCondition conditions[] =
        {
            {AllocationMode_Full, 4, 8, 4},
            {AllocationMode_Full, 6, 4, 2},
        };
        CheckBufferAllocation(conditions, sizeof(conditions) / sizeof(conditions[0]), "Buffer allocation combination test(size, alignment, and count)");
    }

    {
        // テンプレートを用いるバッファアロケーションテスト
        nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
        size_t alignment = 16;
        calculator.Add(1, alignment);
        calculator.Add<uintptr_t>();

        NN_ALIGNAS(4096 * 2) char Temp[1024 * 16];
        void* buffer = reinterpret_cast<char*>(Temp) + alignment;
        size_t bufferSize = calculator.GetSize();

        nn::atk::detail::fnd::WorkBufferAllocator allocator(buffer, bufferSize);
        allocator.Allocate(1);
        allocator.Allocate<uintptr_t>();

        EXPECT_EQ(0u, allocator.GetFreeSize()) << "Buffer allocation combination test(template) failed.";
    }

    {
        // テンプレート、個数を用いるバッファアロケーションテスト
        nn::atk::detail::fnd::WorkBufferSizeCalculator calculator;
        size_t alignment = 16;
        int testCount = 4;
        calculator.Add(1, alignment);
        calculator.Add<float>(testCount);

        NN_ALIGNAS(4096 * 2) char Temp[1024 * 16];
        void* buffer = reinterpret_cast<char*>(Temp) + alignment;
        size_t bufferSize = calculator.GetSize();

        nn::atk::detail::fnd::WorkBufferAllocator allocator(buffer, bufferSize);
        allocator.Allocate(1);
        allocator.Allocate<float>(testCount);

        EXPECT_EQ(0u, allocator.GetFreeSize()) << "Buffer allocation combination test(template and count) failed.";
    }
}
