﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"
#include "testAtk_TestEffectUtil.h"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {
const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup g_FsSetup;
nnt::atk::util::AtkCommonSetup g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

bool IsReverbChannelModeSupported(nn::atk::EffectBase::ChannelMode channelMode)
{
    return channelMode == nn::atk::EffectBase::ChannelMode_1Ch || channelMode == nn::atk::EffectBase::ChannelMode_2Ch
        || channelMode == nn::atk::EffectBase::ChannelMode_4Ch || channelMode == nn::atk::EffectBase::ChannelMode_6Ch;
}
}

// TODO: チャンネル設定以外のテストを追加する
#ifndef NN_SDK_BUILD_RELEASE
TEST(Effect, ReverbChannelSettingDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectReverb reverb;

        if (!IsReverbChannelModeSupported(channelMode))
        {
            continue;
        }

        reverb.SetChannelMode(channelMode);
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);
        nn::atk::ChannelIndex* pInvalidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
        NN_ABORT_UNLESS_NOT_NULL(pInvalidChannelSetting);
        // チャンネルの重複
        if (ChannelCount > 1)
        {
            for (int i = 0; i < ChannelCount; ++i)
            {
                if (i == 0 || i == 1)
                {
                    pInvalidChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
                else
                {
                    pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
                }
            }
            EXPECT_DEATH_IF_SUPPORTED(reverb.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");
        }

        // チャンネル設定の値に -1 が含まれる
        for (int i = 0; i < ChannelCount; ++i)
        {
            pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i - 1);
        }
        EXPECT_DEATH_IF_SUPPORTED(reverb.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");

        // チャンネル設定の値に nn::atk::ChannelIndex_Count が含まれる
        for (int i = 0; i < ChannelCount; ++i)
        {
            if (i == ChannelCount - 1)
            {
                pInvalidChannelSetting[i] = nn::atk::ChannelIndex_Count;
            }
            else
            {
                pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
            }
        }
        EXPECT_DEATH_IF_SUPPORTED(reverb.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");

        g_Allocator.Free(reinterpret_cast<void*>(pInvalidChannelSetting));
    }
    g_Allocator.Finalize();
}
#endif

TEST(Effect, ReverbSetChannelCountTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectReverb reverb;
        if (IsReverbChannelModeSupported(channelMode))
        {
            EXPECT_TRUE(reverb.SetChannelMode(channelMode));
            ASSERT_EQ(channelMode, reverb.GetChannelMode());
            const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);
            nn::atk::ChannelIndex* pReverbChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pReverbChannelSetting);
            reverb.GetChannelIndex(pReverbChannelSetting, ChannelCount);
            for (int i = 0; i < ChannelCount; ++i)
            {
                EXPECT_EQ(static_cast<nn::atk::ChannelIndex>(i), pReverbChannelSetting[i]);
            }
        }
    }

    g_Allocator.Finalize();
}

TEST(Effect, ReverbSetChannelIndexTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectReverb reverb;
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);

        if (IsReverbChannelModeSupported(channelMode))
        {
            // サポートしているチャンネル数の場合
            EXPECT_TRUE(reverb.SetChannelMode(channelMode));
            EXPECT_EQ(channelMode, reverb.GetChannelMode());

            nn::atk::ChannelIndex* pChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pChannelSetting);
            // 初期化
            for (int i = 0; i < ChannelCount; ++i)
            {
                pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
            }

            // 適切なチャンネル設定となるものについて総当たり検査
            while (pChannelSetting[ChannelCount - 1] != nn::atk::ChannelIndex_Count)
            {
                // テストケースとして適切か(チャンネル設定の被りがないか)確認
                bool isValidChannelSetting = true;
                for (int i = 0; i < ChannelCount; ++i)
                {
                    for (int j = i + 1; j < ChannelCount; ++j)
                    {
                        isValidChannelSetting &= (pChannelSetting[i] != pChannelSetting[j]);
                    }
                }

                // テスト本体
                if (isValidChannelSetting)
                {
                    EXPECT_TRUE(reverb.SetChannelIndex(pChannelSetting, channelMode));

                    ASSERT_EQ(channelMode, reverb.GetChannelMode());
                    nn::atk::ChannelIndex* pReverbChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
                    NN_ABORT_UNLESS_NOT_NULL(pReverbChannelSetting);
                    NN_ABORT_UNLESS_NOT_NULL(pReverbChannelSetting);
                    reverb.GetChannelIndex(pReverbChannelSetting, ChannelCount);
                    for (int i = 0; i < ChannelCount; ++i)
                    {
                        EXPECT_EQ(pChannelSetting[i], pReverbChannelSetting[i]);
                    }
                    g_Allocator.Free(reinterpret_cast<void*>(pReverbChannelSetting));
                }

                // テストケースの更新
                for (int i = 0; i < ChannelCount; ++i)
                {
                    pChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(pChannelSetting[i] + 1);
                    // 繰り上がりしない場合
                    if (pChannelSetting[i] != nn::atk::ChannelIndex_Count || i == ChannelCount - 1)
                    {
                        break;
                    }

                    pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
            }

            g_Allocator.Free(reinterpret_cast<void*>(pChannelSetting));
        }
        else
        {
            // サポートしていないチャンネル数の場合
            EXPECT_FALSE(reverb.SetChannelMode(channelMode));
        }
    }

    g_Allocator.Finalize();
}

TEST(Effect, ReverbAppendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectReverb reverb;
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);

        // 使用するチャンネル設定の初期化
        nn::atk::ChannelIndex* pValidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
        NN_ABORT_UNLESS_NOT_NULL(pValidChannelSetting);
        // 初期化
        for (int i = 0; i < ChannelCount; ++i)
        {
            pValidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
        }

        // Append 前に変更できることをテストする
        if (IsReverbChannelModeSupported(channelMode))
        {
            EXPECT_TRUE(reverb.SetChannelMode(channelMode));
            EXPECT_TRUE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));
        }
        else
        {
            // 変更できない場合はそれを確認して次のテスト項目へ
            EXPECT_FALSE(reverb.SetChannelMode(channelMode));
            EXPECT_FALSE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));
            continue;
        }

        // エフェクトの Append
        EXPECT_TRUE(reverb.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000));
        size_t reverbBufferSize = nn::util::align_up(reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);
        nn::audio::MemoryPoolType memoryPoolForReverb;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForReverb, reverbBuffer, reverbBufferSize);
        reverb.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

        // Append で送ったコマンドが処理されるのを待つ
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend(reverb);

        // Append 完了後に変更ができないことをテストする
        EXPECT_FALSE(reverb.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000));
        EXPECT_FALSE(reverb.SetChannelMode(channelMode));
        EXPECT_FALSE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));

        reverb.SetEnabled(false);
        nnt::atk::effectUtil::UpdateAndProcess(soundArchivePlayer);
        nnt::atk::effectUtil::WaitForEffectClear(reverb);
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &reverb);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForReverb);
        g_Allocator.Free(reverbBuffer);

        // Remove 後に変更できることをテストする
        EXPECT_TRUE(reverb.SetChannelMode(channelMode));
        EXPECT_TRUE(reverb.SetChannelIndex(pValidChannelSetting, channelMode));
        EXPECT_TRUE(reverb.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000));

        g_Allocator.Free(reinterpret_cast<void*>(pValidChannelSetting));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
