﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"
#include "testAtk_TestEffectUtil.h"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>


namespace {
const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup g_FsSetup;
nnt::atk::util::AtkCommonSetup g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

const int AdditionalSubMixIndex = 1;
}


#ifndef NN_SDK_BUILD_RELEASE
void AppendEffectForDeathTest(nn::atk::EffectDelay& delay, void* buffer, size_t bufferSize)
{
    delay.SetEnabled( true );
    EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, &delay, buffer, bufferSize, nn::atk::OutputDevice_Main, AdditionalSubMixIndex ) );
    nnt::atk::util::WaitForProcessCommand();
}
TEST(Effect, AdditionalEffectBusDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    //  追加エフェクトバスは 2Ch まで対応であるため、4Ch のエフェクトを追加してデステスト
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();

        nnt::atk::util::AtkCommonSetup::InitializeParam parameter;
        parameter.GetSoundSystemParam().enableAdditionalEffectBus = true;
        g_AtkSetup.Initialize(parameter, g_Allocator);

        nn::atk::EffectDelay delay;
        delay.SetEnabled( true );
        delay.SetChannelMode( nn::atk::EffectBase::ChannelMode_4Ch );

        const size_t bufferSize = nn::util::align_up( delay.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment );
        NN_ABORT_UNLESS_NOT_NULL( buffer );

        nn::audio::MemoryPoolType memoryPool;
        nn::atk::SoundSystem::AttachMemoryPool( &memoryPool, buffer, bufferSize );

        EXPECT_DEATH_IF_SUPPORTED( AppendEffectForDeathTest( delay, buffer, bufferSize ), ".*" );

        nn::atk::SoundSystem::DetachMemoryPool( &memoryPool );
        g_Allocator.Free( buffer );

        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }

    //  enableAdditionalEffectBus を有効にしないデステスト
    {
        g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
        g_FsSetup.Initialize();

        nnt::atk::util::AtkCommonSetup::InitializeParam parameter;
        g_AtkSetup.Initialize(parameter, g_Allocator);

        nn::atk::EffectDelay delay;
        delay.SetEnabled( true );
        delay.SetChannelMode( nn::atk::EffectBase::ChannelMode_2Ch );

        const size_t bufferSize = nn::util::align_up( delay.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment );
        NN_ABORT_UNLESS_NOT_NULL( buffer );

        nn::audio::MemoryPoolType memoryPool;
        nn::atk::SoundSystem::AttachMemoryPool( &memoryPool, buffer, bufferSize );

        EXPECT_DEATH_IF_SUPPORTED( AppendEffectForDeathTest( delay, buffer, bufferSize ), ".*" );

        nn::atk::SoundSystem::DetachMemoryPool( &memoryPool );
        g_Allocator.Free( buffer );

        g_AtkSetup.Finalize(g_Allocator);
        g_FsSetup.Finalize();
        g_Allocator.Finalize();
    }
}
void AppendEffectToAdditionalSubMixForDeathTest(nn::atk::EffectAux& aux, void* buffer, size_t bufferSize)
{
    aux.SetEnabled( true );
    EXPECT_TRUE( nn::atk::SoundSystem::AppendEffectToAdditionalSubMix( &aux, buffer, bufferSize ) );
    nnt::atk::util::WaitForProcessCommand();
}
TEST(Effect, AdditionalSubMixDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    //  enableAdditionalSubMix を有効にしないデステスト
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nnt::atk::util::AtkCommonSetup::InitializeParam parameter;
    g_AtkSetup.Initialize(parameter, g_Allocator);

    nnt::atk::effectUtil::EffectAuxTest aux;

    const size_t bufferSize = nn::util::align_up( nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize( &aux ), nn::audio::MemoryPoolType::SizeGranularity );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment );
    nn::audio::MemoryPoolType memotyPool;
    nn::atk::SoundSystem::AttachMemoryPool( &memotyPool, buffer, bufferSize );

    EXPECT_DEATH_IF_SUPPORTED( AppendEffectToAdditionalSubMixForDeathTest( aux, buffer, bufferSize ), ".*" );

    nn::atk::SoundSystem::DetachMemoryPool( &memotyPool );
    g_Allocator.Free( buffer );

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(Effect, AdditionalEffectBusAppendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nnt::atk::util::AtkCommonSetup::InitializeParam parameter;
    parameter.GetSoundSystemParam().enableAdditionalEffectBus = true;
    g_AtkSetup.Initialize(parameter, g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    //  追加のエフェクトバスの AuxBus_A に対して delay を追加し、ClearEffect で取り除くテスト
    {
        nn::atk::EffectDelay delay;
        delay.SetEnabled( true );
        delay.SetChannelMode( nn::atk::EffectBase::ChannelMode_2Ch );

        const size_t bufferSize = nn::util::align_up( delay.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment );
        NN_ABORT_UNLESS_NOT_NULL( buffer );

        nn::audio::MemoryPoolType memoryPool;
        nn::atk::SoundSystem::AttachMemoryPool( &memoryPool, buffer, bufferSize );

        EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, &delay, buffer, bufferSize, nn::atk::OutputDevice_Main, AdditionalSubMixIndex ) );
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend(delay);

        delay.SetEnabled( false );
        nnt::atk::effectUtil::UpdateAndProcess( soundArchivePlayer );
        nnt::atk::effectUtil::WaitForEffectClear( delay );
        nn::atk::SoundSystem::ClearEffect( nn::atk::AuxBus_A, nn::atk::OutputDevice_Main, AdditionalSubMixIndex );

        nn::atk::SoundSystem::DetachMemoryPool( &memoryPool );
        g_Allocator.Free( buffer );
    }

    //  追加のエフェクトバスの AuxBus_B に対して reverb を追加し、RemoveEffect で取り除くテスト
    {
        nn::atk::EffectReverb reverb;
        reverb.SetEnabled( true );
        reverb.SetChannelMode( nn::atk::EffectBase::ChannelMode_2Ch );

        const size_t bufferSize = nn::util::align_up( reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment );
        NN_ABORT_UNLESS_NOT_NULL( buffer );

        nn::audio::MemoryPoolType memoryPool;
        nn::atk::SoundSystem::AttachMemoryPool( &memoryPool, buffer, bufferSize );

        EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_B, &reverb, buffer, bufferSize, nn::atk::OutputDevice_Main, AdditionalSubMixIndex ) );
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend(reverb);

        reverb.SetEnabled( false );
        nnt::atk::effectUtil::UpdateAndProcess( soundArchivePlayer );
        nnt::atk::effectUtil::WaitForEffectClear( reverb );
        nn::atk::SoundSystem::RemoveEffect( nn::atk::AuxBus_B, &reverb, nn::atk::OutputDevice_Main, AdditionalSubMixIndex );

        nn::atk::SoundSystem::DetachMemoryPool( &memoryPool );
        g_Allocator.Free( buffer );
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(Effect, AdditionalSubMixAppendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nnt::atk::util::AtkCommonSetup::InitializeParam parameter;
    parameter.GetSoundSystemParam().enableAdditionalSubMix = true;
    g_AtkSetup.Initialize(parameter, g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nnt::atk::effectUtil::EffectAuxTest aux;

    const size_t bufferSize = nn::util::align_up( nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize( &aux ), nn::audio::MemoryPoolType::SizeGranularity );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment );
    nn::audio::MemoryPoolType memotyPool;
    nn::atk::SoundSystem::AttachMemoryPool( &memotyPool, buffer, bufferSize );

    //  追加の SubMix に対して AuxEffect を追加し、ClearEffectFromAdditionalSubMix で取り除くテスト
    {
        aux.SetEnabled( true );
        EXPECT_TRUE( nn::atk::SoundSystem::AppendEffectToAdditionalSubMix( &aux, buffer, bufferSize ) );
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend( aux );

        aux.SetEnabled( false );
        nnt::atk::effectUtil::UpdateAndProcess( soundArchivePlayer );
        nnt::atk::effectUtil::WaitForEffectClear( aux );
        nn::atk::SoundSystem::ClearEffectFromAdditionalSubMix();
    }

    //  追加の SubMix に対して AuxEffect を追加し、RemoveEffectFromAdditionalSubMix で取り除くテスト
    {
        aux.SetEnabled( true );
        EXPECT_TRUE( nn::atk::SoundSystem::AppendEffectToAdditionalSubMix( &aux, buffer, bufferSize ) );
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend( aux );

        aux.SetEnabled( false );
        nnt::atk::effectUtil::UpdateAndProcess( soundArchivePlayer );
        nnt::atk::effectUtil::WaitForEffectClear( aux );
        nn::atk::SoundSystem::RemoveEffectFromAdditionalSubMix( &aux );
    }

    nn::atk::SoundSystem::DetachMemoryPool( &memotyPool );
    g_Allocator.Free( buffer );

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
