﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include "testAtk_TestRecorder.h"


//------------------------------------------------------------------------------
//  コンストラクタです。
//------------------------------------------------------------------------------
TestRecorder::TestRecorder() NN_NOEXCEPT :
    m_IsInitialized(false),
    m_IsRecording(false),
    m_pOutputFileName("rec.wav"),
    m_Recorder(),
    m_RecordingFileStream()
{
}

//------------------------------------------------------------------------------
//  デストラクタです。
//------------------------------------------------------------------------------
TestRecorder::~TestRecorder() NN_NOEXCEPT
{
}

//---------------------------------------------------------------------------
//  レコーダーを初期化します。
//---------------------------------------------------------------------------
void TestRecorder::Initialize(void* buffer, size_t bufferSize, void *pThreadStack, size_t threadStackSize) NN_NOEXCEPT
{
    if ( m_IsInitialized )
    {
        return;
    }

    NN_ABORT_UNLESS_NOT_NULL(pThreadStack);
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS_GREATER_EQUAL(threadStackSize, nn::atk::DeviceOutRecorder::RequiredThreadStackSize);
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSize, GetRequiredMemorySize());

    m_Recorder.Initialize(buffer, bufferSize, pThreadStack, threadStackSize);
    m_IsInitialized = true;
}

//---------------------------------------------------------------------------
//  レコーダーの終了処理を行います。
//---------------------------------------------------------------------------
void TestRecorder::Finalize() NN_NOEXCEPT
{
    if ( !m_IsInitialized )
    {
        return;
    }

    // 録音中の場合は録音停止後少し待ってから終了処理を行う
    if ( m_IsRecording )
    {
        StopRecording();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        m_IsRecording = false;
    }

    m_Recorder.Finalize();
    m_IsInitialized = false;
}

//---------------------------------------------------------------------------
//  レコーダーのバッファに必要なサイズを取得します。
//---------------------------------------------------------------------------
size_t TestRecorder::GetRequiredMemorySize() NN_NOEXCEPT
{
    return m_Recorder.GetRequiredMemorySizeForRecording();
}

//---------------------------------------------------------------------------
//  録音を開始します。
//---------------------------------------------------------------------------
void TestRecorder::StartRecording(uint32_t frames) NN_NOEXCEPT
{
    if ( !m_IsInitialized )
    {
        return;
    }

    nn::Result result = nn::fs::DeleteFile(m_pOutputFileName);
    if(!(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result)))
    {
        NN_LOG( "[REC] StartRecording : cannot delete output file: %s\n", m_pOutputFileName );
        return;
    }

    nn::fs::CreateFile(m_pOutputFileName, 0);

    if ( m_RecordingFileStream.Open(
        m_pOutputFileName,
        nn::atk::detail::fnd::FileStream::AccessMode_AllowAppendAndWrite
        ).IsFailed() )
    {
        NN_LOG( "[REC] StartRecording : failed to open wav file: %s\n", m_pOutputFileName );
        return;
    }

    // 先頭の無音区間のトリミングを有効にする
    nn::atk::WavOutRecorder::RecordingOptions options;
    options.SetLeadSilenceTrimmingEnabled( true );
    options.SetMaxFrames( frames );

    m_IsRecording = m_Recorder.Start( m_RecordingFileStream, options );

    if ( !m_IsRecording )
    {
        NN_LOG( "[REC] StartRecording : failed to start recording\n" );
        m_RecordingFileStream.Close();
    }
}

//---------------------------------------------------------------------------
//  録音を開始します。
//---------------------------------------------------------------------------
void TestRecorder::StopRecording() NN_NOEXCEPT
{
    if ( !m_IsInitialized || !m_IsRecording )
    {
        return;
    }
    m_Recorder.Stop(true);
    m_RecordingFileStream.Flush();
    m_RecordingFileStream.Close();
    m_IsRecording = false;
}

//---------------------------------------------------------------------------
//  録音中かどうかを取得します。
//---------------------------------------------------------------------------
bool TestRecorder::IsRecording() const NN_NOEXCEPT
{
    return m_Recorder.GetState() == nn::atk::DeviceOutRecorder::State_Recording;
}
