﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nnt/atkUtil/testAtk_Constants.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/atk/atk_SoundHandle.h>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/util/util_FormatString.h>

#include "util/testAtk_FileManager.h"
#include "util/testAtk_TestRecorder.h"

//---------------------------
//  変数
//---------------------------

enum CompareMode
{
    CompareMode_ComparePerByte,
    CompareMode_AllowByteDelay
};

nn::mem::StandardAllocator      g_Allocator;                         // Allocator
TestHeap                        g_TestHeap;                          // テスト用ヒープ

FileManager                     g_FileManager;                       // ファイルの読み書きを行う

const size_t                    HeapSize = 128 * 1024 * 1024;       // ヒープのサイズ
char                            g_HeapMemory[HeapSize];            // ヒープ

nn::mem::StandardAllocator              g_PoolHeapAllocator;             // メモリプール用アロケータ
nn::audio::MemoryPoolType               g_MemoryPool;                    // メモリプール
const size_t                            PoolHeapSize = 32 * 1024 * 1024; // メモリプール用ヒープのサイズ
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_PoolHeapMemory[PoolHeapSize];  // メモリプール用ヒープ

const uint32_t                  RecordSampleCount = 50000;         // 録音サンプル数

#ifndef NNT_ATK_ENABLE_VOICE_COMMAND
const char                      TestNameSuffix[] = "";
#else
const char                      TestNameSuffix[] = "_Vcmd";
#endif

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup  g_AtkSetup;

//  波形の録音, 波形の比較を行って、波形比較テストを行うクラスです。
class AudioOutputTester
{
public:
    //  初期化します
    void Initialize(nn::mem::StandardAllocator& allocator, const char* testName, const char* referenceFileName)
    {
        //  レコーダーの初期化
        size_t recorderThreadStackSize = nn::atk::DeviceOutRecorder::RequiredThreadStackSize;
        m_pRecorderThreadStack         = nnt::atk::util::AllocateUninitializedMemory(allocator, recorderThreadStackSize, nn::os::ThreadStackAlignment);
        // 高負荷時に DeviceOutRecorder で録音した波形をとりこぼさないため、通常より多めに録音バッファを確保する
        size_t recorderBufferSize  = m_TestRecorder.GetRequiredMemorySize() * 8;
        m_pRecorderBuffer          = nnt::atk::util::AllocateUninitializedMemory(allocator, recorderBufferSize);
        m_TestRecorder.Initialize(m_pRecorderBuffer, recorderBufferSize, m_pRecorderThreadStack, recorderThreadStackSize);

        //  ファイル名の設定
        // TODO: 出力ファイル名、フォルダ名の整理
        nn::util::SNPrintf( m_OutputFileName, sizeof(m_OutputFileName), "%s_%s_%s%s.wav", testName, nnt::atk::util::GetTargetName(), nnt::atk::util::BuildType, TestNameSuffix );
        nn::util::SNPrintf( m_OutputFilePath, sizeof(m_OutputFilePath), "%s:/%s", nnt::atk::util::FsCommonSetup::TestResultsMountName, m_OutputFileName );
        nn::util::SNPrintf( m_ReferenceFilePath, sizeof(m_ReferenceFilePath), "%s:/AudioOutputComp/Reference\\%s", nnt::atk::util::FsCommonSetup::TestBinariesMountName, referenceFileName );
        m_TestRecorder.SetOutputFileName( m_OutputFilePath );
    }
    //  終了処理します
    void Finalize(nn::mem::StandardAllocator& allocator)
    {
        m_TestRecorder.Finalize();
        allocator.Free( m_pRecorderBuffer );
        allocator.Free( m_pRecorderThreadStack );
    }

    //  録音を開始します
    void StartRecording(uint32_t recordSampleCount)
    {
        m_TestRecorder.StartRecording( recordSampleCount );
    }
    //  録音が完了するまで待ちます
    void WaitForStopRecording(nn::atk::SoundArchivePlayer& soundArchivePlayer)
    {
        while( IsRecording() ){
            soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
           nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }

        StopRecording();
    }
    //  録音を停止します
    void StopRecording()
    {
        m_TestRecorder.StopRecording();

        //  確実に録音が完了するのを待つために 1 ゲームフレーム待ちます
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
    //  録音中であるかどうかを取得します
    bool IsRecording() const
    {
        return m_TestRecorder.IsRecording();
    }

    //  CompareMode に従って、出力ファイルの比較を行います
    void CompareFile(bool& result, FileManager& fileManager, CompareMode mode, int allowableError)
    {
        NN_ASSERT_GREATER_EQUAL(allowableError, 0);
        size_t outputFileSize;
        char* outputFileData;
        result &= fileManager.ReadFile( m_OutputFilePath, &outputFileData, &outputFileSize );
        if(!result)
        {
            NN_LOG("NG Couldn't read output file: %s\n",  m_OutputFilePath);
            return;
        }

        size_t referenceFileSize;
        char* referenceFileData;
        result &= fileManager.ReadFile( m_ReferenceFilePath, &referenceFileData, &referenceFileSize );
        if (!result)
        {
            NN_LOG("NG Couldn't read reference file: %s\n", m_ReferenceFilePath);
            return;
        }

        result &= ( outputFileData != NULL );
        if (!result)
        {
            NN_LOG("NG Couldn't load output data: %s\n", m_OutputFilePath);
            return;
        }

        result &= ( referenceFileData != NULL );
        if (!result)
        {
            NN_LOG("NG Couldn't load reference data: %s\n", m_ReferenceFilePath);
            return;
        }

        result &= ( outputFileSize == referenceFileSize );
        if (!result)
        {
            NN_LOG("NG Recorded samples size: %u != %u\n", outputFileSize, referenceFileSize);
            return;
        }

        switch(mode)
        {
        case CompareMode_ComparePerByte :
            result &= CompareFilePerByte(outputFileData, outputFileSize, referenceFileData, referenceFileSize, allowableError);
            break;
        case CompareMode_AllowByteDelay :
            result &= CompareFileAllowByteDelay(outputFileData, outputFileSize, referenceFileData, referenceFileSize, allowableError);
            break;
        default :
            NN_UNEXPECTED_DEFAULT;
        }

        fileManager.ReleaseData( outputFileData );
        fileManager.ReleaseData( referenceFileData );

        if (!result)
        {
            NN_LOG("NG Output samples: %s\n", m_ReferenceFilePath);
        }
    }
    void CompareFile(bool& result, FileManager& fileManager, CompareMode mode)
    {
        CompareFile(result, fileManager, mode, 0);
    }

private:
    bool IsErrorAllowable(char output, char reference, int allowableError)
    {
        int error = std::min(abs(output - reference), (CHAR_MAX - CHAR_MIN + 1) - abs(output - reference)); // オーバーフローが起こった場合も考慮してエラー計算を行う
        return (error <= allowableError);
    }

    bool CompareFilePerByte( const char* outputData, size_t outputLength, const char* referenceData, size_t referenceLength, int allowableError)
    {
        bool result = true;

        for ( int i = 0; i < static_cast<int>(outputLength); ++i )
        {
            if(i == static_cast<int>(referenceLength))
            {
                break;
            }

            result &= IsErrorAllowable(outputData[ i ], referenceData[ i ], allowableError);
        }

        return result;
    }
    bool CompareFileAllowByteDelay( const char* outputData, size_t outputLength, const char* referenceData, size_t referenceLength, int allowableError)
    {
        bool result = true;

        const int DataSizeExceptWaveForm = 44;

        // data チャンクのファイルサイズの格納位置まではビット単位で比較する
        for ( int i = 0; i < DataSizeExceptWaveForm; ++i )
        {
            result &= IsErrorAllowable(outputData[i], referenceData[i], 0);
        }

        int referenceDataCount = DataSizeExceptWaveForm;
        bool isDelayOccured = false;
        for ( int i = referenceDataCount; i < static_cast<int>(outputLength); ++i )
        {
            if(referenceDataCount == static_cast<int>(referenceLength))
            {
                break;
            }

            bool isErrorAllowable = IsErrorAllowable(outputData[i], referenceData[i], allowableError);
            if (!isErrorAllowable && outputData[i] == 0)
            {
                isDelayOccured = true;
                continue;
            }

            result &= isErrorAllowable;
            ++referenceDataCount;
        }

        if(isDelayOccured)
        {
            NN_LOG("Sample delay occured.\n");
        }

        return result;
    }

    TestRecorder m_TestRecorder;    // テスト用レコーダー
    void* m_pRecorderThreadStack;
    void* m_pRecorderBuffer;

    static const int MaxPathLength = nn::fs::EntryNameLengthMax + 1;  // + 1 は終端文字
    char m_OutputFileName[MaxPathLength];
    char m_OutputFilePath[MaxPathLength];
    char m_ReferenceFilePath[MaxPathLength];

};

struct AudioOutputCompInitializeParam
{
    const char* testName;
    const char* referenceFileName;
    nn::atk::OutputMode outputMode;
    nnt::atk::util::AtkCommonSetup::InitializeParam atkCommonParam;
};

NN_DEFINE_STATIC_CONSTANT( const int AudioOutputTester::MaxPathLength );
AudioOutputTester g_AudioOutputTester;

//------------------------------------------------------------------------------
//  Main 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    // GoogleTEST 初期化
    ::testing::InitGoogleTest(&argc, argv);

    // GoogleTEST 実行
    int testResult = RUN_ALL_TESTS();

    nnt::Exit(testResult);

    return;
}

//  FsSetup を初期化します
void InitializeFsSetup(nnt::atk::util::FsCommonSetup& fsSetup, nnt::atk::util::FsCommonSetup::InitializeParam param)
{
    //  波形比較テストで利用する機能を有効にします
    param.SetContentsDirectoryMounted(true);
    param.SetTestResultsDirectoryMounted(true);
    param.SetTestBinariesDirectoryMounted(true);

    fsSetup.Initialize(param);
}
//  FsSetup を初期化します
void InitializeFsSetup(nnt::atk::util::FsCommonSetup& fsSetup)
{
    InitializeFsSetup( fsSetup, nnt::atk::util::FsCommonSetup::InitializeParam() );
}
//  AtkSetup を初期化します
void InitializeAtkSetup(nnt::atk::util::AtkCommonSetup& atkSetup, nn::mem::StandardAllocator& allocator, nnt::atk::util::AtkCommonSetup::InitializeParam param)
{
    //  波形比較テストで利用する機能を有効にします
    param.GetSoundSystemParam().enableRecordingFinalOutputs = true;

    // 処理負荷の高くなる先頭部分などで録音した波形が欠けないように、デフォルトよりも大きい recordingAudioFrameCount をとる
    param.GetSoundSystemParam().recordingAudioFrameCount = 512;

    atkSetup.Initialize(param, allocator);
}
//  AtkSetup を初期化します
void InitializeAtkSetup(nnt::atk::util::AtkCommonSetup& atkSetup, nn::mem::StandardAllocator& allocator)
{
    InitializeAtkSetup( atkSetup, allocator, nnt::atk::util::AtkCommonSetup::InitializeParam() );
}

// サウンドアーカイブのデータを読み込みます
void LoadData(nnt::atk::util::AtkCommonSetup& atkSetup)
{
    atkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    atkSetup.LoadData(SEQ_TEST, "SEQ_TEST");
    atkSetup.LoadData(SEQ_TEST_NO_RELEASE, "SEQ_TEST_NO_RELEASE");
    atkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
}

// 波形比較テストに必要なモジュールを初期化します
void InitializeAudioOutputComp(
    AudioOutputCompInitializeParam& param,
    nnt::atk::util::FsCommonSetup& fsSetup,
    nnt::atk::util::AtkCommonSetup& atkSetup,
    nn::mem::StandardAllocator& allocator,
    nn::mem::StandardAllocator& poolHeapAllocator,
    AudioOutputTester& audioOutputTester,
    char* heapMemory,
    size_t heapSize,
    char* poolHeapMemory,
    size_t poolHeapSize,
    nn::audio::MemoryPoolType& memoryPool,
    TestHeap& testHeap,
    FileManager& fileManager)
{
    NN_ASSERT_NOT_NULL(heapMemory);
    NN_ASSERT_GREATER(heapSize, 0u);
    NN_ASSERT_NOT_NULL(poolHeapMemory);
    NN_ASSERT_GREATER(poolHeapSize, 0u);
    NN_ASSERT(nn::util::is_aligned(reinterpret_cast<uintptr_t>(poolHeapMemory), nn::audio::MemoryPoolType::AddressAlignment));
    NN_ASSERT_EQUAL(poolHeapSize % nn::audio::MemoryPoolType::SizeGranularity, 0u);

    NN_ASSERT_NOT_NULL(param.testName);
    NN_ASSERT_NOT_NULL(param.referenceFileName);

    // アロケータ初期化
    allocator.Initialize(heapMemory, heapSize);
    poolHeapAllocator.Initialize(poolHeapMemory, poolHeapSize);

    // ヒープ初期化
    testHeap.SetAllocator(&allocator);
    fileManager.SetHeap(&testHeap);

    // fs 初期化
    InitializeFsSetup(fsSetup);

    // Atk 初期化
    InitializeAtkSetup(atkSetup, allocator, param.atkCommonParam);
    LoadData(atkSetup);
    nn::atk::SoundSystem::SetOutputMode(param.outputMode);
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPool, poolHeapMemory, poolHeapSize);

    //  波形比較テスターの初期化
    audioOutputTester.Initialize(allocator, param.testName, param.referenceFileName);
}

// グローバル変数を用いて波形比較テストに必要なモジュールを初期化します
void InitializeAudioOutputComp(AudioOutputCompInitializeParam& param)
{
    InitializeAudioOutputComp(
        param,
        g_FsSetup,
        g_AtkSetup,
        g_Allocator,
        g_PoolHeapAllocator,
        g_AudioOutputTester,
        g_HeapMemory,
        HeapSize,
        g_PoolHeapMemory,
        PoolHeapSize,
        g_MemoryPool,
        g_TestHeap,
        g_FileManager);
}

void FinalizeAudioOutputComp(
    nnt::atk::util::FsCommonSetup& fsSetup,
    nnt::atk::util::AtkCommonSetup& atkSetup,
    nn::mem::StandardAllocator& allocator,
    nn::mem::StandardAllocator& poolHeapAllocator,
    AudioOutputTester& audioOutputTester,
    nn::audio::MemoryPoolType& memoryPool)
{
    //  波形比較テスターの終了処理
    audioOutputTester.Finalize(allocator);

    // Atk 終了処理
    nn::atk::SoundSystem::DetachMemoryPool(&memoryPool);
    atkSetup.Finalize(allocator);
    fsSetup.Finalize();

    // アロケータ終了処理
    poolHeapAllocator.Finalize();
    allocator.Finalize();
}

// グローバル変数を用いて波形比較テストに必要なモジュールを終了します
void FinalizeAudioOutputComp()
{
    FinalizeAudioOutputComp(g_FsSetup, g_AtkSetup, g_Allocator, g_PoolHeapAllocator, g_AudioOutputTester, g_MemoryPool);
}

// サウンドハンドルに紐づくサウンドを停止します
void StopSound(nn::atk::SoundHandle& soundHandle, nn::atk::SoundArchivePlayer& soundArchivePlayer)
{
    soundHandle.Stop(0);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
}

void UpdateAndWait( nn::atk::SoundArchivePlayer& archivePlayer ) NN_NOEXCEPT
{
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds( 16 );
    archivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess( 4 );
#endif
    nn::os::SleepThread( WaitTime );
}

TEST(AudioOutputComp, WaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char TestName[] = "WaveSound"; // WSD テスト名
    const char ReferenceFileName[] = "WaveSound.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);

    // Atk 初期化
    InitializeAtkSetup(g_AtkSetup, g_Allocator);
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    nn::atk::SoundSystem::SetOutputMode( nn::atk::OutputMode_Stereo );

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize(g_Allocator, TestName, ReferenceFileName);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    {
        bool result = soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
    }

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    soundHandle.Stop(0);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_EQ( true, result );

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize(g_Allocator);

    // Atk 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
}

TEST(AudioOutputComp, Opus2ch24kSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "Opus2ch24kSound", "Opus2ch24kSound2nd", "Opus2ch24kSound3rd" };
    const char ReferenceFileName[] = "Opus2ch24kSound.wav"; // リファレンスのファイル名
    bool isFileComparationSuccessed = false;

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup( g_FsSetup );

    for (const char* testFileName : TestFileName)
    {
        // Opus デコーダの初期化
        size_t decoderWorkBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize();
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, decoderWorkBufferSize);
        nn::atk::InitializeOpusDecoder(decoderWorkBuffer, decoderWorkBufferSize);

        // Atk 初期化
        InitializeAtkSetup(g_AtkSetup, g_Allocator);
        g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
        nn::atk::SoundSystem::SetOutputMode( nn::atk::OutputMode_Stereo );

        //  波形比較テスターの初期化
        g_AudioOutputTester.Initialize(g_Allocator, testFileName, ReferenceFileName);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        {
            bool result = soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART_OPUS ).IsSuccess();
            NN_LOG("[WSD] StartSound(STRM_MARIOKART_OPUS) ... (%d)\n", result);
        }

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

        //  サウンドの停止
        soundHandle.Stop(0);
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

        // 正解波形との比較
        bool result = true;
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_AllowByteDelay);

        //  波形比較テスターの終了処理
        g_AudioOutputTester.Finalize(g_Allocator);

        // Atk 終了処理
        g_AtkSetup.Finalize(g_Allocator);

        // Opus デコーダの終了処理
        nn::atk::FinalizeOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
        // Opus デコーダ終了時に発生するノイズが次回の波形比較に影響しないように待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    // fs 終了処理
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, HardwareOpus2ch24kSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "HardwareOpus2ch24kSound", "HardwareOpus2ch24kSound2nd", "HardwareOpus2ch24kSound3rd" };
    const char ReferenceFileName[] = "Opus2ch24kSound.wav"; // リファレンスのファイル名
    bool isFileComparationSuccessed = false;

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);
    for (const char* testFileName : TestFileName)
    {
        // Opus デコーダの初期化
        size_t decoderWorkBufferSize = nn::atk::GetRequiredHardwareOpusDecoderBufferSize();
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, decoderWorkBufferSize);
        nn::atk::InitializeHardwareOpusDecoder(decoderWorkBuffer, decoderWorkBufferSize);

        // Atk 初期化
        InitializeAtkSetup(g_AtkSetup, g_Allocator);
        g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
        nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Stereo);

        //  波形比較テスターの初期化
        g_AudioOutputTester.Initialize(g_Allocator, testFileName, ReferenceFileName);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // 再生開始
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundMetaInfo;

        // 既存のパラメータをメタ情報として流用
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        soundArchivePlayer.GetSoundArchive().ReadStreamSoundInfo(&info.streamSoundMetaInfo, STRM_MARIOKART_OPUS);

        // ファイル形式を Opus (ハードウェア再生) に変更
        info.streamSoundMetaInfo.streamFileType = nn::atk::SoundArchive::StreamFileType_Opus;
        info.streamSoundMetaInfo.decodeMode = nn::atk::SoundArchive::DecodeMode_Accelerator;

        nn::atk::SoundHandle soundHandle;
        {
            bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART_OPUS, &info).IsSuccess();
            NN_LOG("[WSD] StartSound(STRM_MARIOKART_OPUS) ... (%d)\n", result);
        }

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        soundHandle.Stop(0);
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

        // 正解波形との比較
        bool result = true;
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_AllowByteDelay);

        //  波形比較テスターの終了処理
        g_AudioOutputTester.Finalize(g_Allocator);

        // Atk 終了処理
        g_AtkSetup.Finalize(g_Allocator);

        // Opus デコーダの終了処理
        nn::atk::FinalizeHardwareOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
        // Opus デコーダ終了時に発生するノイズが次回の波形比較に影響しないように待つ
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    // fs 終了処理
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, 6ChWaveSound)
{
    const char TestName[] = "6ChWaveSound"; // 6chWSD テスト名
    const char ReferenceFileName[] = "6ChWaveSound.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);

    // Atk 初期化
    InitializeAtkSetup(g_AtkSetup, g_Allocator);
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    g_AtkSetup.LoadData(SE_WIHAHO, "SE_WIHAHO");
    g_AtkSetup.LoadData(SE_IDLE, "SE_IDLE");
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Surround);

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize(g_Allocator, TestName, ReferenceFileName);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle seYoshiSoundHandle;
    nn::atk::SoundHandle seWihahoSoundHandle;
    nn::atk::SoundHandle seIdleSoundHandle;
    {
        bool result = soundArchivePlayer.StartSound(&seYoshiSoundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
        result = soundArchivePlayer.StartSound(&seWihahoSoundHandle, SE_WIHAHO).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);
        result = soundArchivePlayer.StartSound(&seIdleSoundHandle, SE_IDLE).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_IDLE) ... (%d)\n", result);
    }

    // SE_YOSHI は FrontLeft, FrontRight から出力
    nn::atk::WaveSoundHandle seYoshiWaveSoundHandle(&seYoshiSoundHandle);
    seYoshiWaveSoundHandle.SetMixMode(nn::atk::MixMode_Mixparameter);
    nn::atk::MixParameter seYoshiMixParameter;
    seYoshiMixParameter.fL = 1.0f;
    seYoshiMixParameter.fR = 1.0f / 2;
    seYoshiMixParameter.rL = 0.0f;
    seYoshiMixParameter.rR = 0.0f;
    seYoshiMixParameter.fC = 0.0f;
    seYoshiMixParameter.lfe = 0.0f;
    seYoshiWaveSoundHandle.SetChannelMixParameter(nn::atk::ChannelIndex_FrontLeft, seYoshiMixParameter);

    // SE_YOSHI は RearLeft, RearRight から出力
    nn::atk::WaveSoundHandle seWihahoWaveSoundHandle(&seWihahoSoundHandle);
    seWihahoWaveSoundHandle.SetMixMode(nn::atk::MixMode_Mixparameter);
    nn::atk::MixParameter seWihahoMixParameter;
    seWihahoMixParameter.fL = 0.0f;
    seWihahoMixParameter.fR = 0.0f;
    seWihahoMixParameter.rL = 1.0f;
    seWihahoMixParameter.rR = 1.0f / 2;
    seWihahoMixParameter.fC = 0.0f;
    seWihahoMixParameter.lfe = 0.0f;
    seWihahoWaveSoundHandle.SetChannelMixParameter(nn::atk::ChannelIndex_FrontLeft, seWihahoMixParameter);

    // SE_IDLE は FrontCenter, LFE から出力
    nn::atk::WaveSoundHandle seIdleWaveSoundHandle(&seIdleSoundHandle);
    seIdleWaveSoundHandle.SetMixMode(nn::atk::MixMode_Mixparameter);
    nn::atk::MixParameter seIdleMixParameter;
    seIdleMixParameter.fL = 0.0f;
    seIdleMixParameter.fR = 0.0f;
    seIdleMixParameter.rL = 0.0f;
    seIdleMixParameter.rR = 0.0f;
    seIdleMixParameter.fC = 1.0f;
    seIdleMixParameter.lfe = 1.0f / 2;
    seIdleWaveSoundHandle.SetChannelMixParameter(nn::atk::ChannelIndex_FrontLeft, seIdleMixParameter);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    seYoshiSoundHandle.Stop(0);
    seWihahoSoundHandle.Stop(0);
    seIdleSoundHandle.Stop(0);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_EQ( true, result );

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize(g_Allocator);

    // Atk 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
} //NOLINT(impl/function_size)

// 外部ファイル再生版
TEST(AudioOutputComp, PlayPrefetchSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char TestName[] = "PlayPrefetchSound"; // WSD テスト名
    const char ReferenceFileName[] = "PlayPrefetchSound.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);

    // Atk 初期化
    InitializeAtkSetup(g_AtkSetup, g_Allocator);
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Stereo);

    // プリフェッチデータのロード
    char prefetchFilePath[nn::fs::EntryNameLengthMax + 1];  //  + 1 は終端文字
    char externalFilePath[nn::fs::EntryNameLengthMax + 1];  //  + 1 は終端文字
    nn::audio::MemoryPoolType prefetchMemoryPool;
    void* pPrefetchDataBuffer;
    void* pPrefetchDataBegin;
    int64_t prefetchFileSize;
    {
        nn::fs::FileHandle handle;
        nn::Result result;

        //  ファイルを開きます
        const char* prefetchFileName = "kart_title.32";
        nn::util::SNPrintf( prefetchFilePath, sizeof(prefetchFilePath), "%s:/%s.bfstp", nnt::atk::util::FsCommonSetup::RomMountName, prefetchFileName );
        nn::util::SNPrintf( externalFilePath, sizeof(externalFilePath), "%s:/%s.bfstm", nnt::atk::util::FsCommonSetup::RomMountName, prefetchFileName );
        result = nn::fs::OpenFile(&handle, prefetchFilePath, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::fs::OpenFile is failed.");

        result = nn::fs::GetFileSize(&prefetchFileSize, handle);
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::fs::GetFileSize is failed.");

        //  バッファを確保します
        const size_t bufferSize = nn::util::align_up( static_cast<size_t>(prefetchFileSize), nn::audio::MemoryPoolType::SizeGranularity );
        pPrefetchDataBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, bufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        nn::atk::SoundSystem::AttachMemoryPool( &prefetchMemoryPool, pPrefetchDataBuffer, bufferSize );

        //  SIGLO-36451 の回帰テストのためにメモリプールの末尾にプリフェッチデータを読み込みます
        pPrefetchDataBegin = nn::util::BytePtr( pPrefetchDataBuffer, static_cast<ptrdiff_t>( bufferSize - static_cast<size_t>(prefetchFileSize) ) ).AlignDown( nn::audio::BufferAlignSize ).Get();

        result = nn::fs::ReadFile(handle, 0 , pPrefetchDataBegin, static_cast<size_t>(prefetchFileSize));
        NN_ABORT_UNLESS(result.IsSuccess(), "nn::fs::ReadFile is failed.");
        nn::fs::CloseFile(handle);
    }

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize(g_Allocator, TestName, ReferenceFileName);

    // 録音開始
    // プリフェッチと通常のストリーム再生の切り替わりがスムーズに行われたことを確認するため
    // プリフェッチファイルサイズの 2 倍のサンプル数を録音します
    g_AudioOutputTester.StartRecording(2 * static_cast<uint32_t>( prefetchFileSize ));

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    {
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
        info.streamSoundInfo.prefetchData = pPrefetchDataBegin;
        info.streamSoundInfo.externalPath = externalFilePath;

        bool result = soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART, &info ).IsSuccess();
        NN_LOG("[STRM] StartSound(STRM_MARIOKART) with prefetch data ... (%d)\n", result);
    }

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_EQ( true, result );

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize(g_Allocator);

    // プリフェッチデータの終了処理
    nn::atk::SoundSystem::DetachMemoryPool( &prefetchMemoryPool );
    g_Allocator.Free( pPrefetchDataBuffer );

    // Atk 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
}

// SoundHeap 読み込み版
TEST( AudioOutputComp, PlayPrefetchSoundOnSoundHeap )
{
    nnt::atk::util::OnPreAtkTest();
    const char TestName[] = "PlayPrefetchSoundOnSoundHeap"; // テスト名
    const char ReferenceFileName[] = "PlayPrefetchSoundOnSoundHeap.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize( g_HeapMemory, HeapSize );

    // ヒープ初期化
    g_TestHeap.SetAllocator( &g_Allocator );
    g_FileManager.SetHeap( &g_TestHeap );

    // fs 初期化
    InitializeFsSetup( g_FsSetup );

    // Atk 初期化
    InitializeAtkSetup( g_AtkSetup, g_Allocator );
    g_AtkSetup.LoadData( STRM_MARIOKART, "STRM_MARIOKART" );
    nn::atk::SoundSystem::SetOutputMode( nn::atk::OutputMode_Stereo );

    // プリフェッチファイルサイズを取得
    int64_t prefetchFileSize;
    {
        char prefetchFilePath[nn::fs::EntryNameLengthMax + 1];  //  + 1 は終端文字
        char externalFilePath[nn::fs::EntryNameLengthMax + 1];  //  + 1 は終端文字

        nn::fs::FileHandle handle;
        nn::Result result;

        //  ファイルを開きます
        const char* prefetchFileName = "kart_title.32";
        nn::util::SNPrintf( prefetchFilePath, sizeof( prefetchFilePath ), "%s:/%s.bfstp", nnt::atk::util::FsCommonSetup::RomMountName, prefetchFileName );
        nn::util::SNPrintf( externalFilePath, sizeof( externalFilePath ), "%s:/%s.bfstm", nnt::atk::util::FsCommonSetup::RomMountName, prefetchFileName );
        result = nn::fs::OpenFile( &handle, prefetchFilePath, nn::fs::OpenMode_Read );
        NN_ABORT_UNLESS( result.IsSuccess(), "nn::fs::OpenFile is failed." );

        result = nn::fs::GetFileSize( &prefetchFileSize, handle );
        NN_ABORT_UNLESS( result.IsSuccess(), "nn::fs::GetFileSize is failed." );

        nn::fs::CloseFile( handle );
    }

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize( g_Allocator, TestName, ReferenceFileName );

    // 録音開始
    // プリフェッチと通常のストリーム再生の切り替わりがスムーズに行われたことを確認するため
    // プリフェッチファイルサイズの 2 倍のサンプル数を録音します
    g_AudioOutputTester.StartRecording( 2 * static_cast<uint32_t>(prefetchFileSize) );

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    {
        bool result = soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART ).IsSuccess();
        NN_LOG( "[STRM] StartSound(STRM_MARIOKART) with prefetch data ... (%d)\n", result );
    }

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile( result, g_FileManager, CompareMode_ComparePerByte );
    EXPECT_EQ( true, result );

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize( g_Allocator );

    // Atk 終了処理
    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
}

TEST(AudioOutputComp, WaveSound32kHz)
{
    nnt::atk::util::OnPreAtkTest();
    const char TestName[] = "WaveSound32kHz"; // WSD テスト名
    const char ReferenceFileName[] = "WaveSound32kHz.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);

    // Atk 初期化
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.GetSoundSystemParam().rendererSampleRate = 32000;
    InitializeAtkSetup(g_AtkSetup, g_Allocator, param);
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Stereo);

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize(g_Allocator, TestName, ReferenceFileName);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    {
        bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
    }

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    soundHandle.Stop(0);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_EQ(true, result);

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize(g_Allocator);

    // Atk 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
}

TEST(AudioOutputComp, BiquadFilterWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char TestName[] = "BiquadFilterWaveSound"; // WSD テスト名
    const char ReferenceFileName[] = "BiquadFilterWaveSound.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);

    // Atk 初期化
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    InitializeAtkSetup(g_AtkSetup, g_Allocator, param);
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Stereo);

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize(g_Allocator, TestName, ReferenceFileName);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    {
        bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
    }
    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter512, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    soundHandle.Stop(0);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_EQ(true, result);

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize(g_Allocator);

    // Atk 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
}

TEST(AudioOutputComp, BiquadFilterLpfWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterLpfWaveSound";
    param.referenceFileName = "BiquadFilterLpfWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_LowPassFilter, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilterHpfWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterHpfWaveSound";
    param.referenceFileName = "BiquadFilterHpfWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_HighPassFilter, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilter1024WaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilter1024WaveSound";
    param.referenceFileName = "BiquadFilter1024WaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter1024, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilter2048WaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilter2048WaveSound";
    param.referenceFileName = "BiquadFilter2048WaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter2048, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilterLpfNw4f48kCompatWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterLpfNw4f48kCompatWaveSound";
    param.referenceFileName = "BiquadFilterLpfNw4f48kCompatWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_LowPassFilterNw4fCompatible48k, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilterHpfNw4f48kCompatWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterHpfNw4f48kCompatWaveSound";
    param.referenceFileName = "BiquadFilterHpfNw4f48kCompatWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_HighPassFilterNw4fCompatible48k, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilterBpf512Nw4f48kCompatWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterBpf512Nw4f48kCompatWaveSound";
    param.referenceFileName = "BiquadFilterBpf512Nw4f48kCompatWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter512Nw4fCompatible48k, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilterBpf1024Nw4f48kCompatWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterBpf1024Nw4f48kCompatWaveSound";
    param.referenceFileName = "BiquadFilterBpf1024Nw4f48kCompatWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter1024Nw4fCompatible48k, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, BiquadFilterBpf2048Nw4f48kCompatWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "BiquadFilterBpf2048Nw4f48kCompatWaveSound";
    param.referenceFileName = "BiquadFilterBpf2048Nw4f48kCompatWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);

    soundHandle.SetBiquadFilter(nn::atk::BiquadFilterType_BandPassFilter2048Nw4fCompatible48k, 0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, LowPassFilterWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char TestName[] = "LowPassFilterWaveSound"; // WSD テスト名
    const char ReferenceFileName[] = "LowPassFilterWaveSound.wav"; // リファレンスのファイル名

    // アロケータ初期化
    g_Allocator.Initialize(g_HeapMemory, HeapSize);

    // ヒープ初期化
    g_TestHeap.SetAllocator(&g_Allocator);
    g_FileManager.SetHeap(&g_TestHeap);

    // fs 初期化
    InitializeFsSetup(g_FsSetup);

    // Atk 初期化
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.GetSoundSystemParam().enableCompatibleLowPassFilter = true;
    InitializeAtkSetup(g_AtkSetup, g_Allocator, param);
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Stereo);

    //  波形比較テスターの初期化
    g_AudioOutputTester.Initialize(g_Allocator, TestName, ReferenceFileName);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    {
        bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
    }
    soundHandle.SetLowPassFilterFrequency(-0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    soundHandle.Stop(0);
    soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

    // 正解波形との比較
    bool result = true;
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_EQ(true, result);

    //  波形比較テスターの終了処理
    g_AudioOutputTester.Finalize(g_Allocator);

    // Atk 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    // アロケータ終了処理
    g_Allocator.Finalize();
}

TEST(AudioOutputComp, PitchWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "Pitch";
    param.referenceFileName = "PitchWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetPitch(0.8f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, PanWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "Pan";
    param.referenceFileName = "PanWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetPan(-0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, 6chSurroundPanWaveSound)
{
    AudioOutputCompInitializeParam param;
    param.testName = "6chSurroundPan";
    param.referenceFileName = "6chSurroundPanWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetSurroundPan(1.8f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, VolumeWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "Volume";
    param.referenceFileName = "VolumeWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetVolume(0.5f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, 6chMixVolumeWaveSound)
{
    AudioOutputCompInitializeParam param;
    param.testName = "6chMixVolume";
    param.referenceFileName = "6chMixVolumeWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    soundHandle.SetMixMode(nn::atk::MixMode_MixVolume);
    nn::atk::MixVolume mixVolume;
    mixVolume.frontLeft          = 0.1f;
    mixVolume.frontRight         = 1.9f;
    mixVolume.rearLeft           = 0.4f;
    mixVolume.rearRight          = 1.6f;
    mixVolume.frontCenter        = 0.7f;
    mixVolume.lowFrequencyEffect = 1.3f;
    soundHandle.SetMixVolume(mixVolume);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
// PC 版ではオーディオフレーム処理が安定しないため、SEQ は実機版のみテスト
TEST(AudioOutputComp, SeqSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "SeqSound", "SeqSound2nd", "SeqSound3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {
        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "SeqSound.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SEQ_TEST).IsSuccess();
        NN_LOG("[SEQ] StartSound(SEQ_TEST) ... (%d)\n", result);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // 正解波形との比較
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, SeqBgmSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "SeqBgmSound", "SeqBgmSound2nd", "SeqBgmSound3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {
        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "SeqBgmSound.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SEQ_MARIOKART).IsSuccess();
        NN_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // 正解波形との比較
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, StartOffsetSeqSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "SeqBgmSound", "SeqBgmSound2nd", "SeqBgmSound3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {

        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "StartOffsetSeqSound.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
        info.startOffset = 2000;
        info.startOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_Tick;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SEQ_MARIOKART, &info).IsSuccess();
        NN_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // 正解波形との比較
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, SeqSoundWithNoRelease)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "SeqSoundWithNoRelease", "SeqSoundWithNoRelease2nd", "SeqSoundWithNoRelease3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {
        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "SeqSoundWithNoRelease.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SEQ_TEST_NO_RELEASE).IsSuccess();
        NN_LOG("[SEQ] StartSound(SEQ_TEST_NO_RELEASE) ... (%d)\n", result);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // 正解波形との比較
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}
#endif

TEST(AudioOutputComp, StartOffsetWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "StartOffsetWaveSound";
    param.referenceFileName = "StartOffsetWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
    info.startOffset = 100;
    info.startOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_MilliSeconds;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI, &info).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, StartOffsetStrmSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "StartOffsetStrmSound";
    param.referenceFileName = "StartOffsetStrmSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag = nn::atk::SoundStartable::StartInfo::EnableFlagBit_StartOffset;
    info.startOffset = 1000;
    info.startOffsetType = nn::atk::SoundStartable::StartInfo::StartOffsetType_Sample;
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART, &info).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

nn::atk::StreamRegionCallbackResult StreamRegionCallback(nn::atk::StreamRegionCallbackParam* param, void* /*arg*/)
{
    param->regionNo = 1;
    return nn::atk::StreamRegionCallbackResult_Continue;
}

TEST(AudioOutputComp, StrmJumpSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "StrmJumpSound";
    param.referenceFileName = "StrmJumpSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    info.streamSoundInfo.regionCallback = StreamRegionCallback;
    info.streamSoundInfo.regionCallbackArg = nullptr;
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_REGION_JUMP, &info).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_REGION_JUMP) ... (%d)\n", result);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

struct TestRegionCallbackParam
{
    uint32_t* pRegionList;
    uint32_t  regionListSize;
    uint32_t* pRegionListIndex;
};

nn::atk::StreamRegionCallbackResult StreamRegionCallbackSimple(nn::atk::StreamRegionCallbackParam* param, void* arg )
{
    NN_ASSERT_NOT_NULL(param);
    NN_ASSERT_NOT_NULL(arg);

    TestRegionCallbackParam* pParam = static_cast<TestRegionCallbackParam*>(arg);
    NN_ASSERT_NOT_NULL( pParam );
    NN_ASSERT_RANGE(*pParam->pRegionListIndex, 0u, pParam->regionListSize);

    param->regionNo = pParam->pRegionList[*pParam->pRegionListIndex];
    (*pParam->pRegionListIndex)++;
    return nn::atk::StreamRegionCallbackResult_Continue;
}

TEST( AudioOutputComp, StrmJumpSoundWithPrefetch )
{
    nnt::atk::util::OnPreAtkTest();
    uint32_t regionList[] = {
        0,
        1,
        1
    };

    uint32_t regionListIndex = 0;

    TestRegionCallbackParam regionCallbackParam;
    regionCallbackParam.pRegionList = regionList;
    regionCallbackParam.regionListSize = sizeof(regionList) / sizeof(regionList[0]);
    regionCallbackParam.pRegionListIndex = &regionListIndex;

    AudioOutputCompInitializeParam param;
    param.testName = "StrmJumpSoundWithPrefetch";
    param.referenceFileName = "StrmJumpSoundWithPrefetch.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    InitializeAudioOutputComp( param );

    g_AtkSetup.LoadData( STRM_REGION_JUMP, "STRM_REGION_JUMP" );

    // 録音開始
    g_AudioOutputTester.StartRecording( RecordSampleCount * 2 );

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    info.streamSoundInfo.regionCallback = StreamRegionCallbackSimple;
    info.streamSoundInfo.regionCallbackArg = &regionCallbackParam;
    bool result = soundArchivePlayer.StartSound( &soundHandle, STRM_REGION_JUMP, &info ).IsSuccess();
    NN_LOG( "[STRM] StartSound(STRM_REGION_JUMP) ... (%d)\n", result );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // 正解波形との比較
    g_AudioOutputTester.CompareFile( result, g_FileManager, CompareMode_ComparePerByte );
    EXPECT_TRUE( result );

    FinalizeAudioOutputComp();
}

TEST( AudioOutputComp, StrmStartJumpSoundWithPrefetch )
{
    nnt::atk::util::OnPreAtkTest();
    uint32_t regionList[] = {
        1,
        0,
        1
    };

    uint32_t regionListIndex = 0;

    TestRegionCallbackParam regionCallbackParam;
    regionCallbackParam.pRegionList = regionList;
    regionCallbackParam.regionListSize = sizeof( regionList ) / sizeof( regionList[0] );
    regionCallbackParam.pRegionListIndex = &regionListIndex;

    AudioOutputCompInitializeParam param;
    param.testName = "StrmStartJumpSoundWithPrefetch";
    param.referenceFileName = "StrmStartJumpSoundWithPrefetch.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    InitializeAudioOutputComp( param );

    g_AtkSetup.LoadData( STRM_REGION_JUMP, "STRM_REGION_JUMP" );

    // 録音開始
    // 1 のリージョンが終わって 0 のリージョンになるのを確認できる長さ、計算によって求めた値ではありません。
    g_AudioOutputTester.StartRecording( RecordSampleCount * 6 );

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    info.streamSoundInfo.regionCallback = StreamRegionCallbackSimple;
    info.streamSoundInfo.regionCallbackArg = &regionCallbackParam;
    bool result = soundArchivePlayer.StartSound( &soundHandle, STRM_REGION_JUMP, &info ).IsSuccess();
    NN_LOG( "[STRM] StartSound(STRM_REGION_JUMP) ... (%d)\n", result );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // 正解波形との比較
    g_AudioOutputTester.CompareFile( result, g_FileManager, CompareMode_ComparePerByte );
    EXPECT_TRUE( result );

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, src44100WaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "src44100WaveSound";
    param.referenceFileName = "src44100WaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Stereo;
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, SE_WIHAHO).IsSuccess();
    NN_LOG("[WSD] StartSound(SE_WIHAHO) ... (%d)\n", result);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, EffectDelayWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "EffectDelayWaveSound", "EffectDelayWaveSound2nd", "EffectDelayWaveSound3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {
        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "EffectDelayWaveSound.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // エフェクトの追加
        nn::atk::EffectDelay delay;
        delay.SetEnabled(true);
        size_t delayBufferSize = delay.GetRequiredMemSize();
        void* delayBuffer = nnt::atk::util::AllocateUninitializedMemory(g_PoolHeapAllocator, delayBufferSize, nn::audio::BufferAlignSize);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &delay, delayBuffer, delayBufferSize));

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
        soundHandle.SetMainSend(-1.0f);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // エフェクトの破棄
        delay.SetEnabled(false);
        while (!delay.IsRemovable())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &delay);
        g_PoolHeapAllocator.Free(delayBuffer);

        // 正解波形との比較
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, EffectReverbWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "EffectReverbWaveSound", "EffectReverbWaveSound2nd", "EffectReverbWaveSound3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {
        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "EffectReverbWaveSound.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // エフェクトの追加
        nn::atk::EffectReverb reverb;
        reverb.SetEnabled(true);
        size_t reverbBufferSize = reverb.GetRequiredMemSize();
        void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_PoolHeapAllocator, reverbBufferSize, nn::audio::BufferAlignSize);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
        soundHandle.SetMainSend(-1.0f);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // エフェクトの破棄
        reverb.SetEnabled(false);
        while (!reverb.IsRemovable())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &reverb);
        g_PoolHeapAllocator.Free(reverbBuffer);

        // 正解波形との比較
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST(AudioOutputComp, EffectI3dl2ReverbWaveSound)
{
    nnt::atk::util::OnPreAtkTest();
    const char *TestFileName[3] = { "EffectI3dl2ReverbWaveSound", "EffectI3dl2ReverbWaveSound2nd", "EffectI3dl2ReverbWaveSound3rd" };
    bool isFileComparationSuccessed = false;

    for (const char* testFileName : TestFileName)
    {
        AudioOutputCompInitializeParam param;
        param.testName = testFileName;
        param.referenceFileName = "EffectI3dl2ReverbWaveSound.wav"; // リファレンスのファイル名
        param.outputMode = nn::atk::OutputMode_Stereo;
        InitializeAudioOutputComp(param);

        // 録音開始
        g_AudioOutputTester.StartRecording(RecordSampleCount);

        // エフェクトの追加
        nn::atk::EffectI3dl2Reverb reverb;
        reverb.SetEnabled(true);
        size_t reverbBufferSize = reverb.GetRequiredMemSize();
        void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_PoolHeapAllocator, reverbBufferSize, nn::audio::BufferAlignSize);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));

        // 再生開始
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundHandle soundHandle;
        bool result = soundArchivePlayer.StartSound(&soundHandle, SE_YOSHI).IsSuccess();
        NN_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);

        soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
        soundHandle.SetMainSend(-1.0f);

        // 既定のサンプル録音し、停止するまで待ちます
        g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

        //  サウンドの停止
        StopSound(soundHandle, soundArchivePlayer);

        // エフェクトの破棄
        reverb.SetEnabled(false);
        while (!reverb.IsRemovable())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &reverb);
        g_PoolHeapAllocator.Free(reverbBuffer);

        // 正解波形との比較 (Win32 版は 1bit の計算誤差が起きるため 1bit の誤差を許容する)
        g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte, 1);

        FinalizeAudioOutputComp();

        if (result)
        {
            isFileComparationSuccessed = true;
            break;
        }
    }

    EXPECT_TRUE(isFileComparationSuccessed);
}

TEST( AudioOutputComp, UnusedEftChMutingWaveSound2ch )
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "UnusedEftChMutingWaveSound2ch";
    param.referenceFileName = "UnusedEftChMutingWaveSound2ch.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    param.atkCommonParam.GetSoundSystemParam().enableUnusedEffectChannelMuting = true;
    InitializeAudioOutputComp( param );

    nn::atk::MixVolume mixVolume;
    mixVolume.frontLeft = 1.0f;
    mixVolume.frontRight = 1.0f;
    mixVolume.rearLeft = 1.0f;
    mixVolume.rearRight = 1.0f;
    mixVolume.frontCenter = 1.0f;
    mixVolume.lowFrequencyEffect = 1.0f;

    // 録音開始
    g_AudioOutputTester.StartRecording( RecordSampleCount );

    // 2ch エフェクトの追加
    nn::atk::EffectDelay delay;
    delay.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);
    delay.SetEnabled( true );
    size_t delayBufferSize = delay.GetRequiredMemSize();
    void* delayBuffer = nnt::atk::util::AllocateUninitializedMemory( g_PoolHeapAllocator, delayBufferSize, nn::audio::BufferAlignSize );
    EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, &delay, delayBuffer, delayBufferSize ) );

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess();
    NN_LOG( "[WSD] StartSound(SE_YOSHI) ... (%d)\n", result );

    nn::atk::WaveSoundHandle waveSoundHandle( &soundHandle );
    waveSoundHandle.SetMixMode( nn::atk::MixMode_MixVolume );
    waveSoundHandle.SetMixVolume( 0, mixVolume );
    waveSoundHandle.SetMixVolume( 1, mixVolume );

    // AuxBusA 以外から音がでないように設定
    soundHandle.SetEffectSend( nn::atk::AuxBus_A, 1.0f );
    soundHandle.SetMainSend( -1.0f );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // エフェクトの破棄
    delay.SetEnabled( false );
    while ( !delay.IsRemovable() )
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
    }
    nn::atk::SoundSystem::RemoveEffect( nn::atk::AuxBus_A, &delay );
    g_PoolHeapAllocator.Free( delayBuffer );

    // 正解波形との比較
    g_AudioOutputTester.CompareFile( result, g_FileManager, CompareMode_ComparePerByte );
    EXPECT_TRUE( result );

    FinalizeAudioOutputComp();
}

TEST( AudioOutputComp, UnusedEftChMutingWaveSound4ch )
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "UnusedEftChMutingWaveSound4ch";
    param.referenceFileName = "UnusedEftChMutingWaveSound4ch.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    param.atkCommonParam.GetSoundSystemParam().enableUnusedEffectChannelMuting = true;
    InitializeAudioOutputComp( param );

    nn::atk::MixVolume mixVolume;
    mixVolume.frontLeft = 1.0f;
    mixVolume.frontRight = 1.0f;
    mixVolume.rearLeft = 1.0f;
    mixVolume.rearRight = 1.0f;
    mixVolume.frontCenter = 1.0f;
    mixVolume.lowFrequencyEffect = 1.0f;

    // 録音開始
    g_AudioOutputTester.StartRecording( RecordSampleCount );

    // 4ch エフェクトの追加
    nn::atk::EffectReverb reverb;
    reverb.SetChannelMode( nn::atk::EffectBase::ChannelMode_4Ch );
    reverb.SetEnabled( true );
    size_t reverbBufferSize = reverb.GetRequiredMemSize();
    void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory( g_PoolHeapAllocator, reverbBufferSize, nn::audio::BufferAlignSize );
    EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize ) );

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess();
    NN_LOG( "[WSD] StartSound(SE_YOSHI) ... (%d)\n", result );

    nn::atk::WaveSoundHandle waveSoundHandle( &soundHandle );
    waveSoundHandle.SetMixMode( nn::atk::MixMode_MixVolume );
    waveSoundHandle.SetMixVolume( 0, mixVolume );
    waveSoundHandle.SetMixVolume( 1, mixVolume );

    // AuxBusA 以外から音がでないように設定
    soundHandle.SetEffectSend( nn::atk::AuxBus_A, 1.0f );
    soundHandle.SetMainSend( -1.0f );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // エフェクトの破棄
    reverb.SetEnabled( false );
    while ( !reverb.IsRemovable() )
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
    }
    nn::atk::SoundSystem::RemoveEffect( nn::atk::AuxBus_A, &reverb );
    g_PoolHeapAllocator.Free( reverbBuffer );

    // 正解波形との比較
    g_AudioOutputTester.CompareFile( result, g_FileManager, CompareMode_ComparePerByte );
    EXPECT_TRUE( result );

    FinalizeAudioOutputComp();
}

TEST( AudioOutputComp, UnusedEftChMutingWithFadeWaveSound )
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "UnusedEftChMutingWithFadeWaveSound";
    param.referenceFileName = "UnusedEftChMutingWithFadeWaveSound.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    param.atkCommonParam.GetSoundSystemParam().enableUnusedEffectChannelMuting = true;
    InitializeAudioOutputComp( param );

    nn::atk::MixVolume mixVolume;
    mixVolume.frontLeft = 1.0f;
    mixVolume.frontRight = 1.0f;
    mixVolume.rearLeft = 1.0f;
    mixVolume.rearRight = 1.0f;
    mixVolume.frontCenter = 1.0f;
    mixVolume.lowFrequencyEffect = 1.0f;

    // エフェクトの追加
    nn::atk::EffectReverb reverb;
    reverb.SetEnabled( true );
    size_t reverbBufferSize = reverb.GetRequiredMemSize();
    void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory( g_PoolHeapAllocator, reverbBufferSize, nn::audio::BufferAlignSize );
    EXPECT_TRUE( nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize ) );

    // エフェクトのフェードアウト、フェードインを事前実行しても、設定が壊れないことを確認
    nn::atk::SoundSystem::SetAuxBusVolume( nn::atk::AuxBus_A, 0.0f, nn::TimeSpan::FromMilliSeconds( 10 ) );
    UpdateAndWait( g_AtkSetup.GetSoundArchivePlayer() );
    nn::atk::SoundSystem::SetAuxBusVolume( nn::atk::AuxBus_A, 1.0f, nn::TimeSpan::FromMilliSeconds( 10 ) );
    UpdateAndWait( g_AtkSetup.GetSoundArchivePlayer() );

    // 録音開始
    g_AudioOutputTester.StartRecording( RecordSampleCount );

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess();
    NN_LOG( "[WSD] StartSound(SE_YOSHI) ... (%d)\n", result );

    nn::atk::WaveSoundHandle waveSoundHandle( &soundHandle );
    waveSoundHandle.SetMixMode( nn::atk::MixMode_MixVolume );
    waveSoundHandle.SetMixVolume( 0, mixVolume );
    waveSoundHandle.SetMixVolume( 1, mixVolume );

    soundHandle.SetEffectSend( nn::atk::AuxBus_A, 1.0f );
    soundHandle.SetMainSend( -1.0f );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording( soundArchivePlayer );

    //  サウンドの停止
    StopSound( soundHandle, soundArchivePlayer );

    // エフェクトの破棄
    reverb.SetEnabled( false );
    while ( !reverb.IsRemovable() )
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
    }
    nn::atk::SoundSystem::RemoveEffect( nn::atk::AuxBus_A, &reverb );
    g_PoolHeapAllocator.Free( reverbBuffer );

    // 正解波形との比較
    g_AudioOutputTester.CompareFile( result, g_FileManager, CompareMode_ComparePerByte );
    EXPECT_TRUE( result );

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, CompatibleFxSend)
{
    nnt::atk::util::OnPreAtkTest();
    AudioOutputCompInitializeParam param;
    param.testName = "CompatibleFxSend";
    param.referenceFileName = "CompatibleFxSend.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    // NW4F 互換のエフェクトセンドを有効にする
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableCompatibleBusVolume = true;
    param.atkCommonParam.SetSoundSystemParam(soundSystemParam);
    InitializeAudioOutputComp(param);

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);

    // EffectSend を上げる(enableCompatibleBusVolume が無効だと音量が上がりテストに失敗する)
    soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, SubMixBusVolume)
{
    nnt::atk::util::OnPreAtkTest();
    const int BusCount = 2;
    const int ChannelCount = 6;
    AudioOutputCompInitializeParam param;
    param.testName = "SubMixBusVolume";
    param.referenceFileName = "SubMixBusVolume.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    // CustomSubMix を有効にする
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableCustomSubMix = true;
    soundSystemParam.subMixTotalChannelCount = BusCount * ChannelCount;
    soundSystemParam.subMixCount = 1;
    param.atkCommonParam.SetSoundSystemParam(soundSystemParam);
    InitializeAudioOutputComp(param);

    // SubMix の用意
    nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
    nn::atk::SubMix subMix;
    const size_t bufferSize = nn::atk::SubMix::GetRequiredMemorySize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount() );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize );
    subMix.Initialize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount(), buffer, bufferSize );
    subMix.SetDestination( &finalMix );

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer::StartInfo info;
    info.enableFlag |= nn::atk::SoundArchivePlayer::StartInfo::EnableFlagBit_OutputReceiver;
    info.pOutputReceiver = &subMix;
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART, &info).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);

    // Voice からは Main, Aux_A に 1.0f ずつ送り、
    // subMix の Main バスはボリュームを 0.5f, Aux_A バスはボリュームを 1.25f にします
    const int MainBus = 0;
    const int AuxBusA = 1;
    soundHandle.SetMainSend( 1.0f );
    soundHandle.SetEffectSend(nn::atk::AuxBus_A, 1.0f);
    subMix.SetBusVolume( MainBus, 0.5f, 0 );
    subMix.SetBusVolume( AuxBusA, 1.25f, 0 );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    nnt::atk::util::WaitForProcessCommand();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  TODO: SIGLO-65917 の暫定対処として subMix.Finalize() より先に
    //        soundArchivePlayer.Finalize() を行います
    //        この暫定対処は SIGLO-65970 の対応時に、適切に修正される予定です
    soundArchivePlayer.Finalize();
#endif

    subMix.Finalize();

    g_Allocator.Free( buffer );
    FinalizeAudioOutputComp();
}

TEST(AudioOutputComp, SubMixChannelVolume)
{
    nnt::atk::util::OnPreAtkTest();
    const int BusCount = 2;
    const int ChannelCount = 6;
    AudioOutputCompInitializeParam param;
    param.testName = "SubMixChannelVolume";
    param.referenceFileName = "SubMixChannelVolume.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    // CustomSubMix を有効にする
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableCustomSubMix = true;
    soundSystemParam.subMixTotalChannelCount = BusCount * ChannelCount;
    soundSystemParam.subMixCount = 1;
    param.atkCommonParam.SetSoundSystemParam(soundSystemParam);
    InitializeAudioOutputComp(param);

    // SubMix の用意
    nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
    nn::atk::SubMix subMix;
    const size_t bufferSize = nn::atk::SubMix::GetRequiredMemorySize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount() );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize );
    subMix.Initialize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount(), buffer, bufferSize );
    subMix.SetDestination( &finalMix );

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer::StartInfo info;
    info.enableFlag |= nn::atk::SoundArchivePlayer::StartInfo::EnableFlagBit_OutputReceiver;
    info.pOutputReceiver = &subMix;
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART, &info).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);

    // subMix の FL からは 0.5f, FR からは 1.25f でセンドするようにします
    subMix.SetChannelVolume( nn::atk::ChannelIndex_FrontLeft, 0.5f, 0 );
    subMix.SetChannelVolume( nn::atk::ChannelIndex_FrontRight, 1.25f, 0 );

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    nnt::atk::util::WaitForProcessCommand();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  TODO: SIGLO-65917 の暫定対処として subMix.Finalize() より先に
    //        soundArchivePlayer.Finalize() を行います
    //        この暫定対処は SIGLO-65970 の対応時に、適切に修正される予定です
    soundArchivePlayer.Finalize();
#endif

    subMix.Finalize();

    g_Allocator.Free( buffer );
    FinalizeAudioOutputComp();
}

// TODO: Send 系 API の変更にあった、新しいテストを追加する
/*
TEST(AudioOutputComp, SubMixSendVolume)
{
    nnt::atk::util::OnPreAtkTest();
    const int BusCount = 2;
    const int ChannelCount = 6;
    AudioOutputCompInitializeParam param;
    param.testName = "SubMixSendVolume";
    param.referenceFileName = "SubMixSendVolume.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    // CustomSubMix を有効にする
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableCustomSubMix = true;
    soundSystemParam.mixBufferCount = BusCount * ChannelCount;
    soundSystemParam.subMixCount = 1;
    param.atkCommonParam.SetSoundSystemParam(soundSystemParam);
    InitializeAudioOutputComp(param);

    // SubMix の用意
    nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
    nn::atk::SubMix subMix;
    const size_t bufferSize = nn::atk::SubMix::GetRequiredMemorySize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount() );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize );
    subMix.Initialize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount(), buffer, bufferSize );
    subMix.SetDestination( &finalMix );

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer::StartInfo info;
    info.enableFlag |= nn::atk::SoundArchivePlayer::StartInfo::EnableFlagBit_OutputReceiver;
    info.pOutputReceiver = &subMix;
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART, &info).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);

    // subMix のセンド先を設定
    float volume = 1.0f;
    for(int srcBus = 0; srcBus < subMix.GetBusCount(); srcBus++)
    {
        for(int srcChannel = 0; srcChannel < subMix.GetChannelCount(); srcChannel++)
        {
            for(int dstBus = 0; dstBus < finalMix.GetBusCount(); dstBus++)
            {
                for(int dstChannel = 0; dstChannel < finalMix.GetChannelCount(); dstChannel++)
                {
                    subMix.SetSendVolume( srcBus, srcChannel, dstBus, dstChannel, volume );

                    //  各センド量を異なる値にするために volume を更新します
                    volume *= 15.0f / 16.0f;
                }
            }
        }
    }

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    nnt::atk::util::WaitForProcessCommand();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  TODO: SIGLO-65917 の暫定対処として subMix.Finalize() より先に
    //        soundArchivePlayer.Finalize() を行います
    //        この暫定対処は SIGLO-65970 の対応時に、適切に修正される予定です
    soundArchivePlayer.Finalize();
#endif

    subMix.Finalize();

    g_Allocator.Free( buffer );
    FinalizeAudioOutputComp();
}
*/

TEST(AudioOutputComp, SubMixSendFinalMix)
{
    nnt::atk::util::OnPreAtkTest();
    const int BusCount = 2;
    const int ChannelCount = 6;
    AudioOutputCompInitializeParam param;
    param.testName = "SubMixSendFinalMix";
    param.referenceFileName = "SubMixSendFinalMix.wav"; // リファレンスのファイル名
    param.outputMode = nn::atk::OutputMode_Surround;
    // CustomSubMix を有効にする
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableCustomSubMix = true;
    soundSystemParam.subMixTotalChannelCount = BusCount * ChannelCount;
    soundSystemParam.subMixCount = 1;
    param.atkCommonParam.SetSoundSystemParam(soundSystemParam);
    InitializeAudioOutputComp(param);

    // SubMix の用意
    nn::atk::FinalMix& finalMix = nn::atk::SoundSystem::GetFinalMix();
    nn::atk::SubMix subMix;
    const size_t bufferSize = nn::atk::SubMix::GetRequiredMemorySize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount() );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize );
    subMix.Initialize( BusCount, ChannelCount, finalMix.GetBusCount(), finalMix.GetChannelCount(), buffer, bufferSize );
    subMix.SetDestination( &finalMix );

    // 録音開始
    g_AudioOutputTester.StartRecording(RecordSampleCount);

    // 再生開始
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;
    //  StartInfo で指定しない場合は FinalMix に送られます
    bool result = soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART).IsSuccess();
    NN_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);

    // 既定のサンプル録音し、停止するまで待ちます
    g_AudioOutputTester.WaitForStopRecording(soundArchivePlayer);

    //  サウンドの停止
    StopSound(soundHandle, soundArchivePlayer);

    // 正解波形との比較
    g_AudioOutputTester.CompareFile(result, g_FileManager, CompareMode_ComparePerByte);
    EXPECT_TRUE(result);

    nnt::atk::util::WaitForProcessCommand();

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    //  TODO: SIGLO-65917 の暫定対処として subMix.Finalize() より先に
    //        soundArchivePlayer.Finalize() を行います
    //        この暫定対処は SIGLO-65970 の対応時に、適切に修正される予定です
    soundArchivePlayer.Finalize();
#endif

    subMix.Finalize();

    g_Allocator.Free( buffer );
    FinalizeAudioOutputComp();
}
