﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>

#include <nnt/codecUtil/testCodec_TesterManager.h>

namespace nnt {
namespace codec {
namespace util {

void TesterManager::Initialize(int threadCount, AllocatorType allocator) NN_NOEXCEPT
{
    NN_ASSERT(nullptr != allocator);
    NN_ASSERT(0 < threadCount);
    NN_ASSERT(!IsInitialized());

    // スレッド数の初期化
    m_ThreadCount = threadCount;

    // 構造体の初期化
    m_ThreadStructSize = sizeof(struct TesterThreadType) * m_ThreadCount;
    m_pThreadStruct = static_cast<struct TesterThreadType*>( allocator(m_ThreadStructSize) );
    NN_ABORT_UNLESS_NOT_NULL(m_pThreadStruct);

    std::memset(m_pThreadStruct, 0, m_ThreadStructSize);

    // スレッドスタックの初期化
    m_ThreadStackSize = m_ThreadCount * ThreadStackSize + (nn::os::ThreadStackAlignment - 1);
    m_ThreadStackBase = static_cast<char*>( allocator(m_ThreadStackSize) );
    NN_ABORT_UNLESS_NOT_NULL(m_ThreadStackBase);

    // 整列したアドレスを pointer に格納する。
    const ptrdiff_t diff = nn::os::ThreadStackAlignment
        - reinterpret_cast<uintptr_t>(m_ThreadStackBase) % nn::os::ThreadStackAlignment;
    char* pointer = m_ThreadStackBase + diff % nn::os::ThreadStackAlignment;
    NN_ABORT_UNLESS(reinterpret_cast<uintptr_t>(pointer) % nn::os::ThreadStackAlignment == 0);

    for (auto i = 0; i < m_ThreadCount; ++i)
    {
        m_pThreadStruct[i].stackBase = pointer;
        pointer += ThreadStackSize;
    }

    // 多重待オブジェクトの初期化処理
    nn::os::InitializeMultiWait(&m_MultiWait);

    m_IsInitialized = true;
}

void TesterManager::Finalize(DeallocatorType deallocator) NN_NOEXCEPT
{
    NN_ASSERT(nullptr != deallocator);
    NN_ASSERT(IsInitialized());

    m_IsInitialized = false;

    for (auto i = 0; i < m_ThreadCount; ++i)
    {
        auto& thread = m_pThreadStruct[i];
        if (true == thread.isRunning)
        {
            nn::os::WaitThread( &thread.handle );
            FinalizeTesterThread( &thread );
            thread.stackBase = nullptr;
        }
    }

    // 多重待オブジェクトの終了処理
    nn::os::FinalizeMultiWait(&m_MultiWait);

    deallocator(m_ThreadStackBase);
    deallocator(m_pThreadStruct);
}

bool TesterManager::Run(nn::fs::DirectoryEntry& entry, nn::os::ThreadFunction function,int core) NN_NOEXCEPT
{
    if (true == IsRunningAll(m_pThreadStruct, m_ThreadCount))
    {
        const auto holder = nn::os::WaitAny( &m_MultiWait );
        bool isFound = false;
        for (auto i = 0; i < m_ThreadCount; ++i)
        {
            if (holder == &m_pThreadStruct[i].holder)
            {
                FinalizeTesterThread(&m_pThreadStruct[i]);
                isFound = true;
                break;
            }
        }
        if (false == isFound)
        {
            return false;
        }
    }
    const int id = GetAvailableThreadNumber(m_pThreadStruct, m_ThreadCount);
    InitializeTesterThread(&m_pThreadStruct[id], function, &entry, core);
    return true;
}

void TesterManager::InitializeTesterThread(struct TesterThreadType* pThread, nn::os::ThreadFunction function, nn::fs::DirectoryEntry* pEntry, int core) NN_NOEXCEPT
{
    NN_ASSERT(nullptr != pThread);
    NN_ASSERT(nullptr != pEntry);
    NN_ASSERT(nullptr != function);
    NN_ASSERT(false == pThread->isRunning);

    nn::os::CreateThread(
        &pThread->handle,
        function,
        pEntry,
        pThread->stackBase,
        ThreadStackSize,
        nn::os::DefaultThreadPriority,
        core
    );

    nn::os::StartThread( &pThread->handle );
    pThread->isRunning = true;

    nn::os::InitializeMultiWaitHolder( &pThread->holder, &pThread->handle );
    nn::os::LinkMultiWaitHolder( &m_MultiWait, &pThread->holder );
}

void TesterManager::FinalizeTesterThread(struct TesterThreadType* pThread) NN_NOEXCEPT
{
    NN_ASSERT(true == pThread->isRunning);

    nn::os::UnlinkMultiWaitHolder( &pThread->holder );
    nn::os::FinalizeMultiWaitHolder( &pThread->holder );

    nn::os::DestroyThread( &pThread->handle );

    pThread->isRunning = false;
}

bool TesterManager::IsRunningAll(const struct TesterThreadType* pThread, int threadCount) const NN_NOEXCEPT
{
    for (auto i = 0; i < threadCount; ++i)
    {
        if (false == IsRunning(&pThread[i]))
        {
            return false;
        }
    }
    return true;
}

int TesterManager::GetAvailableThreadNumber(const struct TesterThreadType* pThread, int threadCount) const NN_NOEXCEPT
{
    for (auto i = 0; i < threadCount; ++i)
    {
        if (false == pThread[i].isRunning)
        {
            return i;
        }
    }
    NN_ABORT(0); // ここには到達しないはず

    return -1;
}

}}} // nnt::codec::util
