﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// std
#include <cstddef> // nullptr, std::size_t
#include <cstdlib> // std::malloc, std::free

// sdk
#include <nn/nn_Assert.h>
#include <nn/codec/detail/codec_OpusEncoderTypesInternal.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

// Externals/libopus
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <opus.h>
#include <opus_multistream.h>
#include <opus_private.h>

// Tests
#include <nnt.h>
#include <nnt/codecUtil/testCodec_UtilOpus.h>
#include <nnt/codecUtil/testCodec_NativeOpusEncoder.h>

// Programs/Eris/Sources/Libraries/codec
#include "codec_OpusUtilityInternal.h"

namespace nnt {
namespace codec {

//------------------------------------------------------------------------------
// OpusEncoder
//------------------------------------------------------------------------------
#define EncoderType ::OpusEncoder

template <>
std::size_t NativeOpusEncoder<EncoderType>::GetWorkBufferSize(int sampleRate, int streamCount, int coupledStreamCount) NN_NOEXCEPT
{
    NN_UNUSED(sampleRate);
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_EQUAL(streamCount, 1);
    NN_ASSERT(coupledStreamCount == 0 || coupledStreamCount == 1);
    return static_cast<std::size_t>(::opus_encoder_get_size(streamCount + coupledStreamCount));
}

template <>
nn::codec::OpusResult NativeOpusEncoder<EncoderType>::Initialize(
    int sampleRate,
    int channelCount,
    int streamCount,
    int coupledStreamCount,
    const uint8_t* pChannleMappingTable,
    void* workBufferAddress,
    std::size_t workBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(streamCount);
    NN_UNUSED(coupledStreamCount);
    NN_UNUSED(pChannleMappingTable);
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_NOT_EQUAL(channelCount, 0);
    NN_ASSERT_NOT_NULL(workBufferAddress);
    NN_ASSERT_GREATER_EQUAL(workBufferSize, static_cast<std::size_t>(::opus_encoder_get_size(channelCount)));
    // Initializing
    m_WorkBufferAddress = workBufferAddress;
    m_WorkBufferSize = workBufferSize;
    int status = ::opus_encoder_init(static_cast<EncoderType*>(m_WorkBufferAddress), sampleRate, channelCount, OPUS_APPLICATION_VOIP);
    if (OPUS_OK != status)
    {
        return util::ConvertResult(status);
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_StreamCount = streamCount;
    m_CoupledStreamCount = coupledStreamCount;
    m_IsInitialized = true;
    return nn::codec::OpusResult_Success;
}

template <>
nn::codec::OpusResult NativeOpusEncoder<EncoderType>::EncodeInterleaved(size_t* pOutputSize, void* outputBuffer, size_t outputBufferSize, const int16_t* inputBuffer, int inputSampleCountPerChannel) NN_NOEXCEPT
{
    using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
    const auto headerSize = sizeof(OpusHeader);
    const int status = ::opus_encode(
            static_cast<EncoderType*>(m_WorkBufferAddress),
            inputBuffer,
            inputSampleCountPerChannel,
            static_cast<uint8_t*>(outputBuffer) + headerSize,
            static_cast<int>(outputBufferSize));
    if (status < 0)
    {
        *pOutputSize = 0;
        return util::ConvertResult(status);
    }

    OpusHeader header =
    {
        static_cast<uint32_t>(status), // size
        static_cast<uint32_t>(0) // final range
    };
    nn::codec::detail::OpusPacketInternal::SetHeaderInPacket(static_cast<uint8_t*>(outputBuffer), header);

    // パケットサイズの書き戻し
    *pOutputSize = status + headerSize;
    return nn::codec::OpusResult_Success;
}

template <>
void NativeOpusEncoder<EncoderType>::Finalize() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return ;
    }
    m_IsInitialized = false;
    m_WorkBufferAddress = nullptr;
    m_WorkBufferSize = 0;
}

template <>
void NativeOpusEncoder<EncoderType>::SetBitRate(int bitRate) NN_NOEXCEPT
{
    nn::codec::detail::SetBitRateInternal(static_cast<EncoderType*>(m_WorkBufferAddress), bitRate, GetChannelCount());
}

template <>
void NativeOpusEncoder<EncoderType>::SetBitRateControl(nn::codec::OpusBitRateControl bitRateControl) NN_NOEXCEPT
{
    nn::codec::detail::SetBitRateControlInternal(static_cast<EncoderType*>(m_WorkBufferAddress), bitRateControl);
}

template <>
void NativeOpusEncoder<EncoderType>::BindCodingMode(nn::codec::OpusCodingMode  codingMode) NN_NOEXCEPT
{
    const auto bandwidth   = codingMode == nn::codec::OpusCodingMode_Silk ? nn::codec::OpusBandwidth_WideBand : nn::codec::OpusBandwidth_Auto;
    const auto application = codingMode == nn::codec::OpusCodingMode_Silk ? nn::codec::OpusApplication_Voip   : nn::codec::OpusApplication_RestrictedLowDelay;
    const auto forceMode   = codingMode == nn::codec::OpusCodingMode_Silk ? nn::codec::OpusCodingMode_Silk    : nn::codec::OpusCodingMode_Auto;
    auto pOpusEncoder = static_cast<EncoderType*>(m_WorkBufferAddress);
    // nn::codec::OpusEncoder の符号化モードは、opus バンド幅と opus 強制モード指定 (Auto/Silk/Celt) で制御している。
    nn::codec::detail::SetForceModeInternal  (pOpusEncoder, forceMode);
    nn::codec::detail::SetBandwidthInternal  (pOpusEncoder, bandwidth);
    nn::codec::detail::SetApplicationInternal(pOpusEncoder, application);
}

template <>
void NativeOpusEncoder<EncoderType>::GetCodingMode() NN_NOEXCEPT
{
    // Not Implemented
}

template class NativeOpusEncoder<EncoderType>;

#undef EncoderType

//------------------------------------------------------------------------------
// OpusMSEncoder
//------------------------------------------------------------------------------
#define EncoderType ::OpusMSEncoder

template <>
std::size_t NativeOpusEncoder<EncoderType>::GetWorkBufferSize(int sampleRate, int streamCount, int coupledStreamCount) NN_NOEXCEPT
{
    NN_UNUSED(sampleRate);
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_GREATER(streamCount, 0);
    NN_ASSERT_LESS_EQUAL(coupledStreamCount, streamCount);
    return static_cast<std::size_t>(::opus_multistream_encoder_get_size(streamCount, coupledStreamCount));
}

template <>
nn::codec::OpusResult NativeOpusEncoder<EncoderType>::Initialize(
    int sampleRate,
    int channelCount,
    int streamCount,
    int coupledStreamCount,
    const uint8_t* pChannleMappingTable,
    void* workBufferAddress,
    std::size_t workBufferSize) NN_NOEXCEPT
{
    NN_ASSERT_NOT_EQUAL(sampleRate, 0);
    NN_ASSERT_NOT_EQUAL(channelCount, 0);
    NN_ASSERT_NOT_NULL(workBufferAddress);
    NN_ASSERT_GREATER_EQUAL(workBufferSize, static_cast<std::size_t>(::opus_multistream_encoder_get_size(streamCount, coupledStreamCount)));
    // Initializing
    m_WorkBufferAddress = workBufferAddress;
    m_WorkBufferSize = workBufferSize;
    const int status = ::opus_multistream_encoder_init(
        static_cast<EncoderType*>(m_WorkBufferAddress),
        sampleRate,
        channelCount,
        streamCount,
        coupledStreamCount,
        pChannleMappingTable,
        OPUS_APPLICATION_VOIP
    );
    if (OPUS_OK != status)
    {
        return util::ConvertResult(status);
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_StreamCount = streamCount;
    m_CoupledStreamCount = coupledStreamCount;
    m_IsInitialized = true;
    return nn::codec::OpusResult_Success;
}

template <>
nn::codec::OpusResult NativeOpusEncoder<EncoderType>::EncodeInterleaved(size_t* pOutputSize, void* outputBuffer, size_t outputBufferSize, const int16_t* inputBuffer, int inputSampleCountPerChannel) NN_NOEXCEPT
{
    NN_UNUSED(pOutputSize);
    NN_UNUSED(outputBuffer);
    NN_UNUSED(outputBufferSize);
    NN_UNUSED(inputBuffer);
    NN_UNUSED(inputSampleCountPerChannel);
    using OpusHeader = nn::codec::detail::OpusPacketInternal::Header;
    const auto headerSize = sizeof(OpusHeader);
    const int status = ::opus_multistream_encode(
        static_cast<EncoderType*>(m_WorkBufferAddress),
        inputBuffer,
        inputSampleCountPerChannel,
        static_cast<uint8_t*>(outputBuffer) + headerSize,
        static_cast<opus_int32>(outputBufferSize)
    );
    if (status < 0)
    {
        *pOutputSize = 0;
        return util::ConvertResult(status);
    }

    OpusHeader header =
    {
        static_cast<uint32_t>(status), // size
        static_cast<uint32_t>(0) // final range
    };
    nn::codec::detail::OpusPacketInternal::SetHeaderInPacket(static_cast<uint8_t*>(outputBuffer), header);

    // パケットサイズの書き戻し
    *pOutputSize = status + headerSize;
    return nn::codec::OpusResult_Success;
}

template <>
void NativeOpusEncoder<EncoderType>::Finalize() NN_NOEXCEPT
{
    if (!IsInitialized())
    {
        return ;
    }
    m_IsInitialized = false;
    m_WorkBufferAddress = nullptr;
    m_WorkBufferSize = 0;
}

template <>
void NativeOpusEncoder<EncoderType>::SetBitRate(int bitRate) NN_NOEXCEPT
{
    NN_UNUSED(bitRate);
    ::opus_multistream_encoder_ctl(
        static_cast<EncoderType*>(m_WorkBufferAddress),
        OPUS_SET_BITRATE(bitRate)
    );
}

template <>
void NativeOpusEncoder<EncoderType>::SetBitRateControl(nn::codec::OpusBitRateControl bitRateControl) NN_NOEXCEPT
{
    NN_UNUSED(bitRateControl);
    // Cvbr at default
    int vbr = 1;
    int cvbr = 1;
    if (bitRateControl == nn::codec::OpusBitRateControl_Vbr)
    {
        cvbr = 0;
    }
    else if (bitRateControl == nn::codec::OpusBitRateControl_Cbr)
    {
        vbr = 0;
    }
    ::opus_multistream_encoder_ctl(
        static_cast<EncoderType*>(m_WorkBufferAddress),
        OPUS_SET_VBR(vbr)
    );
    ::opus_multistream_encoder_ctl(
        static_cast<EncoderType*>(m_WorkBufferAddress),
        OPUS_SET_VBR_CONSTRAINT(cvbr)
    );
}

template <>
void NativeOpusEncoder<EncoderType>::BindCodingMode(nn::codec::OpusCodingMode  codingMode) NN_NOEXCEPT
{
    NN_UNUSED(codingMode);
    const auto bandwidth   = codingMode == nn::codec::OpusCodingMode_Silk ? OPUS_BANDWIDTH_WIDEBAND : OPUS_AUTO;
    const auto application = codingMode == nn::codec::OpusCodingMode_Silk ? OPUS_APPLICATION_VOIP : OPUS_APPLICATION_RESTRICTED_LOWDELAY;
    const auto forceMode   = codingMode == nn::codec::OpusCodingMode_Silk ? MODE_SILK_ONLY : MODE_CELT_ONLY;
    auto pOpusEncoder = static_cast<EncoderType*>(m_WorkBufferAddress);
    // nn::codec::OpusEncoder の符号化モードは、opus バンド幅と opus 強制モード指定 (Auto/Silk/Celt) で制御している。
    ::opus_multistream_encoder_ctl(pOpusEncoder, OPUS_SET_BANDWIDTH(bandwidth));
    ::opus_multistream_encoder_ctl(pOpusEncoder, OPUS_SET_APPLICATION(application));
    ::opus_multistream_encoder_ctl(pOpusEncoder, OPUS_SET_FORCE_MODE(forceMode));
}

template <>
void NativeOpusEncoder<EncoderType>::GetCodingMode() NN_NOEXCEPT
{
#if 0
    auto pOpusEncoder = static_cast<EncoderType*>(m_WorkBufferAddress);
    ::OpusEncoder* streamenc = nullptr;
    for (auto k = 0; k < GetStreamCount(); ++k)
    {
        int status = ::opus_multistream_encoder_ctl(pOpusEncoder, OPUS_MULTISTREAM_GET_ENCODER_STATE(k, &streamenc));
        NN_ASSERT_EQUAL(status, OPUS_OK);
        int app;
        ::opus_encoder_ctl(streamenc, OPUS_GET_APPLICATION(&app));
        int mode;
        ::opus_encoder_ctl(streamenc, OPUS_GET_FORCE_MODE(&mode));
        int bitrate;
        ::opus_encoder_ctl(streamenc, OPUS_GET_BITRATE(&bitrate));
    }
#endif
}

template class NativeOpusEncoder<EncoderType>;

#undef EncoderType

}} // nnt::codec
