﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nnt/audioUtil/testAudio_NodeAllocator.h>

namespace nnt {
namespace audio {
namespace util {


template <typename SelfType, typename BaseType>
template <typename DestType>
SelfType* MultiDestinationBase<SelfType, BaseType>::SetDestinationImpl(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, DestType* destination)
{
    m_DestinationArray[destinationIndex].dest = destination;
    nn::audio::SetSplitterDestination(pConfig, &m_Splitter, destinationIndex, destination->GetBase());
    return static_cast<SelfType*>(this);
}

template <typename SelfType, typename BaseType>
template <typename DestType>
void MultiDestinationBase<SelfType, BaseType>::SetMixVolumeImpl(int destinationIndex, float volume, int sourceChannel, int destChannel, DestType* destination)
{
    nn::audio::SetSplitterMixVolume(&m_Splitter, destinationIndex, destination->GetBase(), volume, sourceChannel, destChannel);
}

template <typename SelfType, typename BaseType>
template <typename DestType>
void MultiDestinationBase<SelfType, BaseType>::SetMixVolumeMatrixImpl(int destinationIndex, const float* volumes, int sourceChannelCount, int destChannelCount, DestType* destination)
{
    for (auto i = 0; i < sourceChannelCount; ++i)
    {
        for (auto j = 0; j < destChannelCount; ++j)
        {
            nn::audio::SetSplitterMixVolume(&m_Splitter, destinationIndex, destination->GetBase(), volumes[i * destChannelCount + j], i, j);
        }
    }
}


//////////////////////////////////////////////////////////////////////////
// FinalMix
//////////////////////////////////////////////////////////////////////////
FinalMix::FinalMix()
{
    GetBase()->_pMixInfo = nullptr;
}

AudioNode::NodeTypeInfo FinalMix::GetType() const
{
    return FinalMix::_TypeInfo;
}

nn::audio::NodeId FinalMix::GetNodeId() const
{
    return nn::audio::GetFinalMixNodeId(GetBase());
}

void FinalMix::SetVolume(float volume)
{
    nn::audio::SetFinalMixVolume(GetBase(), volume);
}

int FinalMix::GetChannelCount() const
{
    return nn::audio::GetFinalMixBufferCount(GetBase());
}


//////////////////////////////////////////////////////////////////////////
// SubMix
//////////////////////////////////////////////////////////////////////////

AudioNode::NodeTypeInfo SubMix::GetType() const
{
    return SubMix::_TypeInfo;
}

SubMix::SubMix()
{
    GetBase()->_pMixInfo = nullptr;
    m_Destination = nullptr;
    m_EffectList.clear();
}

nn::audio::NodeId SubMix::GetNodeId() const
{
    return nn::audio::GetSubMixNodeId(GetBase());
}

void SubMix::SetVolume(float volume)
{
    nn::audio::SetSubMixVolume(GetBase(), volume);
}

int SubMix::GetChannelCount() const
{
    return nn::audio::GetSubMixBufferCount(GetBase());
}

template <typename DestType>
void SubMix::SetMixVolumeImpl(float volume, int sourceChannel, int destChannel, DestType* destination)
{
    nn::audio::SetSubMixMixVolume(GetBase(), destination->GetBase(), volume, sourceChannel, destChannel);
}

SubMix* SubMix::SetMixVolume(float volume, int sourceChannel, int destChannel)
{
    if (m_Destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        SetMixVolumeImpl(volume, sourceChannel, destChannel, m_Destination->As<SubMix>());
    }
    else if (m_Destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        SetMixVolumeImpl(volume, sourceChannel, destChannel, m_Destination->As<MultiDestinationSubMix>());
    }
    else if (m_Destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        SetMixVolumeImpl(volume, sourceChannel, destChannel, m_Destination->As<FinalMix>());
    }
    else if (m_Destination->GetType() == AudioNode::NodeTypeInfo::Splitter)
    {
        // 接続先が Splitter の場合、ミックスボリュームが設定出来ないのでとりあえず何もしないことにしておく
        NN_LOG("[SubMix::SetMixVolume] destination == splitter is not supported.");
    }
    else
    {
        NN_ABORT(0);
    }
    return this;
}

template <typename DestType>
SubMix* SubMix::SetDestinationImpl(nn::audio::AudioRendererConfig* pConfig, DestType* destination)
{
    if (m_Destination == destination)
    {
        return this;
    }
    nn::audio::SetSubMixDestination(pConfig, GetBase(), destination->GetBase());
    m_Destination = destination;
    return this;
}

SubMix* SubMix::SetDestination(nn::audio::AudioRendererConfig* pConfig, AudioNode* destination)
{
    if (destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        return SetDestinationImpl(pConfig, destination->As<SubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        return SetDestinationImpl(pConfig, destination->As<MultiDestinationSubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        return SetDestinationImpl(pConfig, destination->As<FinalMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::Splitter)
    {
        return SetDestinationImpl(pConfig, destination->As<Splitter>());
    }
    NN_ABORT(0);
}

MultiDestinationSubMix::MultiDestinationSubMix()
{
    GetBase()->_pMixInfo = nullptr;
    m_Splitter._pSplitterInfo = nullptr;
    for (auto& d : m_DestinationArray)
    {
        d.dest = nullptr;
    }
    m_EffectList.clear();
}

AudioNode::NodeTypeInfo MultiDestinationSubMix::GetType() const
{
    return MultiDestinationSubMix::_TypeInfo;
}

nn::audio::NodeId MultiDestinationSubMix::GetNodeId() const
{
    return nn::audio::GetSubMixNodeId(GetBase());
}

void MultiDestinationSubMix::SetVolume(float volume)
{
    nn::audio::SetSubMixVolume(GetBase(), volume);
}

int MultiDestinationSubMix::GetChannelCount() const
{
    return nn::audio::GetSubMixBufferCount(GetBase());
}

MultiDestinationSubMix* MultiDestinationSubMix::SetDestination(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, AudioNode* destination)
{
    if (destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<SubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<MultiDestinationSubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<FinalMix>());
    }
    NN_ABORT(0);
}

MultiDestinationSubMix* MultiDestinationSubMix::SetMixVolume(int destinationIndex, float volume, int sourceChannel, int destChannel)
{
    if (m_DestinationArray[destinationIndex].dest == nullptr)
    {
        return nullptr;
    }

    AudioNode* node = m_DestinationArray[destinationIndex].dest;
    if (node->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<SubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<MultiDestinationSubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<FinalMix>());
    }
    else
    {
        NN_ABORT(0);
    }
    return this;
}


//////////////////////////////////////////////////////////////////////////
// Voice
//////////////////////////////////////////////////////////////////////////

Voice::Voice()
{
    GetBase()->_pVoiceInfo = nullptr;
    m_Destination = nullptr;
}

AudioNode::NodeTypeInfo Voice::GetType() const
{
    return Voice::_TypeInfo;
}

nn::audio::NodeId Voice::GetNodeId() const
{
    return nn::audio::GetVoiceNodeId(GetBase());
}

void Voice::SetVolume(float volume)
{
    nn::audio::SetVoiceVolume(GetBase(), volume);
}

int Voice::GetChannelCount() const
{
    return nn::audio::GetVoiceChannelCount(GetBase());
}

Voice* Voice::Start()
{
    nn::audio::SetVoicePlayState(GetBase(), nn::audio::VoiceType::PlayState_Play);
    return this;
}

Voice* Voice::Stop()
{
    nn::audio::SetVoicePlayState(GetBase(), nn::audio::VoiceType::PlayState_Stop);
    return this;
}

Voice* Voice::AppendWaveBuffer(nn::audio::WaveBuffer* pWaveBuffer)
{
    nn::audio::AppendWaveBuffer(GetBase(), pWaveBuffer);
    return this;
}

Voice* Voice::SetMixVolume(float volume, int sourceChannel, int destChannel)
{
    if (m_Destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        SetMixVolumeImpl(volume, sourceChannel, destChannel, m_Destination->As<SubMix>());
    }
    else if (m_Destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        SetMixVolumeImpl(volume, sourceChannel, destChannel, m_Destination->As<MultiDestinationSubMix>());
    }
    else if (m_Destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        SetMixVolumeImpl(volume, sourceChannel, destChannel, m_Destination->As<FinalMix>());
    }
    else if (m_Destination->GetType() == AudioNode::NodeTypeInfo::Splitter)
    {
        // 接続先が Splitter の場合、ミックスボリュームが設定出来ないのでとりあえず何もしないことにしておく
        NN_LOG("[Voice::SetMixVolume] destination == splitter is not supported.");
    }
    else
    {
        NN_ABORT(0);
    }
    return this;
}

template <typename DestType>
Voice* Voice::SetDestinationImpl(nn::audio::AudioRendererConfig* pConfig, DestType* destination)
{
    nn::audio::SetVoiceDestination(pConfig, this->GetBase(), destination->GetBase());
    m_Destination = destination;
    return this;
}

Voice* Voice::SetDestination(nn::audio::AudioRendererConfig* pConfig, AudioNode* destination)
{
    if (destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        return SetDestinationImpl(pConfig, destination->As<SubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        return SetDestinationImpl(pConfig, destination->As<MultiDestinationSubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        return SetDestinationImpl(pConfig, destination->As<FinalMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::Splitter)
    {
        return SetDestinationImpl(pConfig, destination->As<Splitter>());
    }
    else
    {
        NN_ABORT(0);
        return this;
    }
}

template <typename DestType>
void Voice::SetMixVolumeImpl(float volume, int sourceChannel, int destChannel, DestType* destination)
{
    nn::audio::SetVoiceMixVolume(GetBase(), destination->GetBase(), volume, sourceChannel, destChannel);
}

MultiDestinationVoice::MultiDestinationVoice()
{
    GetBase()->_pVoiceInfo = nullptr;
}

AudioNode::NodeTypeInfo MultiDestinationVoice::GetType() const
{
    return MultiDestinationVoice::_TypeInfo;
}

nn::audio::NodeId MultiDestinationVoice::GetNodeId() const
{
    return nn::audio::GetVoiceNodeId(GetBase());
}

void MultiDestinationVoice::SetVolume(float volume)
{
    nn::audio::SetVoiceVolume(GetBase(), volume);
}

int MultiDestinationVoice::GetChannelCount() const
{
    return nn::audio::GetVoiceChannelCount(GetBase());
}

MultiDestinationVoice* MultiDestinationVoice::SetDestination(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, AudioNode* destination)
{
    if (destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<SubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<MultiDestinationSubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<FinalMix>());
    }
    NN_ABORT(0);
}

MultiDestinationVoice* MultiDestinationVoice::SetMixVolume(int destinationIndex, float volume, int sourceChannel, int destChannel)
{
    if (m_DestinationArray[destinationIndex].dest == nullptr)
    {
        return nullptr;
    }

    auto node = m_DestinationArray[destinationIndex].dest;

    if (node->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<SubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<MultiDestinationSubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<FinalMix>());
    }
    else
    {
        NN_ABORT(0);
    }
    return this;
}


MultiDestinationVoice* MultiDestinationVoice::Start()
{
    nn::audio::SetVoicePlayState(GetBase(), nn::audio::VoiceType::PlayState_Play);
    return this;
}

MultiDestinationVoice* MultiDestinationVoice::Stop()
{
    nn::audio::SetVoicePlayState(GetBase(), nn::audio::VoiceType::PlayState_Stop);
    return this;
}

MultiDestinationVoice* MultiDestinationVoice::AppendWaveBuffer(nn::audio::WaveBuffer* pWaveBuffer)
{
    nn::audio::AppendWaveBuffer(GetBase(), pWaveBuffer);
    return this;
}

//////////////////////////////////////////////////////////////////////////
// Splitter
//////////////////////////////////////////////////////////////////////////

Splitter::Splitter()
{
    GetSplitterBase()->_pSplitterInfo = nullptr;
}

AudioNode::NodeTypeInfo Splitter::GetType() const
{
    return Splitter::_TypeInfo;
}

nn::audio::NodeId Splitter::GetNodeId() const
{
    NN_SDK_ASSERT(false, "not supported.");
    return 0u;
}

void Splitter::SetVolume(float volume)
{
    NN_UNUSED(volume);
    NN_SDK_ASSERT(false, "not supported.");
}

int Splitter::GetChannelCount() const
{
    return nn::audio::GetSplitterSourceChannelCount(GetSplitterBase());
}

Splitter* Splitter::SetDestination(nn::audio::AudioRendererConfig* pConfig, int destinationIndex, AudioNode* destination)
{
    if (destination->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<SubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<MultiDestinationSubMix>());
    }
    else if (destination->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        return SetDestinationImpl(pConfig, destinationIndex, destination->As<FinalMix>());
    }
    NN_ABORT(0);
}

Splitter* Splitter::SetMixVolume(int destinationIndex, float volume, int sourceChannel, int destChannel)
{
    if (m_DestinationArray[destinationIndex].dest == nullptr)
    {
        return nullptr;
    }

    AudioNode* node = m_DestinationArray[destinationIndex].dest;
    if (node->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<SubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<MultiDestinationSubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        SetMixVolumeImpl(destinationIndex, volume, sourceChannel, destChannel, node->As<FinalMix>());
    }
    else
    {
        NN_ABORT(0);
    }
    return this;
}

nn::audio::SplitterType* Splitter::GetBase()
{
    return GetSplitterBase();
}

const nn::audio::SplitterType* Splitter::GetBase() const
{
    return GetSplitterBase();
}

//////////////////////////////////////////////////////////////////////////
// Aux
//////////////////////////////////////////////////////////////////////////
nnt::audio::util::AudioEffect::EffectTypeInfo Aux::GetType() const
{
    return _TypeInfo;
}

size_t Aux::GetWorkBufferSize(const nn::audio::AudioRendererParameter* param, int channelCount, int frameCount)
{
    return nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(param, frameCount, channelCount);
}

nnt::audio::util::Aux* Aux::SetupBuffers(void* sendBuffer, void* returnBuffer, size_t bufferSize)
{
    m_SendBuffer = sendBuffer;
    m_ReturnBuffer = returnBuffer;
    m_BufferSize = bufferSize;
    return this;
}

nnt::audio::util::Aux* Aux::SetInOut(const int8_t* input, const int8_t* output, const int count)
{
    nn::audio::SetAuxInputOutput(GetBase(), input, output, count);
    return this;
}

int Aux::Read(int32_t* readBuffer, int count)
{
    return nn::audio::ReadAuxSendBuffer(GetBase(), readBuffer, count);
}

void Aux::Write(const int32_t* writeBuffer, int count)
{
    nn::audio::WriteAuxReturnBuffer(GetBase(), writeBuffer, count);
}
template <typename TargetNode>
Aux* Aux::AddToImpl(nn::audio::AudioRendererConfig* pConfig, TargetNode* node)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::audio::AddAux(pConfig, GetBase(), node->GetBase(), m_SendBuffer, m_ReturnBuffer, m_BufferSize));
    return this;
}

Aux* Aux::AddTo(nn::audio::AudioRendererConfig* pConfig, AudioNode* node)
{
    if (node->GetType() == AudioNode::NodeTypeInfo::SubMix)
    {
        return AddToImpl(pConfig, node->As<SubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::MultiDestinationSubMix)
    {
        return AddToImpl(pConfig, node->As<MultiDestinationSubMix>());
    }
    else if (node->GetType() == AudioNode::NodeTypeInfo::FinalMix)
    {
        return AddToImpl(pConfig, node->As<FinalMix>());
    }
    NN_ABORT(0);
}

int Aux::GetSampleCount() const NN_NOEXCEPT
{
    return nn::audio::GetAuxSampleCount(GetBase());
}

//////////////////////////////////////////////////////////////////////////
// AudioNodeAllocator
//////////////////////////////////////////////////////////////////////////
AudioNodeAllocator::AudioNodeAllocator(
    void* voiceBuffer, int voiceCount,
    void* multiVoiceBuffer, int multiVoiceCount,
    void* splitterBuffer, int splitterCount,
    void* subMixBuffer, int subMixCount,
    void* multiSubMixBuffer, int multiSubMixCount,
    void* finalMixBuffer, int finalMixCount)
    : m_VoiceCount(voiceCount)
    , m_MultiDestVoiceCount(multiVoiceCount)
    , m_SplitterCount(splitterCount)
    , m_SubMixCount(subMixCount)
    , m_MultiDestSubMixCount(multiSubMixCount)
    , m_FinalMixCount(finalMixCount)
{
    m_pVoices = reinterpret_cast<Voice*>(voiceBuffer);
    m_pMultiDestinationVoices = reinterpret_cast<MultiDestinationVoice*>(multiVoiceBuffer);
    m_pSplitters = reinterpret_cast<Splitter*>(splitterBuffer);
    m_pSubMixes = reinterpret_cast<SubMix*>(subMixBuffer);
    m_pFinalMixes = reinterpret_cast<FinalMix*>(finalMixBuffer);
    m_pMultiDestinationSubMixes = reinterpret_cast<MultiDestinationSubMix*>(multiSubMixBuffer);

    InitializeList<Voice>(m_FreeVoice, m_pVoices, m_VoiceCount);
    InitializeList<MultiDestinationVoice>(m_FreeMultidestinationVoice, m_pMultiDestinationVoices, m_MultiDestVoiceCount);
    InitializeList<Splitter>(m_FreeSplitter, m_pSplitters, m_SplitterCount);
    InitializeList<SubMix>(m_FreeSubMix, m_pSubMixes, m_SubMixCount);
    InitializeList<MultiDestinationSubMix>(m_FreeMultidestinationSubMix, m_pMultiDestinationSubMixes, m_MultiDestSubMixCount);
    InitializeList<FinalMix>(m_FreeFinalMix, m_pFinalMixes, m_FinalMixCount);
}

nnt::audio::util::FinalMix* AudioNodeAllocator::AllocateFinalMix(nn::audio::AudioRendererConfig* pConfig, int bufferCount)
{
    FinalMix* rt = nullptr;
    if (m_FreeFinalMix.size() > 0 &&
        nn::audio::AcquireFinalMix(pConfig, m_FreeFinalMix.front().As<FinalMix>()->GetBase(), bufferCount))
    {
        rt = m_FreeFinalMix.front().As<FinalMix>();
        m_FreeFinalMix.pop_front();
    }
    return rt;
}

nnt::audio::util::SubMix* AudioNodeAllocator::AllocateSubMix(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int bufferCount)
{
    SubMix* rt = nullptr;
    if (m_FreeSubMix.size() > 0 &&
        nn::audio::AcquireSubMix(pConfig, m_FreeSubMix.front().As<SubMix>()->GetBase(), sampleRate, bufferCount))
    {
        rt = m_FreeSubMix.front().As<SubMix>();
        m_FreeSubMix.pop_front();
    }
    return rt;
}

nnt::audio::util::MultiDestinationSubMix* AudioNodeAllocator::AllocateMultiDestinationSubMix(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int bufferCount, int destinationCount)
{
    if (m_FreeMultidestinationSubMix.size() <= 0)
    {
        return nullptr;
    }

    MultiDestinationSubMix*  rt = m_FreeMultidestinationSubMix.front().As<MultiDestinationSubMix>();
    if (nn::audio::AcquireSubMix(pConfig, rt->GetBase(), sampleRate, bufferCount) &&
        nn::audio::AcquireSplitter(pConfig, rt->GetSplitterBase(), sampleRate, bufferCount, destinationCount))
    {
        m_FreeMultidestinationSubMix.pop_front();
        nn::audio::SetSubMixDestination(pConfig, rt->GetBase(), rt->GetSplitterBase());
        return rt;
    }

    // Failed to allocate. clean up
    if (rt->GetBase()->_pMixInfo != nullptr)
    {
        nn::audio::ReleaseSubMix(pConfig, rt->GetBase());
    }
    return nullptr;
}

nnt::audio::util::Voice* AudioNodeAllocator::AllocateVoice(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int channelCount, nn::audio::SampleFormat format, int priority, const void* pParameter, size_t paramSize)
{
    Voice* rt = nullptr;
    if (m_FreeVoice.size() > 0 &&
        nn::audio::AcquireVoiceSlot(pConfig, m_FreeVoice.front().As<Voice>()->GetBase(), sampleRate, channelCount, format, priority, pParameter, paramSize))
    {
        rt = m_FreeVoice.front().As<Voice>();
        m_FreeVoice.pop_front();
    }
    return rt;
}

nnt::audio::util::MultiDestinationVoice* AudioNodeAllocator::AllocateMultiDestinationVoice(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int channelCount, nn::audio::SampleFormat format, int priority, const void* pParameter, size_t paramSize, int destinationCount)
{
    if (m_FreeMultidestinationVoice.size() <= 0)
    {
        return nullptr;
    }

    MultiDestinationVoice* rt = m_FreeMultidestinationVoice.front().As<MultiDestinationVoice>();
    if (nn::audio::AcquireVoiceSlot(pConfig, rt->GetBase(), sampleRate, channelCount, format, priority, pParameter, paramSize) &&
        nn::audio::AcquireSplitter(pConfig, rt->GetSplitterBase(), sampleRate, channelCount, destinationCount))
    {
        m_FreeMultidestinationVoice.pop_front();
        nn::audio::SetVoiceDestination(pConfig, rt->GetBase(), rt->GetSplitterBase());
        return rt;
    }

    if (rt->GetBase()->_pVoiceInfo != nullptr)
    {
        nn::audio::ReleaseVoiceSlot(pConfig, rt->GetBase());
    }
    return nullptr;
}

nnt::audio::util::Splitter* AudioNodeAllocator::AllocateSplitter(nn::audio::AudioRendererConfig* pConfig, int sampleRate, int bufferCount, int destinationCount)
{
    if (m_FreeSplitter.size() <= 0)
    {
        return nullptr;
    }

    Splitter*  rt = m_FreeSplitter.front().As<Splitter>();
    if (nn::audio::AcquireSplitter(pConfig, rt->GetSplitterBase(), sampleRate, bufferCount, destinationCount))
    {
        m_FreeSplitter.pop_front();
        return rt;
    }

    return nullptr;
}

} // namespace nnt
} // namespace audio
} // namespace util
