﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/atkUtil/testAtk_ProfileStatistics.h>
#include <nn/nn_Log.h>

namespace nnt { namespace atk { namespace util {

ProfileStatistics::ProfileStatistics(int offset) NN_NOEXCEPT
    : m_OffsetCount(offset)
{
}

ProfileStatistics::ProfileStatistics() NN_NOEXCEPT
    : m_OffsetCount(0)
{
}

void ProfileStatistics::AddProfile(nn::atk::SoundProfile& profile, const nn::os::Tick previousAtkFrameProcessBegin) NN_NOEXCEPT
{
    if(m_OffsetCount > 0)
    {
        --m_OffsetCount;
        return;
    }

    m_TotalVoice.AddData(profile.totalVoiceCount);
    m_NwFrameProcess.AddData(profile.nwFrameProcess.GetSpan().GetMicroSeconds());
    m_RendererFrameProcess.AddData(profile.rendererFrameProcess.GetSpan().GetMicroSeconds());
    m_VoiceProcess.AddData(profile.voiceProcess.GetSpan().GetMicroSeconds());
    if(profile.totalVoiceCount != 0)
    {
        float voiceProcessPerVoice = static_cast<float>(profile.voiceProcess.GetSpan().GetMicroSeconds()) / profile.totalVoiceCount;
        m_VoiceProcessPerVoice.AddData(voiceProcessPerVoice);
    }
    m_MainMixProcess.AddData(profile.mainMixProcess.GetSpan().GetMicroSeconds());
    m_FinalMixProcess.AddData(profile.finalMixProcess.GetSpan().GetMicroSeconds());
    m_SinkProcess.AddData(profile.sinkProcess.GetSpan().GetMicroSeconds());
    m_NwFrameProcessInterval.AddData(profile.GetNwFrameProcessInterval(previousAtkFrameProcessBegin).GetMicroSeconds());
}

void ProfileStatistics::PrintMeanProfile(const char* label) NN_NOEXCEPT
{
    NN_LOG("##teamcity[buildStatisticValue key='%sAtkFrameAverage(us)' value='%f']\n", label, m_NwFrameProcess.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sRendererFrameAverage(us)' value='%f']\n", label, m_RendererFrameProcess.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sVoiceAverage(us)' value='%f']\n", label, m_VoiceProcess.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sVoiceProcessPerVoiceAverage(us)' value='%f']\n", label, m_VoiceProcessPerVoice.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sSubMixAverage(us)' value='%f']\n", label, m_MainMixProcess.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sFinalMixAverage(us)' value='%f']\n", label, m_FinalMixProcess.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sSinkAverage(us)' value='%f']\n", label, m_SinkProcess.GetMean());
    NN_LOG("##teamcity[buildStatisticValue key='%sAtkFrameIntervalAverage(us)' value='%f']\n", label, m_NwFrameProcessInterval.GetMean());
}

void ProfileStatistics::PrintMeanProfile() NN_NOEXCEPT
{
    PrintMeanProfile("");
}

void ProfileStatistics::PrintMaxProfile(const char* label) NN_NOEXCEPT
{
    int64_t max = m_NwFrameProcess.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sAtkFrameMax(us)' value='%d']\n", label, static_cast<int>(max));
    max = m_RendererFrameProcess.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sRendererFrameMax(us)' value='%d']\n", label, static_cast<int>(max));
    max = m_VoiceProcess.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sVoiceMax(us)' value='%d']\n", label, static_cast<int>(max));
    NN_LOG("##teamcity[buildStatisticValue key='%sVoiceProcessPerVoiceMax(us)' value='%f']\n", label, m_VoiceProcessPerVoice.GetMax());
    max = m_MainMixProcess.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sSubMixMax(us)' value='%d']\n", label, static_cast<int>(max));
    max = m_FinalMixProcess.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sFinalMixMax(us)' value='%d']\n", label, static_cast<int>(max));
    max = m_SinkProcess.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sSinkMax(us)' value='%d']\n", label, static_cast<int>(max));
    max = m_NwFrameProcessInterval.GetMax();
    NN_LOG("##teamcity[buildStatisticValue key='%sAtkFrameIntervalMax(us)' value='%d']\n", label, static_cast<int>(max));
}

void ProfileStatistics::PrintMaxProfile() NN_NOEXCEPT
{
    PrintMaxProfile("");
}

void ProfileStatistics::Reset() NN_NOEXCEPT
{
    m_TotalVoice.ResetData();
    m_NwFrameProcess.ResetData();
    m_RendererFrameProcess.ResetData();
    m_VoiceProcess.ResetData();
    m_VoiceProcessPerVoice.ResetData();
    m_MainMixProcess.ResetData();
    m_FinalMixProcess.ResetData();
    m_SinkProcess.ResetData();
    m_NwFrameProcessInterval.ResetData();
}

}}}
