﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>

// 実装
#include <nn/sf/sf_Types.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>

using namespace nn::sf;
using namespace nn::sf::cmif::server;

#define MY_EXPECT_EQ(a, b) EXPECT_TRUE((a) == (b))

class IFoo : public nn::sf::IServiceObject
{
private:

    virtual int sync_Bar(int n) NN_NOEXCEPT = 0;

public:

    static const int MethodId_Bar = 100;

    int Bar(int n) NN_NOEXCEPT
    {
        return this->sync_Bar(n);
    }

};

class Foo : public IFoo
{
public:

    int refCount;

    Foo() : refCount(1) {}

private:

    virtual int sync_Bar(int n) NN_NOEXCEPT
    {
        return n * n;
    }

    virtual void AddReference() NN_NOEXCEPT
    {
        ++this->refCount;
    }

    virtual void Release() NN_NOEXCEPT
    {
        --this->refCount;
    }

};

class MyServerMessage : public CmifServerMessage
{
public:

    int rawOutInt;

    virtual nn::Result PrepareForProcess(const nn::sf::cmif::CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT
    {
        NN_UNUSED(metaInfo);
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetMessageBuffersInfo(const CmifServerMessageBuffersInfo& info) const NN_NOEXCEPT
    {
        NN_UNUSED(info);
        NN_ABORT();
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetBuffers(nn::sf::cmif::PointerAndSize* buffers) const NN_NOEXCEPT
    {
        NN_UNUSED(buffers);
        NN_ABORT();
    }

    virtual void BeginPreparingForReply(nn::sf::cmif::PointerAndSize* pOutRawData) NN_NOEXCEPT
    {
        pOutRawData->pointer = reinterpret_cast<uintptr_t>(&rawOutInt);
        pOutRawData->size = sizeof(rawOutInt);
    }

    virtual void BeginPreparingForErrorReply(nn::sf::cmif::PointerAndSize* pOutRawData, size_t outRawSize) NN_NOEXCEPT
    {
        NN_UNUSED(pOutRawData);
        NN_UNUSED(outRawSize);
        NN_ABORT();
    }

};

namespace nn { namespace sf { namespace cmif { namespace server { namespace detail {

template <typename Tag>
struct CmifProcessFunctionTableGetter<IFoo, Tag>
{
    static const CmifProcessFunctionTable s_Table;
    static nn::Result ProcessServerMessage(IServiceObject* p, CmifServerMessage* pMessage, const PointerAndSize& inRawData) NN_NOEXCEPT
    {
        auto& message = *pMessage;
        auto arg = *reinterpret_cast<int*>(inRawData.pointer);
        auto ret = static_cast<IFoo*>(p)->Bar(arg);
        PointerAndSize outRawData;
        message.BeginPreparingForReply(&outRawData);
        *reinterpret_cast<int*>(outRawData.pointer) = ret;
        message.EndPreparingForReply();
        NN_RESULT_SUCCESS;
    }
};

template <typename Tag>
const CmifProcessFunctionTable CmifProcessFunctionTableGetter<IFoo, Tag>::s_Table =
{
    &CmifProcessFunctionTableGetter<IFoo, Tag>::ProcessServerMessage,
};

}}}}}

TEST(sf, Cmif_ProcessServerMessage)
{
    Foo foo;
    SharedPointer<IFoo> p(&foo, false);
    CmifServerObjectInfo info(std::move(p));
    MyServerMessage message;
    int rawInInt = 10;
    nn::sf::cmif::PointerAndSize inRaw;
    inRaw.pointer = reinterpret_cast<uintptr_t>(&rawInInt);
    inRaw.size = sizeof(rawInInt);
    auto result = info.ProcessServerMessage(&message, inRaw);
    ASSERT_TRUE(result.IsSuccess());
    MY_EXPECT_EQ(100, message.rawOutInt);
}

TEST(sf, Cmif_RefCount)
{
    {
        Foo foo;
        SharedPointer<IFoo> p(&foo, false);
        MY_EXPECT_EQ(1, foo.refCount);
        // 実装用コンストラクタ
        CmifServerObjectInfo info(std::move(p));
        MY_EXPECT_EQ(1, foo.refCount);
        ASSERT_TRUE(static_cast<bool>(info));
        // ムーブコンストラクタ
        CmifServerObjectInfo info2(std::move(info));
        ASSERT_TRUE(!info);
        ASSERT_TRUE(static_cast<bool>(info2));
        MY_EXPECT_EQ(1, foo.refCount);
        // ムーブ代入
        CmifServerObjectInfo info3;
        ASSERT_TRUE(!info3);
        info3 = std::move(info2);
        ASSERT_TRUE(!info2);
        ASSERT_TRUE(static_cast<bool>(info3));
        MY_EXPECT_EQ(1, foo.refCount);
        // メンバ swap
        info3.swap(info2);
        ASSERT_TRUE(static_cast<bool>(info2));
        ASSERT_TRUE(!info3);
        MY_EXPECT_EQ(1, foo.refCount);
        // 外部 swap
        swap(info2, info);
        ASSERT_TRUE(static_cast<bool>(info));
        ASSERT_TRUE(!info2);
        MY_EXPECT_EQ(1, foo.refCount);
        // Reset
        info.Reset();
        ASSERT_TRUE(!info);
        MY_EXPECT_EQ(0, foo.refCount);
    }
    {
        // Clone のテスト
        Foo foo;
        SharedPointer<IFoo> p(&foo, false);
        MY_EXPECT_EQ(1, foo.refCount);
        // 実装用コンストラクタ
        CmifServerObjectInfo info(std::move(p));
        MY_EXPECT_EQ(1, foo.refCount);
        ASSERT_TRUE(static_cast<bool>(info));
        // Clone
        CmifServerObjectInfo info2(info.Clone());
        MY_EXPECT_EQ(2, foo.refCount);
        info.Reset();
        MY_EXPECT_EQ(1, foo.refCount);
        info = info2.Clone();
        MY_EXPECT_EQ(2, foo.refCount);
        {
            CmifServerObjectInfo info3(info.Clone());
            MY_EXPECT_EQ(3, foo.refCount);
        }
        MY_EXPECT_EQ(2, foo.refCount);
    }
    {
        // デストラクタ
        Foo foo;
        SharedPointer<IFoo> p(&foo, false);
        MY_EXPECT_EQ(1, foo.refCount);
        {
            CmifServerObjectInfo info(std::move(p));
            MY_EXPECT_EQ(1, foo.refCount);
        }
        MY_EXPECT_EQ(0, foo.refCount);
    }
}
