﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 設定
#define TEST_DFC
#define TEST_HIPC_PARALELL
#define TEST_HIPC_PARALELL_WITH_DOMAIN
const int RepeatCount = 10;

#include <nnt/nntest.h>
#define MY_EXPECT_EQ(a, b) EXPECT_TRUE((a) == (b))

// テスト対象
#include "testSf_IAllFunctionTests.sfdl.h"
#include "testSf_AllFunctionTestsImpl.h"

// hipc
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_HipcServerSessionManagerHandler.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <type_traits>
#include <new>
#include <map>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nnt/sfutil/sfutil_MultiWaitUtils.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/sf/cmif/sf_InlineContext.h>
#include <nn/sf/sf_FsInlineContext.h>

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#include <nn/svc/svc_Base.h>
#endif

namespace {

struct MyAllocatorTag
{
};

typedef nn::sf::ExpHeapStaticAllocator<1024 * 1024, MyAllocatorTag> MyAllocator;

class MyAllocatorInitializer
{
public:
    MyAllocatorInitializer() NN_NOEXCEPT
    {
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyAllocatorInitializer;

void TestMethods0(nn::sf::SharedPointer<nnt::testsf::IAllFunctionTestsBase0> p)
{
    {
        for (auto i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(i, p->EchoInt(i));
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            p->StoreInt(i);
            MY_EXPECT_EQ(i, p->LoadInt());
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            auto result = p->StoreIntWithResult(i);
            ASSERT_TRUE(result.IsSuccess());
            p->StoreInt(i);
            MY_EXPECT_EQ(i, p->LoadInt());
        }
    }
    {
        const auto size = 100;
        char src[size];
        for (auto i = 0; i < size; ++i)
        {
            src[i] = static_cast<char>(i);
        }
        char dst[size];
        p->FlipBits(
            nn::sf::OutBuffer(reinterpret_cast<char*>(dst), size),
            nn::sf::InBuffer(reinterpret_cast<const char*>(src), size)
        );
        for (auto i = 0; i < size; ++i)
        {
            MY_EXPECT_EQ(static_cast<char>(~src[i]), dst[i]);
        }
    }
}

void TestMethods(nn::sf::SharedPointer<nnt::testsf::IAllFunctionTestsBase> p, bool testObjects = false)
{
    TestMethods0(p);
    {
        for (auto i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(i, p->EchoInt(i));
        }
    }
    {
        MY_EXPECT_EQ(false, p->EchoBool(false));
        MY_EXPECT_EQ(true, p->EchoBool(true));
    }
    {
        for (uint16_t i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(i, p->EchoUInt16(i));
        }
    }
    {
        for (uint64_t i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(i, p->EchoUInt64(i));
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            p->StoreInt(i);
            MY_EXPECT_EQ(i, p->LoadInt());
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            auto result = p->StoreIntWithResult(i);
            ASSERT_TRUE(result.IsSuccess());
            p->StoreInt(i);
            MY_EXPECT_EQ(i, p->LoadInt());
        }
    }
    {
        const auto size = 100;
        char src[size];
        for (auto i = 0; i < size; ++i)
        {
            src[i] = static_cast<char>(i);
        }
        char dst[size];
        p->FlipBits(
            nn::sf::OutBuffer(reinterpret_cast<char*>(dst), size),
            nn::sf::InBuffer(reinterpret_cast<const char*>(src), size)
        );
        for (auto i = 0; i < size; ++i)
        {
            MY_EXPECT_EQ(static_cast<char>(~src[i]), dst[i]);
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            nnt::testsf::AddParameter addValue = { i, i };
            MY_EXPECT_EQ(i * 2, p->Add(addValue));
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            p->StoreInt(i);
            int n;
            p->LoadIntByOut(&n);
            MY_EXPECT_EQ(i, n);
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            nnt::testsf::Struct1 in;
            in.b = i % 2 == 0;
            in.addParameter.a = static_cast<std::int32_t>(i * 1);
            in.addParameter.b = static_cast<std::int32_t>(i * 2);
            in.c = static_cast<char>(i * 3);
            in.c16 = static_cast<char16_t>(i * 4);
            in.c32 = static_cast<char32_t>(i * 5);
            in.i8 = static_cast<std::int8_t>(i * 6);
            in.i16 = static_cast<std::int16_t>(i * 7);
            in.i32 = static_cast<std::int32_t>(i * 8);
            in.i64 = static_cast<std::int64_t>(i * 9);
            auto out = p->EchoStruct(in);
            EXPECT_EQ(in.addParameter.a, out.addParameter.a);
            EXPECT_EQ(in.addParameter.b, out.addParameter.b);
            EXPECT_EQ(in.b, out.b);
            EXPECT_EQ(in.c, out.c);
            EXPECT_EQ(in.c16, out.c16);
            EXPECT_EQ(in.c32, out.c32);
            EXPECT_EQ(in.i8, out.i8);
            EXPECT_EQ(in.i16, out.i16);
            EXPECT_EQ(in.i32, out.i32);
            EXPECT_EQ(in.i64, out.i64);
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            nnt::testsf::Struct1 in;
            in.b = i % 2 == 0;
            in.addParameter.a = static_cast<std::int32_t>(i * 1);
            in.addParameter.b = static_cast<std::int32_t>(i * 2);
            in.c = static_cast<char>(i * 3);
            in.c16 = static_cast<char16_t>(i * 4);
            in.c32 = static_cast<char32_t>(i * 5);
            in.i8 = static_cast<std::int8_t>(i * 6);
            in.i16 = static_cast<std::int16_t>(i * 7);
            in.i32 = static_cast<std::int32_t>(i * 8);
            in.i64 = static_cast<std::int64_t>(i * 9);
            nnt::testsf::Struct2 out;
            p->EchoStruct2(&out, in);
            EXPECT_EQ(in.addParameter.a, out.addParameter.a);
            EXPECT_EQ(in.addParameter.b, out.addParameter.b);
            EXPECT_EQ(in.b, out.b);
            EXPECT_EQ(in.c, out.c);
            EXPECT_EQ(in.c16, out.c16);
            EXPECT_EQ(in.c32, out.c32);
            EXPECT_EQ(in.i8, out.i8);
            EXPECT_EQ(in.i16, out.i16);
            EXPECT_EQ(in.i32, out.i32);
            EXPECT_EQ(in.i64, out.i64);
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            nnt::testsf::Struct1 in;
            in.b = i % 2 == 0;
            in.addParameter.a = static_cast<std::int32_t>(i * 1);
            in.addParameter.b = static_cast<std::int32_t>(i * 2);
            in.c = static_cast<char>(i * 3);
            in.c16 = static_cast<char16_t>(i * 4);
            in.c32 = static_cast<char32_t>(i * 5);
            in.i8 = static_cast<std::int8_t>(i * 6);
            in.i16 = static_cast<std::int16_t>(i * 7);
            in.i32 = static_cast<std::int32_t>(i * 8);
            in.i64 = static_cast<std::int64_t>(i * 9);
            nnt::testsf::Struct2 out;
            p->EchoStruct3(&out, in);
            EXPECT_EQ(in.addParameter.a, out.addParameter.a);
            EXPECT_EQ(in.addParameter.b, out.addParameter.b);
            EXPECT_EQ(in.b, out.b);
            EXPECT_EQ(in.c, out.c);
            EXPECT_EQ(in.c16, out.c16);
            EXPECT_EQ(in.c32, out.c32);
            EXPECT_EQ(in.i8, out.i8);
            EXPECT_EQ(in.i16, out.i16);
            EXPECT_EQ(in.i32, out.i32);
            EXPECT_EQ(in.i64, out.i64);
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            nnt::testsf::Struct1 in;
            in.b = i % 2 == 0;
            in.addParameter.a = static_cast<std::int32_t>(i * 1);
            in.addParameter.b = static_cast<std::int32_t>(i * 2);
            in.c = static_cast<char>(i * 3);
            in.c16 = static_cast<char16_t>(i * 4);
            in.c32 = static_cast<char32_t>(i * 5);
            in.i8 = static_cast<std::int8_t>(i * 6);
            in.i16 = static_cast<std::int16_t>(i * 7);
            in.i32 = static_cast<std::int32_t>(i * 8);
            in.i64 = static_cast<std::int64_t>(i * 9);
            nnt::testsf::Struct2 out;
            p->EchoStruct4(&out, in);
            EXPECT_EQ(in.addParameter.a, out.addParameter.a);
            EXPECT_EQ(in.addParameter.b, out.addParameter.b);
            EXPECT_EQ(in.b, out.b);
            EXPECT_EQ(in.c, out.c);
            EXPECT_EQ(in.c16, out.c16);
            EXPECT_EQ(in.c32, out.c32);
            EXPECT_EQ(in.i8, out.i8);
            EXPECT_EQ(in.i16, out.i16);
            EXPECT_EQ(in.i32, out.i32);
            EXPECT_EQ(in.i64, out.i64);
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            int32_t src[100] = {};
            int32_t dst[100] = {};
            for (auto j = 0; j < i; ++j)
            {
                src[j] = j;
            }
            p->SquareArrayElement(nn::sf::OutArray<int32_t>(dst, i), nn::sf::InArray<int32_t>(src, i));
            for (auto j = 0; j < i; ++j)
            {
                EXPECT_EQ(dst[j], j * j);
            }
            for (auto j = i; j < 100; ++j)
            {
                EXPECT_EQ(dst[j], 0);
            }
        }
    }
    {
        for (auto i = 0; i < 100; ++i)
        {
            int32_t src[100] = {};
            int32_t dst[100] = {};
            for (auto j = 0; j < i; ++j)
            {
                src[j] = j;
            }
            p->SquareArrayElement2(nn::sf::OutArray<int32_t>(dst, i), nn::sf::InArray<int32_t>(src, i));
            for (auto j = 0; j < i; ++j)
            {
                EXPECT_EQ(dst[j], j * j);
            }
            for (auto j = i; j < 100; ++j)
            {
                EXPECT_EQ(dst[j], 0);
            }
        }
    }
    if (testObjects)
    {
        const int N = 5;
        decltype(p->OpenSession(0)) pointers[N];
        for (int i = 0; i < N; ++i)
        {
            EXPECT_EQ(i, p->GetSessionCount());
            pointers[i] = p->OpenSession(i);
        }
        for (int i = 0; i < N; ++i)
        {
            EXPECT_EQ(i, pointers[i]->GetValue());
        }
    }
    EXPECT_EQ(0, p->GetSessionCount());
    if (testObjects)
    {
        const int N = 5;
        decltype(p->OpenSession(0)) pointers[N];
        for (int i = 0; i < N; ++i)
        {
            EXPECT_EQ(i, p->GetSessionCount());
            p->OpenSession2(pointers + i, i);
        }
        for (int i = 0; i < N; ++i)
        {
            EXPECT_EQ(i, pointers[i]->GetValue());
        }
    }
    EXPECT_EQ(0, p->GetSessionCount());
    if (testObjects)
    {
        const int N = 5;
        decltype(p->OpenSession(0)) pointers[N];
        for (char i = 0; i < N; ++i)
        {
            EXPECT_EQ(i, p->GetSessionCount());
            char j;
            p->OpenSessionChar(pointers + i, &j, i);
            EXPECT_EQ(i, j);
        }
        for (int i = 0; i < N; ++i)
        {
            EXPECT_EQ(i, pointers[i]->GetValue());
        }
    }
    EXPECT_EQ(0, p->GetSessionCount());
    if (testObjects)
    {
        const int N = 100;
        for (int i = 0; i < N; ++i)
        {
            EXPECT_FALSE(p->GetNullObject());

            decltype(p->GetNullObject()) ptr;
            p->GetNullObject2(&ptr);
            EXPECT_FALSE(ptr);
        }
    }
    for (int i = 0; i < 100; ++i)
    {
        const auto size = 100;
        std::uint8_t src1[size];
        std::uint8_t src2[size];
        std::uint8_t sum[size];
        std::uint8_t sub[size];
        auto result = p->CauseError(
            nn::sf::OutBuffer(reinterpret_cast<char*>(sum), size),
            nn::sf::OutBuffer(reinterpret_cast<char*>(sub), size),
            nn::sf::InBuffer(reinterpret_cast<const char*>(src1), size),
            nn::sf::InBuffer(reinterpret_cast<const char*>(src2), size)
        );
        EXPECT_TRUE(result <= nnt::testsf::ResultSfTestError());
    }
    {
        auto q = static_cast<nnt::testsf::IAllFunctionTests*>(p.Get());
        for (auto i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(i, q->EchoIntInherited(i));
        }
    }
    {
        nn::sf::cmif::SetInlineContext(0);
        for (uint32_t i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(0, nn::sf::cmif::GetInlineContext());
            nn::sf::cmif::ScopedInlineContextChanger sicc{i};
            MY_EXPECT_EQ(i, nn::sf::cmif::GetInlineContext());
            MY_EXPECT_EQ(i, p->GetInlineContextTest());
        }
    }
    {
        const nn::Bit32 BaseValue = 0x12345600;
        nn::sf::cmif::SetInlineContext(BaseValue);
        for (uint8_t i = 0; i < 100; ++i)
        {
            MY_EXPECT_EQ(BaseValue, nn::sf::cmif::GetInlineContext());
            nn::sf::SetFsInlineContext(i);
            MY_EXPECT_EQ(i | BaseValue, nn::sf::cmif::GetInlineContext());
            MY_EXPECT_EQ(i, p->GetFsInlineContextTest());
            nn::sf::SetFsInlineContext(0);
        }
    }
} // NOLINT(readability/fn_size)

void TestSumAndSubBytesImpl(nn::sf::SharedPointer<nnt::testsf::IAllFunctionTestsBase> p, int maxSize, int stride, int32_t (nnt::testsf::IAllFunctionTestsBase::*f)(const nn::sf::OutBuffer&, const nn::sf::OutBuffer&, const nn::sf::InBuffer&, const nn::sf::InBuffer&))
{
    std::unique_ptr<std::uint8_t[]> src1(new uint8_t[maxSize]);
    std::unique_ptr<std::uint8_t[]> src2(new uint8_t[maxSize]);
    std::unique_ptr<std::uint8_t[]> sum(new uint8_t[maxSize]);
    std::unique_ptr<std::uint8_t[]> sub(new uint8_t[maxSize]);
    for (auto size = 0; size < maxSize; size += stride)
    {
        std::memset(src1.get(), 0, maxSize);
        std::memset(src2.get(), 0, maxSize);
        std::memset(sum.get(), 0, maxSize);
        std::memset(sub.get(), 0, maxSize);
        for (auto i = 0; i < size; ++i)
        {
            src1[i] = static_cast<char>(i);
            src2[i] = static_cast<char>(i * i);
        }
        auto n = (p.Get()->*f)(
            nn::sf::OutBuffer(reinterpret_cast<char*>(sum.get()), size),
            nn::sf::OutBuffer(reinterpret_cast<char*>(sub.get()), size),
            nn::sf::InBuffer(reinterpret_cast<const char*>(src1.get()), size),
            nn::sf::InBuffer(reinterpret_cast<const char*>(src2.get()), size)
        );
        MY_EXPECT_EQ(size, n);
        for (auto i = 0; i < size; ++i)
        {
            EXPECT_EQ(static_cast<std::uint8_t>(src1[i] + src2[i]), sum[i]);
            EXPECT_EQ(static_cast<std::uint8_t>(src1[i] - src2[i]), sub[i]);
        }
    }
}

void TestSumAndSubBytes(nn::sf::SharedPointer<nnt::testsf::IAllFunctionTestsBase> p)
{
    // ※コメントアウトしているものはポインタ転送バッファが足りないもの
    // SumAndSubBytes:  全てマップ転送
    // SumAndSubBytes2: マップ転送 2, ポインタ転送 2
    // SumAndSubBytes3: 全て自動選択
    // SumAndSubBytes4: マップ転送 1, ポインタ転送 1, 自動選択 2
    {
        auto const MaxSize = 1024 * 16;
        auto const Stride = 257;
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes2);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes3);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes4);
    }
    {
        auto const MaxSize = 1024 * 32;
        auto const Stride = 2029;
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes);
        //TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes2);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes3);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes4);
    }
    {
        auto const MaxSize = 1024 * 128;
        auto const Stride = 10831;
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes);
        //TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes2);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes3);
        //TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes4);
    }
    {
        auto const MaxSize = 1024 * 1024;
        auto const Stride = 292133;
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes);
        //TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes2);
        TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes3);
        //TestSumAndSubBytesImpl(p, MaxSize, Stride, &nnt::testsf::IAllFunctionTestsBase::SumAndSubBytes4);
    }
}

void TestNativeHandleMethods(
    const nn::sf::SharedPointer<nnt::testsf::IAllFunctionTests>& p,      // 対象のオブジェクト
    const nn::sf::SharedPointer<nnt::testsf::IAllFunctionTests>& pInner, // 対象のオブジェクトの内部実装
    bool handleManagement
)
{
    // ReadableHandle を Get で取得し、e2 には false でアタッチするテスト
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto readableHandle = e.GetReadableHandle();
        p->InNativeHandle(nn::sf::NativeHandle(readableHandle, false));

        // コピーでの取得であるため、複数回の取得が可能であるはず
        for (int i = 0; i < 10; ++i)
        {
            nn::sf::NativeHandle handle;
            p->GetNativeHandle2(&handle);
            EXPECT_EQ(handleManagement, handle.IsManaged());
            {
                nn::os::SystemEvent e2;
                e2.AttachReadableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

                e.Clear();
                ASSERT_TRUE(!e2.TryWait());
                e.Signal();
                ASSERT_TRUE(e2.TryWait());
            }
        }
    }

    // ReadableHandle を Detach で取得し、e2 には false でアタッチするテスト
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto readableHandle = e.DetachReadableHandle();
        p->InNativeHandle(nn::sf::NativeHandle(readableHandle, true));

        // コピーでの取得であるため、複数回の取得が可能であるはず
        for (int i = 0; i < 10; ++i)
        {
            nn::sf::NativeHandle handle;
            p->GetNativeHandle2(&handle);
            EXPECT_EQ(handleManagement, handle.IsManaged());
            {
                nn::os::SystemEvent e2;
                e2.AttachReadableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

                e.Clear();
                ASSERT_TRUE(!e2.TryWait());
                e.Signal();
                ASSERT_TRUE(e2.TryWait());
            }
        }
    }

    // move in は HIPC で使用できなくなったため、テストしない
    #if 0
    // ReadableHandle を Detach で取得して move で送り、move で取得するテスト
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto readableHandle = e.DetachReadableHandle();
        p->MoveInNativeHandle(nn::sf::NativeHandle(readableHandle, true));

        nn::sf::NativeHandle handle;
        p->MoveNativeHandle2(&handle);
        EXPECT_EQ(true, handle.IsManaged());
        {
            nn::os::SystemEvent e2;
            e2.AttachReadableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

            ASSERT_TRUE(!e2.TryWait());
            e.Signal();
            ASSERT_TRUE(e2.TryWait());
        }

        // move で取得しているので、2 回目は invalid が返ることを確認
        nn::sf::NativeHandle handle2;
        p->GetNativeHandle2(&handle2);
        EXPECT_EQ(nn::os::InvalidNativeHandle, handle2.GetOsHandle());
    }

    // ReadableHandle を Detach で取得して move で送り、move で取得するテスト (MoveOut 版)
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto readableHandle = e.DetachReadableHandle();
        p->MoveInNativeHandle(nn::sf::NativeHandle(readableHandle, true));

        nn::sf::NativeHandle handle;
        p->MoveNativeHandle(&handle);
        EXPECT_EQ(true, handle.IsManaged());
        {
            nn::os::SystemEvent e2;
            e2.AttachReadableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

            ASSERT_TRUE(!e2.TryWait());
            e.Signal();
            ASSERT_TRUE(e2.TryWait());
        }

        // move で取得しているので、2 回目は invalid が返ることを確認
        nn::sf::NativeHandle handle2;
        p->GetNativeHandle2(&handle2);
        EXPECT_EQ(nn::os::InvalidNativeHandle, handle2.GetOsHandle());
    }

    // ReadableHandle を Detach で取得して move で送り、copy で取得するテスト
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto readableHandle = e.DetachReadableHandle();
        p->MoveInNativeHandle(nn::sf::NativeHandle(readableHandle, true));

        // コピーでの取得であるため、複数回の取得が可能であるはず
        for (int i = 0; i < 10; ++i)
        {
            nn::sf::NativeHandle handle;
            p->GetNativeHandle2(&handle);
            EXPECT_EQ(handleManagement, handle.IsManaged());
            {
                nn::os::SystemEvent e2;
                e2.AttachReadableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

                e.Clear();
                ASSERT_TRUE(!e2.TryWait());
                e.Signal();
                ASSERT_TRUE(e2.TryWait());
            }
        }
    }
    #endif

    // WritableHandle を Get で取得し、e2 には false でアタッチするテスト
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto writableHandle = e.GetWritableHandle();
        p->InNativeHandle(nn::sf::NativeHandle(writableHandle, false));

        nn::sf::NativeHandle handle;
        p->GetNativeHandle2(&handle);
        {
            nn::os::SystemEvent e2;
            e2.AttachWritableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

            ASSERT_TRUE(!e.TryWait());
            e2.Signal();
            ASSERT_TRUE(e.TryWait());
        }
    }

    // WritableHandle を Detach で取得し、e2 には false でアタッチするテスト
    {
        nn::os::SystemEvent e(nn::os::EventClearMode_ManualClear, true);
        auto writableHandle = e.DetachWritableHandle();
        p->InNativeHandle(nn::sf::NativeHandle(writableHandle, true));

        nn::sf::NativeHandle handle;
        p->GetNativeHandle2(&handle);
        {
            nn::os::SystemEvent e2;
            e2.AttachWritableHandle(handle.GetOsHandle(), false, nn::os::EventClearMode_ManualClear);

            ASSERT_TRUE(!e.TryWait());
            e2.Signal();
            ASSERT_TRUE(e.TryWait());
        }
    }

    NN_UNUSED(pInner);
}   // NOLINT(readability/fn_size)

void TestClientProcessId(
    const nn::sf::SharedPointer<nnt::testsf::IAllFunctionTests>& p, bool useActualPid = false)
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN32)
    NN_UNUSED(p);
    NN_UNUSED(useActualPid);
    return;

#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    const int N = 5;
    for (int i = 0; i < N; i++)
    {
        nn::Bit64 processId;
        if (useActualPid)
        {
            ASSERT_TRUE(nn::svc::GetProcessId(&processId, static_cast<nn::svc::Handle>(0xFFFF8001)).IsSuccess());
            // いかなる値を入力しても、実際の PID で上書きされることのテスト
            p->SetClientProcessId(static_cast<nn::Bit64>(i));
        }
        else
        {
            // 入力した値が何であれ変更されないことのテスト
            processId = static_cast<nn::Bit64>(i);
            p->SetClientProcessId(processId);
        }

        nn::Bit64 outProcessId;
        p->GetClientProcessId(&outProcessId);
        EXPECT_EQ(processId, outProcessId);

        // ARUID チェック
        EXPECT_TRUE(p->CheckAruid(processId).IsSuccess());
        EXPECT_TRUE(p->CheckAruid(0).IsSuccess());
        EXPECT_TRUE(p->CheckAruid2({processId}).IsSuccess());
        EXPECT_TRUE(p->CheckAruid2({0}).IsSuccess());

        if (useActualPid)
        {
            // DFC のときはチェックがないためテストしない
            EXPECT_TRUE(!p->CheckAruid(processId + 1).IsSuccess());
            EXPECT_TRUE(!p->CheckAruid2({processId + 1}).IsSuccess());
        }
    }

#else
    #error "サポートされない OS 種別が指定されています。"
#endif
}

// DFC

#ifdef TEST_DFC

TEST(sf_AllInOne, Services_Dfc)
{
    for (int i = 0; i < 10; ++i)
    {
        auto p = nnt::testsf::CreateAllFunctionTests(0);
        TestMethods(p, true);
        if (i == 0)
        {
            TestSumAndSubBytes(p);
        }
        TestNativeHandleMethods(p, p, false);
        TestClientProcessId(p);
    }
}

#endif // TEST_DFC

// HIPC 並列

#if defined(TEST_HIPC_PARALELL) || defined(TEST_HIPC_PARALELL_WITH_DOMAIN)

class MyServerSessionManagerHandler
    : public nn::sf::IHipcServerSessionManagerHandler
{
private:

    mutable nn::os::Mutex m_Mutex{false};
    std::map<nn::Bit64, uint64_t> m_Map;

    virtual nn::Result BeforeInvoke() NN_NOEXCEPT NN_OVERRIDE
    {
        auto clientId = nn::sf::GetCurrentHipcClientIdOnServer();
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        EXPECT_TRUE(m_Map[clientId] > 0);
        NN_RESULT_SUCCESS;
    }

    virtual void AfterInvoke() NN_NOEXCEPT NN_OVERRIDE
    {
        auto clientId = nn::sf::GetCurrentHipcClientIdOnServer();
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        EXPECT_TRUE(m_Map[clientId] > 0);
    }

    virtual void OnAddClient(nn::Bit64 clientId) NN_NOEXCEPT NN_OVERRIDE
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        auto&& count = m_Map[clientId];
        ++count;
    }

    virtual void OnRemoveClient(nn::Bit64 clientId) NN_NOEXCEPT NN_OVERRIDE
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        auto&& it = m_Map.find(clientId);
        auto&& count = it->second;
        EXPECT_TRUE(count > 0);
        if (count > 1)
        {
            --count;
        }
        else
        {
            m_Map.erase(it);
        }
    }

public:

    void AssertZero() NN_NOEXCEPT
    {
        EXPECT_EQ(0u, m_Map.size());
    }

};

MyServerSessionManagerHandler g_MyServerSessionManagerHandler;

const int ServerThreadCount = 3;
const size_t ServerStackSize = nn::os::StackRegionAlignment * 4;
NN_ALIGNAS(4096) nn::Bit8 g_SreverThreadStack[ServerStackSize * ServerThreadCount];

struct MyOption
{
    static const size_t PointerTransferBufferSize = 32 * 1024;
    static const bool CanDeferInvokeRequest = true;
    static const int SubDomainCountMax = 10;
    static const int ObjectInSubDomainCountMax = 100;
};

class MyHipcParallelServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<100, 1, MyOption>
{
private:

    nnt::sfutil::TestThread m_Threads[ServerThreadCount];
    int m_NumStartThreads;

    nn::os::Event m_TerminationEvent;
    nnt::sfutil::MultiWaitHodler m_TerminationEventHolder;
    int m_ProcessingThreadCount;

    virtual nn::Result OnNeedsToAccept(int portIndex, PortForAllInOne* pPort) NN_NOEXCEPT NN_OVERRIDE final
    {
        EXPECT_EQ(portIndex, 123);
        return this->AcceptImpl(pPort, nnt::testsf::CreateAllFunctionTests(0));
    }

    nn::Result ProcessMyInvocation(nn::os::MultiWaitHolderType* p) NN_NOEXCEPT
    {
        NN_RESULT_TRY(this->ProcessInvokeRequestWithDefer(p))
            NN_RESULT_CATCH(nn::sf::ResultProcessDeferred)
            {
                // OnceInvalidContext の呼び出しは一回失敗することになっているので、再度行う。
                auto result = this->ProcessInvokeRequestWithDefer(p);
                EXPECT_TRUE(result.IsSuccess());
            }
        NN_RESULT_END_TRY
        NN_RESULT_SUCCESS;
    }

    void ThreadFunctionImpl() NN_NOEXCEPT
    {
        while (auto p = Wait())
        {
            if (p == m_TerminationEventHolder.GetBase())
            {
                // シグナルされたものがユーザカスタム終了リクエストだった場合には、acquire して、
                m_TerminationEvent.Wait();
                --this->m_ProcessingThreadCount;
                if (m_ProcessingThreadCount > 0)
                {
                    // まだシグナルを受けるものがいる場合には、再登録する
                    AddUserWaitHolder(p);
                }
                // ループを抜ける
                return;
            }
            // それ以外のときはセッションやポートとして扱う
            switch (nn::os::GetMultiWaitHolderUserData(p))
            {
                case MyHipcParallelServerManager::InvokeTag:
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessMyInvocation(p));
                    continue;
                }
                default:
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(ProcessAuto(p));
                    continue;
                }
            }
        }
    }

    static void ThreadFunction(MyHipcParallelServerManager* pManager) NN_NOEXCEPT
    {
        pManager->SetManagerHandler(&g_MyServerSessionManagerHandler);
        pManager->ThreadFunctionImpl();
    }

public:

    explicit MyHipcParallelServerManager(const char* serviceName) NN_NOEXCEPT
        : m_NumStartThreads(0)
        , m_TerminationEvent(nn::os::EventClearMode_ManualClear)
        , m_TerminationEventHolder(m_TerminationEvent.GetBase())
    {
        this->AddUserWaitHolder(m_TerminationEventHolder.GetBase());
        NN_ABORT_UNLESS_RESULT_SUCCESS(this->InitializePort(123, 10, serviceName));
        this->Start();
    }

    void StartThreads(int numThreads) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(numThreads <= ServerThreadCount);
        this->m_ProcessingThreadCount = numThreads;
        for (int i = 0; i < numThreads; ++i)
        {
            auto&& thread = m_Threads[i];
            thread.Initialize(ThreadFunction, this, &g_SreverThreadStack[i * ServerStackSize], ServerStackSize, 10);
            thread.Start();
        }
        this->m_NumStartThreads = numThreads;
    }

    ~MyHipcParallelServerManager() NN_NOEXCEPT
    {
        // ユーザカスタム終了リクエストの発行
        m_TerminationEvent.Signal();
        for (int i = 0; i < m_NumStartThreads; ++i)
        {
            auto&& thread = m_Threads[i];
            thread.Wait();
        }
        EXPECT_EQ(0, m_ProcessingThreadCount);
        this->RequestStop();
    }

};

const int ClientThreadCount = 4;
const size_t ClientStackSize = nn::os::StackRegionAlignment * 4;
NN_ALIGNAS(4096) nn::Bit8 g_ClientThreadStack[ClientStackSize * ClientThreadCount];
const char ServiceName[] = "sf_test0";

// サイズ的にスタックに置けないため、グローバルに置く
// 再利用のために aligned_storage とする
std::aligned_storage<sizeof(MyHipcParallelServerManager), NN_ALIGNOF(MyHipcParallelServerManager)>::type g_MyHipcParallelServerManagerStorage;

#endif

#ifdef TEST_HIPC_PARALELL

nn::sf::SharedPointer<nnt::testsf::IAllFunctionTests> GetHipcProxyObjectByName(const char* serviceName) NN_NOEXCEPT
{
    NN_SF_RESULT_AND_TRY(ret, (nn::sf::CreateHipcProxyByName<nnt::testsf::IAllFunctionTests, nn::sf::DefaultAllocationPolicy>(serviceName)))
        NN_RESULT_CATCH_ALL
        {
            EXPECT_TRUE(false);
        }
    NN_SF_RESULT_AND_END_TRY
    return ret;
}

void TestHipcParallel(int serverThreadCount, int clientThreadCount)
{
    nn::sf::InitializeForHipcServerSessionManagerHandler();
    auto count = 0;
    for (int n = 0; n < RepeatCount; ++n)
    {
        auto pManager = new (&g_MyHipcParallelServerManagerStorage) MyHipcParallelServerManager(ServiceName);
        NN_UTIL_SCOPE_EXIT
        {
            pManager->~MyHipcParallelServerManager();
        };
        pManager->StartThreads(serverThreadCount);

        {
            for (int i = 0; i < 10; ++i)
            {
                // オブジェクトのプロキシを取得してテスト
                void (*withTestSumAndSubBytes)() = []
                {
                    auto p = GetHipcProxyObjectByName(ServiceName);
                    TestMethods(p, true);
                    TestSumAndSubBytes(p);
                    TestNativeHandleMethods(p, p, true);
                    TestClientProcessId(p, true);
                    EXPECT_TRUE(p->DeferProcess().IsSuccess());
                };
                void (*withoutTestSumAndSubBytes)() = []
                {
                    auto p = GetHipcProxyObjectByName(ServiceName);
                    TestMethods(p, true);
                    TestNativeHandleMethods(p, p, true);
                    TestClientProcessId(p, true);
                    EXPECT_TRUE(p->DeferProcess().IsSuccess());
                };
                auto f = count == 0 ? withTestSumAndSubBytes : withoutTestSumAndSubBytes; // 一つ目のクライアントだけ TestSumAndSubBytes を行う
                ++count;
                nnt::sfutil::TestThread threads[ClientThreadCount];
                for (int j = 0; j < clientThreadCount; ++j)
                {
                    auto&& thread = threads[j];
                    thread.Initialize(f, &g_ClientThreadStack[j * ClientStackSize], ClientStackSize, 10);
                    thread.Start();
                }
                for (int j = 0; j < clientThreadCount; ++j)
                {
                    auto&& thread = threads[j];
                    thread.Wait();
                }
            }
        }
    }
    g_MyServerSessionManagerHandler.AssertZero();
}

TEST(sf_AllInOne, Services_Hipc_Server1_Client1)
{
    TestHipcParallel(1, 1);
}

TEST(sf_AllInOne, Services_Hipc_Server3_Client1)
{
    TestHipcParallel(3, 1);
}

TEST(sf_AllInOne, Services_Hipc_Server1_Client4)
{
    TestHipcParallel(1, 4);
}

TEST(sf_AllInOne, Services_Hipc_Server3_Client4)
{
    TestHipcParallel(3, 4);
}

#endif // TEST_HIPC_PARALELL

#ifdef TEST_HIPC_PARALELL_WITH_DOMAIN

nn::sf::SharedPointer<nnt::testsf::IAllFunctionTests> GetHipcProxyObjectByName(nn::sf::HipcSimpleClientSessionManager* pManager, const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nnt::testsf::IAllFunctionTests> ret;
    auto result = pManager->InitializeByName<nnt::testsf::IAllFunctionTests, MyAllocator::Policy>(&ret, serviceName);
    EXPECT_TRUE(result.IsSuccess());
    return ret;
}

void TestHipcParallelWithDomain(int serverThreadCount, int clientThreadCount)
{
    auto count = 0;
    for (int n = 0; n < RepeatCount; ++n)
    {
        auto pManager = new (&g_MyHipcParallelServerManagerStorage) MyHipcParallelServerManager(ServiceName);
        NN_UTIL_SCOPE_EXIT
        {
            pManager->~MyHipcParallelServerManager();
        };
        pManager->StartThreads(serverThreadCount);

        {
            for (int i = 0; i < 10; ++i)
            {
                // オブジェクトのプロキシを取得してテスト
                void (*withTestSumAndSubBytes)() = []
                {
                    nn::sf::HipcSimpleClientSessionManager m;
                    auto p = GetHipcProxyObjectByName(&m, ServiceName);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(m.IncreaseSession());
                    TestMethods(p, true);
                    TestSumAndSubBytes(p);
                    TestNativeHandleMethods(p, p, true);
                    TestClientProcessId(p, true);
                    EXPECT_TRUE(p->DeferProcess().IsSuccess());
                };
                void (*withoutTestSumAndSubBytes)() = []
                {
                    nn::sf::HipcSimpleClientSessionManager m;
                    auto p = GetHipcProxyObjectByName(&m, ServiceName);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(m.IncreaseSession());
                    TestMethods(p, true);
                    TestNativeHandleMethods(p, p, true);
                    TestClientProcessId(p, true);
                    EXPECT_TRUE(p->DeferProcess().IsSuccess());
                };
                auto f = count == 0 ? withTestSumAndSubBytes : withoutTestSumAndSubBytes; // 一つ目のクライアントだけ TestSumAndSubBytes を行う
                ++count;
                nnt::sfutil::TestThread threads[ClientThreadCount];
                for (int j = 0; j < clientThreadCount; ++j)
                {
                    auto&& thread = threads[j];
                    thread.Initialize(f, &g_ClientThreadStack[j * ClientStackSize], ClientStackSize, 10);
                    thread.Start();
                }
                for (int j = 0; j < clientThreadCount; ++j)
                {
                    auto&& thread = threads[j];
                    thread.Wait();
                }
            }
        }
    }
    g_MyServerSessionManagerHandler.AssertZero();
}

TEST(sf_AllInOne, Services_Hipc_Server1_Client1_WithDomain)
{
    TestHipcParallelWithDomain(1, 1);
}

TEST(sf_AllInOne, Services_Hipc_Server3_Client1_WithDomain)
{
    TestHipcParallelWithDomain(3, 1);
}

TEST(sf_AllInOne, Services_Hipc_Server1_Client4_WithDomain)
{
    TestHipcParallelWithDomain(1, 4);
}

TEST(sf_AllInOne, Services_Hipc_Server3_Client4_WithDomain)
{
    TestHipcParallelWithDomain(3, 4);
}

#endif // TEST_HIPC_PARALELL_WITH_DOMAIN

}
