﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include <nn/sf/sf_ObjectFactory.h>

// SF インターフェイス
#include "testSf_IMultiTests.h"

// 実装
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <memory>

namespace {

class MyStatelessAllocatorTag
{
};

typedef nn::sf::impl::ExpHeapStaticAllocator<1024 * 1024, MyStatelessAllocatorTag> MyStatelessAllocator;
typedef MyStatelessAllocator::Policy MyStatelessAllocationPolicy;

struct MyStatelessAllocatorInitializer
{
    MyStatelessAllocatorInitializer() NN_NOEXCEPT
    {
        MyStatelessAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyStatelessAllocatorInitializer;

class MultiImpl
{
public:

    int Add(int x, int y) NN_NOEXCEPT
    {
        return x + y;
    }

    int Sub(int x, int y) NN_NOEXCEPT
    {
        return x - y;
    }

    int Multiply(int x, int y) NN_NOEXCEPT
    {
        return x * y;
    }

    static void* operator new(size_t size) NN_NOEXCEPT
    {
        return MyStatelessAllocator::Allocate(size);
    }

    static void operator delete(void* p, size_t size) NN_NOEXCEPT
    {
        MyStatelessAllocator::Deallocate(p, size);
    }
};

}

TEST(sf_AllInOne, Multi_UnmanagedServiceObject)
{
    typedef MultiImpl Impl;
    {
        nn::sf::UnmanagedServiceObject<nnt::testsf::IMulti1, Impl> x;
        auto p = &x;
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(-1, p->Sub(1, 2));
    }
    {
        nn::sf::UnmanagedServiceObject<nnt::testsf::IMulti2, Impl> x;
        auto p = &x;
        ASSERT_EQ(-1, p->Sub(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
    {
        nn::sf::UnmanagedServiceObject<nnt::testsf::IMulti3, Impl> x;
        auto p = &x;
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
}

TEST(sf_AllInOne, Multi_UnmanagedServiceObjectByPointer)
{
    typedef MultiImpl Impl;
    Impl impl;
    {
        nn::sf::UnmanagedServiceObjectByPointer<nnt::testsf::IMulti1, Impl> x(&impl);
        auto p = &x;
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(-1, p->Sub(1, 2));
    }
    {
        nn::sf::UnmanagedServiceObjectByPointer<nnt::testsf::IMulti2, Impl> x(&impl);
        auto p = &x;
        ASSERT_EQ(-1, p->Sub(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
    {
        nn::sf::UnmanagedServiceObjectByPointer<nnt::testsf::IMulti3, Impl> x(&impl);
        auto p = &x;
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
}

TEST(sf_AllInOne, Multi_Embraced)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef MultiImpl Impl;
    {
        auto p = Factory::CreateSharedEmplaced<nnt::testsf::IMulti1, Impl>();
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(-1, p->Sub(1, 2));
    }
    {
        auto p = Factory::CreateSharedEmplaced<nnt::testsf::IMulti2, Impl>();
        ASSERT_EQ(-1, p->Sub(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
    {
        auto p = Factory::CreateSharedEmplaced<nnt::testsf::IMulti3, Impl>();
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
}

TEST(sf_AllInOne, Multi_SharedPointer)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef MultiImpl Impl;
    auto all = Factory::CreateSharedEmplaced<nnt::testsf::IMultiAll, Impl>();
    {
        auto p = Factory::CreateShared<nnt::testsf::IMulti1>(all);
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(-1, p->Sub(1, 2));
    }
    {
        auto p = Factory::CreateShared<nnt::testsf::IMulti2>(all);
        ASSERT_EQ(-1, p->Sub(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
    {
        auto p = Factory::CreateShared<nnt::testsf::IMulti3>(all);
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
}

TEST(sf_AllInOne, Multi_StdUnique)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef MultiImpl Impl;
    {
        std::unique_ptr<Impl> unique(new Impl());
        auto p = Factory::CreateShared<nnt::testsf::IMulti1>(std::move(unique));
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(-1, p->Sub(1, 2));
    }
    {
        std::unique_ptr<Impl> unique(new Impl());
        auto p = Factory::CreateShared<nnt::testsf::IMulti2>(std::move(unique));
        ASSERT_EQ(-1, p->Sub(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
    {
        std::unique_ptr<Impl> unique(new Impl());
        auto p = Factory::CreateShared<nnt::testsf::IMulti3>(std::move(unique));
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
}

// TODO:
// TEST(sf_AllInOne, Multi_StdShared)
//   std::shared_ptr が内部アロケートをするため、後回し
// TEST(sf_AllInOne, Multi_Pointer)
//   煩雑で、かつ、需要が少なそうなので後回し

TEST(sf_AllInOne, Multi_PointerWithoutManagement)
{
    typedef nn::sf::ObjectFactory<MyStatelessAllocationPolicy> Factory;
    typedef MultiImpl Impl;
    Impl impl;
    {
        auto p = Factory::CreateSharedWithoutManagement<nnt::testsf::IMulti1>(&impl);
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(-1, p->Sub(1, 2));
    }
    {
        auto p = Factory::CreateSharedWithoutManagement<nnt::testsf::IMulti2>(&impl);
        ASSERT_EQ(-1, p->Sub(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
    {
        auto p = Factory::CreateSharedWithoutManagement<nnt::testsf::IMulti3>(&impl);
        ASSERT_EQ(3, p->Add(1, 2));
        ASSERT_EQ(2, p->Multiply(1, 2));
    }
}
