﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 設定
const int RepeatCount = 3;

#include <nnt/nntest.h>

// テスト対象
#include "testSf_IDomainTests.h"

// hipc
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ContextControl.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_StdAllocationPolicy.h>
#include <type_traits>
#include <new>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <memory>
#include <atomic>
#include <nn/sf/sf_ShimLibraryUtility.h>

namespace {

typedef nn::sf::StdAllocationPolicy<std::allocator> MyAllocationPolicy;

class DomainTestObjectImpl
    : public nn::sf::ISharedObject
{
private:

    nn::sf::SharedPointer<DomainTestObjectImpl> m_Parent;
    std::atomic<int64_t> m_Value;

public:

    explicit DomainTestObjectImpl(DomainTestObjectImpl* parent)
        : m_Parent(parent, true)
        , m_Value(0)
    {
    }

    nn::Result Open(nn::sf::Out<nn::sf::SharedPointer<nnt::testsf::IDomainTestObject>> pOut) NN_NOEXCEPT;

    void AddValueImpl(std::int64_t n) NN_NOEXCEPT
    {
        m_Value += n;
        // 親がいるときは、親の AddValueImpl(n) も呼び出す
        if (m_Parent)
        {
            m_Parent->AddValueImpl(n);
        }
    }

    nn::Result AddValue(std::int64_t n) NN_NOEXCEPT
    {
        AddValueImpl(n);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetValue(nn::sf::Out<std::int64_t> pOut) NN_NOEXCEPT
    {
        *pOut = m_Value;
        NN_RESULT_SUCCESS;
    }

    nn::Result AddValueWith(nn::sf::SharedPointer<nnt::testsf::IDomainTestObject> p, std::int64_t n) NN_NOEXCEPT
    {
        return p->AddValue(n);
    }

    nn::Result AddCharValueWith(nn::sf::SharedPointer<nnt::testsf::IDomainTestObject> p, std::int8_t n) NN_NOEXCEPT
    {
        return p->AddValue(n);
    }

};

nn::sf::SharedPointer<nnt::testsf::IDomainTestObject> CreateTestObject(DomainTestObjectImpl* parent) NN_NOEXCEPT
{
    return nn::sf::ObjectFactory<MyAllocationPolicy>::CreateSharedEmplaced<nnt::testsf::IDomainTestObject, DomainTestObjectImpl>(parent);
}

nn::Result DomainTestObjectImpl::Open(nn::sf::Out<nn::sf::SharedPointer<nnt::testsf::IDomainTestObject>> pOut) NN_NOEXCEPT
{
    *pOut = CreateTestObject(this);
    NN_RESULT_SUCCESS;
}

const int ServerThreadCount = 3;
const size_t ServerStackSize = nn::os::StackRegionAlignment * 4;
NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_SreverThreadStack[ServerStackSize * ServerThreadCount];

const int ClientThreadCount = 4;
const size_t ClientStackSize = nn::os::StackRegionAlignment * 4;
NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ClientThreadStack[ClientStackSize * ClientThreadCount];

const int UpperObjectCount = 100;
const int LowerObjectCountPerUpper = 100;
const int LowerObjectCount = UpperObjectCount * LowerObjectCountPerUpper;
const int AllObjectCount = 1 + (UpperObjectCount + LowerObjectCount) * ClientThreadCount;

struct MyHipcServerManagerOption
{
    static const bool CanDeferInvokeRequest = true;
    static const int SubDomainCountMax = ClientThreadCount;
    static const int ObjectInSubDomainCountMax = AllObjectCount;
};

class MyHipcServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<ServerThreadCount, 1, MyHipcServerManagerOption>
{
};

const char ServiceName[] = "sf_test1";
std::aligned_storage<sizeof(MyHipcServerManager), NN_ALIGNOF(MyHipcServerManager)>::type g_MyHipcServerManagerStorage;

void ServerFunction(MyHipcServerManager* pManager) NN_NOEXCEPT
{
    pManager->LoopAuto();
}

void ClientFunction(nnt::testsf::IDomainTestObject* p) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nnt::testsf::IDomainTestObject> master(p, true);
    typedef nn::sf::SharedPointer<nnt::testsf::IDomainTestObject> TestObjectPointer;

    // master から uppers[0..(UpperObjectCount - 1)] を取得
    std::unique_ptr<TestObjectPointer[]> uppers(new TestObjectPointer[UpperObjectCount]);
    for (int i = 0; i < UpperObjectCount; ++i)
    {
        auto&& upper = uppers[i];
        EXPECT_TRUE(master->Open(&upper).IsSuccess());
    }

    // 各 uppers[i] から lowers[(i * LowerObjectCountPerUpper)..((i + 1) * LowerObjectCountPerUpper - 1)] を取得
    std::unique_ptr<TestObjectPointer[]> lowers(new TestObjectPointer[LowerObjectCount]);
    for (int i = 0; i < UpperObjectCount; ++i)
    {
        auto&& upper = uppers[i];
        for (int j = 0; j < LowerObjectCountPerUpper; ++j)
        {
            auto&& lower = lowers[i * LowerObjectCountPerUpper + j];
            EXPECT_TRUE(upper->Open(&lower).IsSuccess());
        }
    }

    // 各 lowers[0..(LowerObjectCount - 1)] に対し AddValue(1) を呼ぶ
    // 間接的に対応する uppers[] にも LowerObjectCountPerUpper 回 AddValueImpl(1) が呼ばれ、
    // 間接的に master にも LowerObjectCount 回 AddValueImpl(1) が呼ばれる
    for (int i = 0; i < LowerObjectCount; ++i)
    {
        auto&& lower = lowers[i];
        EXPECT_TRUE(lower->AddValue(1).IsSuccess());
    }
    for (int i = 0; i < UpperObjectCount; ++i)
    {
        auto&& upper = uppers[i];
        int64_t value;
        EXPECT_TRUE(upper->GetValue(&value).IsSuccess());
        EXPECT_EQ(LowerObjectCountPerUpper, value);
    }

    // uppers を解放
    // 内部的には lowers[] は対応する uppers[] への参照を持っているため、内部的には解放されず、
    // 以降の lowers[].AddValue() も対応する uppers[].AddValueImpl() や master.AddValueImpl() を内部的に呼ぶことができる
    uppers.reset();

    // 各 lowers[0..(LowerObjectCount - 1)] に対し master->AddValueWith を呼び間接的に lower.AddValue(-1) を呼ぶ
    // 間接的に master にも LowerObjectCount 回 AddValueImpl(-1) が呼ばれる
    for (int i = 0; i < LowerObjectCount; ++i)
    {
        auto&& lower = lowers[i];
        if (i % 2)
        {
            EXPECT_TRUE(master->AddValueWith(lower, -1).IsSuccess());
        }
        else
        {
            EXPECT_TRUE(master->AddCharValueWith(lower, -1).IsSuccess());
        }
    }

    // この時点で master に対する AddValueImpl(1) の呼び出し回数と AddValueImpl(-1) の呼び出し回数は同じはず
}

nn::sf::ShimLibraryObjectHolder<nnt::testsf::IDomainTestObject> g_Library = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
nn::sf::SimpleAllInOneHipcSubDomainClientManager<AllObjectCount> g_HipcSub = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;

void DoDomainTests(int serverThreadCount, int clientThreadCount)
{
    auto serverMasterObject = CreateTestObject(nullptr);
    auto pManager = new (&g_MyHipcServerManagerStorage) MyHipcServerManager;
    NN_UTIL_SCOPE_EXIT
    {
        pManager->~MyHipcServerManager();
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->RegisterObjectForPort(serverMasterObject, 10, ServiceName));
    pManager->Start();

    nnt::sfutil::TestThread serverThreads[ServerThreadCount];
    for (int i = 0; i < serverThreadCount; ++i)
    {
        auto&& thread = serverThreads[i];
        thread.Initialize(&ServerFunction, pManager, &g_SreverThreadStack[i * ServerStackSize], ServerStackSize, 10);
        thread.Start();
    }

    // 一回のサーバ起動に対しクライアント処理を 3 回行う (リークの簡易チェック)
    // クライアントマネージャを各クライアントスレッドで共有
    //nn::sf::HipcSimpleClientSessionManager clientManager;
    for (int i = 0; i < RepeatCount; ++i)
    {
        // ライブラリを初期化し、マスターオブジェクトを取得
        auto result = g_HipcSub.InitializeShimLibraryHolder(&g_Library, ServiceName);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_HipcSub.GetClientSessionManager().SetSessionCount(ServerThreadCount));
        EXPECT_TRUE(result.IsSuccess());
        auto clientMasterObject = g_Library.GetObject();

        // マスターオブジェクトを各クライアントスレッドに渡して起動
        nnt::sfutil::TestThread clientThreads[ClientThreadCount];
        for (int j = 0; j < clientThreadCount; ++j)
        {
            auto&& thread = clientThreads[j];
            thread.Initialize(&ClientFunction, clientMasterObject.Get(), &g_ClientThreadStack[j * ClientStackSize], ClientStackSize, 10);
            thread.Start();
        }
        for (int j = 0; j < clientThreadCount; ++j)
        {
            auto&& thread = clientThreads[j];
            thread.Wait();
        }

        // 各クライアントスレッドはマスターオブジェクトの子オブジェクトに同じ回数だけ AddValueImpl(1) と AddValueImpl(-1) を呼んでいるはずで、
        // マスターオブジェクトの値は 0 になっているはず
        {
            int64_t value;
            EXPECT_TRUE(clientMasterObject->GetValue(&value).IsSuccess());
            EXPECT_EQ(0, value);
        }

        // マスターオブジェクトを解放
        clientMasterObject.Reset();

        // ライブラリを解放
        g_Library.FinalizeHolder();
    }

    pManager->RequestStop();

    for (int i = 0; i < serverThreadCount; ++i)
    {
        auto&& thread = serverThreads[i];
        thread.Wait();
    }
}

TEST(sf_AllInOne, DomainTests_Hipc_Server1_Client1)
{
    DoDomainTests(1, 1);
}

TEST(sf_AllInOne, DomainTests_Hipc_Server3_Client1)
{
    DoDomainTests(3, 1);
}

TEST(sf_AllInOne, DomainTests_Hipc_Server1_Client4)
{
    DoDomainTests(1, 4);
}

TEST(sf_AllInOne, DomainTests_Hipc_Server3_Client4)
{
    DoDomainTests(3, 4);
}

}
