﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/settings/system/settings_Web.h>
#include <nnt/nntest.h>

namespace nn { namespace settings { namespace fwdbg {

void SetWebInspectorFlag(bool value) NN_NOEXCEPT;

void SetAllowedSslHosts(
    const ::nn::settings::system::AllowedSslHost values[],
    size_t count) NN_NOEXCEPT;

void SetHostFsMountPoint(
    const ::nn::settings::system::HostFsMountPoint& value) NN_NOEXCEPT;

}}} // namespace nn::settings::fwdbg

//!< Web インスペクタを使用するか否かを表す値の設定を変更可能か
TEST(WebSuite, WebInspectorFlagTest1)
{
    ::nn::settings::fwdbg::SetWebInspectorFlag(false);

    EXPECT_FALSE(::nn::settings::system::UsesWebInspector());

    ::nn::settings::fwdbg::SetWebInspectorFlag(true);

    EXPECT_TRUE(::nn::settings::system::UsesWebInspector());

    ::nn::settings::fwdbg::SetWebInspectorFlag(false);

    EXPECT_FALSE(::nn::settings::system::UsesWebInspector());
}

//!< SSL チェック無効化ホストの設定を変更可能か
TEST(WebSuite, AllowedSslHostsTest1)
{
    static ::nn::settings::system::AllowedSslHost
        s_ValuesA[::nn::settings::system::AllowedSslHostCountMax] = {};
    static ::nn::settings::system::AllowedSslHost
        s_ValuesB[::nn::settings::system::AllowedSslHostCountMax] = {};

    for (size_t i = 0; i < NN_ARRAY_SIZE(s_ValuesA); ++i)
    {
        for (size_t j = 0; j < NN_ARRAY_SIZE(s_ValuesA[i].name) - 1; ++j)
        {
            s_ValuesA[i].name[j] = static_cast<char>('a' + (i + j) % 26);
        }
    }

    ::nn::settings::fwdbg::SetAllowedSslHosts(
        s_ValuesA, NN_ARRAY_SIZE(s_ValuesA));

    EXPECT_EQ(
        NN_ARRAY_SIZE(s_ValuesA),
        ::nn::settings::system::GetAllowedSslHosts(
            s_ValuesB, NN_ARRAY_SIZE(s_ValuesB)));

    for (size_t i = 0; i < NN_ARRAY_SIZE(s_ValuesA); ++i)
    {
        EXPECT_STREQ(s_ValuesA[i].name, s_ValuesB[i].name);

        s_ValuesB[i] = ::nn::settings::system::AllowedSslHost();
    }

    ::nn::settings::fwdbg::SetAllowedSslHosts(&s_ValuesA[6], 2u);

    EXPECT_EQ(
        2u,
        ::nn::settings::system::GetAllowedSslHosts(
            s_ValuesB, NN_ARRAY_SIZE(s_ValuesB)));

    for (size_t i = 0; i < 2u; ++i)
    {
        EXPECT_STREQ(s_ValuesA[i + 6u].name, s_ValuesB[i].name);

        s_ValuesB[i] = ::nn::settings::system::AllowedSslHost();
    }

    ::nn::settings::fwdbg::SetAllowedSslHosts(nullptr, 0);

    EXPECT_EQ(
        0u,
        ::nn::settings::system::GetAllowedSslHosts(
            s_ValuesB, NN_ARRAY_SIZE(s_ValuesB)));
}

//!< ホスト FS のマウントポイントの設定を変更可能か
TEST(WebSuite, HostFsMountPointTest1)
{
    static ::nn::settings::system::HostFsMountPoint s_ValueA = {};
    static ::nn::settings::system::HostFsMountPoint s_ValueB = {};

    ::nn::settings::fwdbg::SetHostFsMountPoint(s_ValueA);

    ::nn::settings::system::GetHostFsMountPoint(&s_ValueB);

    EXPECT_STREQ(s_ValueA.path, s_ValueB.path);

    for (size_t i = 0; i < NN_ARRAY_SIZE(s_ValueA.path) - 1; ++i)
    {
        s_ValueA.path[i] = static_cast<char>('a' + i % 26);
    }

    ::nn::settings::fwdbg::SetHostFsMountPoint(s_ValueA);

    ::nn::settings::system::GetHostFsMountPoint(&s_ValueB);

    EXPECT_STREQ(s_ValueA.path, s_ValueB.path);

    for (size_t i = 0; i < NN_ARRAY_SIZE(s_ValueA.path); ++i)
    {
        s_ValueA.path[i] = '\0';
        s_ValueB.path[i] = '\0';
    }

    ::nn::settings::fwdbg::SetHostFsMountPoint(s_ValueA);

    ::nn::settings::system::GetHostFsMountPoint(&s_ValueB);

    EXPECT_STREQ(s_ValueA.path, s_ValueB.path);
}
