﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Region.h>
#include <nnt/nntest.h>

namespace {

//!< テスト用本体 EULA の同意バージョン A
const ::nn::settings::system::EulaVersion EulaVersionA = {
    0xDEADBEEF,
    static_cast<int32_t>(::nn::settings::system::RegionCode_Europe),
    static_cast<int32_t>(
        ::nn::settings::system::EulaVersionClockType_SteadyClock),
    { 1234 },
    { 5678, {} },
};

//!< テスト用本体 EULA の同意バージョン B
const ::nn::settings::system::EulaVersion EulaVersionB = {
    0xFEEDCAFE,
    static_cast<int32_t>(::nn::settings::system::RegionCode_Usa),
    static_cast<int32_t>(
        ::nn::settings::system::EulaVersionClockType_NetworkSystemClock),
    { 5678 },
    { 1234, {} },
};

::nn::settings::system::EulaVersion g_EulaVersion[
    ::nn::settings::system::EulaVersionCountMax];

//!< 本体 EULA の同意バージョンが等しいことを期待します。
void ExpectEqualEulaVersion(
    const ::nn::settings::system::EulaVersion& lhs,
    const ::nn::settings::system::EulaVersion& rhs) NN_NOEXCEPT;

} // namespace

//!< 本体 EULA の同意バージョンを取得可能か
TEST(EulaSuite, EulaVersionTest1)
{
    ::nn::settings::system::GetEulaVersions(&g_EulaVersion[0], 1);
}

//!< 本体 EULA の同意バージョンを変更可能か
TEST(EulaSuite, EulaVersionTest2)
{
    ::nn::settings::system::SetEulaVersions(&EulaVersionA, 1);

    ::nn::settings::system::EulaVersion* settings = &g_EulaVersion[0];

    const auto countMax = ::nn::settings::system::EulaVersionCountMax;

    EXPECT_EQ(1, ::nn::settings::system::GetEulaVersions(settings, countMax));

    {
        SCOPED_TRACE("");
        ExpectEqualEulaVersion(EulaVersionA, settings[0]);
    }

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = (i % 2 == 1) ? EulaVersionA : EulaVersionB;
    }

    ::nn::settings::system::SetEulaVersions(settings, countMax);

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = ::nn::settings::system::EulaVersion();
    }

    EXPECT_EQ(countMax,
              ::nn::settings::system::GetEulaVersions(settings, countMax));

    for (size_t i = 0; i < countMax; ++i)
    {
        SCOPED_TRACE("");
        ExpectEqualEulaVersion(
            (i % 2 == 1) ? EulaVersionA : EulaVersionB, settings[i]);
    }

    ::nn::settings::system::SetEulaVersions(nullptr, 0);

    EXPECT_EQ(0, ::nn::settings::system::GetEulaVersions(settings, countMax));
}

namespace {

void ExpectEqualEulaVersion(
    const ::nn::settings::system::EulaVersion& lhs,
    const ::nn::settings::system::EulaVersion& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.version, rhs.version);
    EXPECT_EQ(lhs.regionCode, rhs.regionCode);
    EXPECT_EQ(lhs.clockType, rhs.clockType);
    EXPECT_EQ(lhs.networkSystemClock, rhs.networkSystemClock);
    EXPECT_EQ(lhs.steadyClock, rhs.steadyClock);
}

} // namespace
