﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <string>

#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include "Types.h"

class MountPath
{
    friend class FileAccessor;
    std::string m_MountName;
    std::string m_TargetPath;

public:
    MountPath(std::string mountName = "", std::string targetPath = "")
    {
        UpdatePath(mountName, targetPath);
    }

    void UpdatePath(std::string mountName, std::string targetPath)
    {
        m_MountName = mountName;
        m_TargetPath = targetPath;
    }

    const std::string GetMountPath(std::string path = "")
    {
        return (m_MountName + ":/" + path);
    }
};

class FileAccessor
{
private:
    FileAccessor();
    bool m_IsMounted;

public:
    static MountPath defaultMountPath;

    static FileAccessor &GetInstance()
    {
        static FileAccessor FileAccessor;
        return FileAccessor;
    }
    nn::Result Mount(const char* mountName, const char* targetPath);
    nn::Result Mount(const MountPath& path = defaultMountPath)
    {
        return Mount(path.m_MountName.c_str(), path.m_TargetPath.c_str());
    }
    void Unmount(const char* mountName);
    void Unmount(const MountPath& path = defaultMountPath)
    {
        Unmount(path.m_MountName.c_str());
    }

    // Open して Read/Write, Close する
    void Read(u8* outBuffer, size_t* outReadLength, const size_t bufferLength, const char* filePath);
    void Write(const u8* buffer, const size_t bufferLength, const char* filePath);

    nn::Result Open(nn::fs::FileHandle* handle, const char* filePath, int64_t fileSize, nn::fs::OpenMode mode);

    void GetDirectoryList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const u32 directoryEntryBufferLength, const char* directoryPath)
    {
        GetDirectoryOrFileList(outDirectoryEntryBuffer, outDirectoryEntryCount, directoryEntryBufferLength, directoryPath, nn::fs::OpenDirectoryMode_Directory);
    }
    void GetFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const u32 directoryEntryBufferLength, const char* directoryPath)
    {
        GetDirectoryOrFileList(outDirectoryEntryBuffer, outDirectoryEntryCount, directoryEntryBufferLength, directoryPath, nn::fs::OpenDirectoryMode_File);
    }
    bool IsDirectoryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const u32 directoryEntryBufferLength, const char* directoryPath, const char* targetName)
    {
        return IsEntryExisted(workDirectoryEntryBuffer, directoryEntryBufferLength, directoryPath, targetName, nn::fs::OpenDirectoryMode_Directory);
    }
    bool IsFileExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const u32 directoryEntryBufferLength, const char* directoryPath, const char* targetName)
    {
        return IsEntryExisted(workDirectoryEntryBuffer, directoryEntryBufferLength, directoryPath, targetName, nn::fs::OpenDirectoryMode_File);
    }
    bool IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const u32 directoryEntryBufferLength, const char* directoryPath, const char* targetName)
    {
        return IsEntryExisted(workDirectoryEntryBuffer, directoryEntryBufferLength, directoryPath, targetName, nn::fs::OpenDirectoryMode_All);
    }
    bool IsMounted()
    {
        return m_IsMounted;
    }

private:
    void GetDirectoryOrFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const u32 directoryEntryBufferLength, const char* directoryPath, int openDirectoryMode);
    bool IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const u32 directoryEntryBufferLength, const char* directoryPath, const char* targetName, int openDirectoryMode);
};

