﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>

#include "FileAccessor.h"
#include "Util.h"

using namespace ::nn::fs;

MountPath FileAccessor::defaultMountPath;

FileAccessor::FileAccessor()
{
    m_IsMounted = false;

}

nn::Result FileAccessor::Mount(const char* mountName, const char* targetPath)
{
    nn::Result result = nn::fs::MountHost(mountName, targetPath);
    if(result.IsFailure())
    {
        m_IsMounted = false;
        NN_SC_DETAIL_FILE_LOG("failed to mount '%s'\n", targetPath);
        return result;
    }

    m_IsMounted = true;
    NN_SC_DETAIL_FILE_LOG("'%s' mounted on '%s'\n", targetPath, mountName);
    return result;
}

void FileAccessor::Unmount(const char* mountName)
{
    nn::fs::Unmount(mountName);
    NN_SC_DETAIL_FILE_LOG("unmounted '%s' \n", mountName);
}

bool FileAccessor::IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const u32 directoryEntryBufferLength, const char* directoryPath, const char* targetName, int openDirectoryMode)
{
    int64_t directoryEntryListLength = 0;

    this->GetDirectoryOrFileList(workDirectoryEntryBuffer, &directoryEntryListLength, directoryEntryBufferLength, directoryPath, openDirectoryMode);
    if ( directoryEntryListLength > 0 )
    {
        for(u32 i=0; i<directoryEntryListLength; i++)
        {
            u32 entryNameLength = strnlen(workDirectoryEntryBuffer[i].name, sizeof(workDirectoryEntryBuffer[i].name));
            u32 targetNameLength = strnlen(targetName, sizeof(workDirectoryEntryBuffer[i].name));
            if ( entryNameLength == targetNameLength && std::memcmp(workDirectoryEntryBuffer[i].name, targetName, targetNameLength) == 0 )
            {
                return true;
            }
        }
    }
    return false;
}

void FileAccessor::GetDirectoryOrFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const u32 directoryEntryBufferLength, const char* directoryPath, int openDirectoryMode)
{
    nn::fs::DirectoryHandle directory;
    nn::Result result = nn::fs::OpenDirectory(&directory, directoryPath, openDirectoryMode);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Open Directory '%s' Failure\n", directoryPath);
        return;
    }

    NN_SC_DETAIL_FILE_LOG("Read Directory '%s'\n", directoryPath);
    result = nn::fs::ReadDirectory(outDirectoryEntryCount, outDirectoryEntryBuffer, directory, directoryEntryBufferLength);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Read Directory '%s' Failure\n", directoryPath);
        return;
    }
    NN_SC_DETAIL_FILE_LOG("%ld Directories/Files Found\n", *outDirectoryEntryCount);

    nn::fs::CloseDirectory(directory);
}

void FileAccessor::Read(u8* outBuffer, size_t* outReadLength, const size_t bufferLength, const char* filePath)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = OpenFile(&file, filePath, nn::fs::OpenMode::OpenMode_Read);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Open File '%s' Failure\n", filePath);
    }
    NN_SC_DETAIL_FILE_LOG("Read File '%s'\n", filePath);
    result = ReadFile(outReadLength, file, 0, outBuffer, bufferLength);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Read File '%s' Failure\n", filePath);
    }
    CloseFile(file);
}

void FileAccessor::Write(const u8* buffer, const size_t bufferLength, const char* filePath)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = DeleteFile(filePath);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Could not delete file '%s'\n", filePath);
    }
    result = nn::fs::CreateFile(filePath, bufferLength);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Create File '%s' Error\n", filePath);
        NN_SC_DETAIL_FILE_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }
    NN_SC_DETAIL_FILE_LOG("Create File '%s' Success\n", filePath);
    result = OpenFile(&file, filePath, nn::fs::OpenMode::OpenMode_Write);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Open File '%s' Failure\n", filePath);
        return;
    }
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    result = WriteFile(file, 0, buffer, bufferLength, option);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Write File '%s' Failure\n", filePath);
    }
    CloseFile(file);
}

nn::Result FileAccessor::Open(nn::fs::FileHandle* handle, const char* filePath, int64_t fileSize, nn::fs::OpenMode mode)
{
    nn::Result result = nn::ResultSuccess();
    if(mode & nn::fs::OpenMode::OpenMode_Write)
    {
        result = DeleteFile(filePath);
        if(result.IsFailure())
        {
            NN_SC_DETAIL_FILE_LOG("Could not delete file '%s'\n", filePath);
        }
        result = nn::fs::CreateFile(filePath, fileSize);
        if(result.IsFailure())
        {
            NN_SC_DETAIL_FILE_LOG("Create File '%s' Error\n", filePath);
            NN_SC_DETAIL_FILE_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }
        NN_SC_DETAIL_FILE_LOG("Create File '%s' Success\n", filePath);
    }
    result = OpenFile(handle, filePath, mode);
    if(result.IsFailure())
    {
        NN_SC_DETAIL_FILE_LOG("Open File '%s' Failure\n", filePath);
        return result;
    }
    return result;
}
