﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>
#include <cstdlib>
#include <cstring>
#include <cinttypes>

//#define SELECT_AGING_MODE

#define CHECK_SEQUENTIAL_WRITE
#define CHECK_SEQUENTIAL_READ
#define CHECK_RANDOM_WRITE
#define CHECK_RANDOM_READ

namespace
{
    const uint32_t RandomAlignment      = 4 * 1024;           // ランダムアクセス時に守るアライメント
    const uint32_t NumCheckCountMax     = 128;                // 1 回のチェックでアクセスする回数
    const size_t   BufferSizeMax        = 8 * 1024 * 1024;    // 作業バッファの最大（チェックする最大チャンク以上にする）
    const uint32_t CheckStartingSector  = 0x400000;           // Write でシステムを壊さないためのオフセット
    uint8_t g_DataBuffer[BufferSizeMax];

    const size_t SectorSize = 512;

    #define RETURN_IF_FAILURE(result, ...) \
            if (result.IsFailure()){ NN_LOG( __VA_ARGS__ ); return result; }

    #define ABORT_IF_FAILURE(result, ...) \
            if (result.IsFailure()) { NN_LOG(__VA_ARGS__); NN_LOG(" Result: Module %d, Description %d\n", result.GetModule(), result.GetDescription()); NN_ABORT_UNLESS(result.IsSuccess()); }

    typedef struct Measurements_t
    {
        int64_t speed;
        int64_t iops;
        int64_t min;
        int64_t max;
    } Measurements;

    nn::fs::BisPartitionId TestPartitionId = nn::fs::BisPartitionId::UserDataRoot;
    std::unique_ptr<nn::fs::IStorage> g_pStorage;

    nn::Result GetMemoryCapacity(uint32_t* pNumSectorsOfMemoryCapacity)
    {
        int64_t value;
        nn::Result result = g_pStorage->GetSize(&value);
        if (result.IsFailure())
        {
            return result;
        }

        *pNumSectorsOfMemoryCapacity = static_cast<uint32_t>(value / SectorSize);

        return result;
    }

    nn::Result GetReadTick(uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize)
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        nn::Result result = g_pStorage->Read(sectorIndex * SectorSize, pBuffer, chunkSize);
        *diffTick = nn::os::GetSystemTick() - tick;

        return result;
    }

    nn::Result GetWriteTick(uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize)
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        nn::Result result = g_pStorage->Write(sectorIndex * SectorSize, pBuffer, chunkSize);
        *diffTick = nn::os::GetSystemTick() - tick;

        return result;
    }

    nn::Result MeasureReadSpeed(Measurements* pOutMeasurements, size_t chunkSize, bool isRandomAccess)
    {
        // デバイスサイズの取得（セクタ数）
        uint32_t numSectorsOfMemoryCapacity = 0;
        nn::Result result = GetMemoryCapacity(&numSectorsOfMemoryCapacity);
        RETURN_IF_FAILURE(result, "Error : Failed to Get Memory Capacity\n");
        NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

        uint8_t* pBuffer = g_DataBuffer;
        nn::os::Tick totalTick(0);
        nn::os::Tick minTick(0);
        nn::os::Tick maxTick(0);
        for (uint32_t checkCount = 0; checkCount < NumCheckCountMax; checkCount++)
        {
            srand(checkCount);
            // システム領域を破壊しないようにある程度のオフセットを付ける（Read では破壊しないが Write に合わせる）
            // Sequential アクセス時も一定以下の Chunk サイズのときはアライメントを守ってアクセスする
            uint32_t sectorIndex = CheckStartingSector + (isRandomAccess ? (rand() % (numSectorsOfMemoryCapacity - CheckStartingSector)) & ~(RandomAlignment / SectorSize - 1)
                                                          : ((chunkSize < RandomAlignment) ? RandomAlignment : chunkSize) * checkCount / SectorSize);

            nn::os::Tick tick(0);
            result = GetReadTick(pBuffer, &tick, sectorIndex, chunkSize);
            RETURN_IF_FAILURE(result, "Error : SectorIndex = 0x%08X, ChunkSize = 0x%08zX\n", sectorIndex, chunkSize);

            totalTick += tick;

            //NN_LOG("%llu (usec)\n", ConvertToTimeSpan(tick).GetMicroSeconds());

            minTick = (checkCount == 0 || tick < minTick) ? tick : minTick;
            maxTick = (checkCount == 0 || maxTick < tick) ? tick : maxTick;
        }

        nn::TimeSpan timeSpan = ConvertToTimeSpan(totalTick);
        pOutMeasurements->speed = ((static_cast<int64_t>(chunkSize) * NumCheckCountMax * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds()) / 1024; // X KByte/s
        pOutMeasurements->iops = (static_cast<int64_t>(NumCheckCountMax) * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds();
        pOutMeasurements->min = ConvertToTimeSpan(minTick).GetMicroSeconds();
        pOutMeasurements->max = ConvertToTimeSpan(maxTick).GetMicroSeconds();

        NN_LOG("%.1f (KB) %" PRId64 " (KB/sec) %" PRId64 " (IOPS) %" PRId64 " (usec) %" PRId64 " (usec)\n",
               chunkSize / 1024.0, pOutMeasurements->speed, pOutMeasurements->iops, pOutMeasurements->min, pOutMeasurements->max);

        return result;
    }

    nn::Result MeasureWriteSpeed(Measurements* pOutMeasurements, size_t chunkSize, bool isRandomAccess)
    {
        // デバイスサイズの取得（セクタ数）
        uint32_t numSectorsOfMemoryCapacity = 0;
        nn::Result result = GetMemoryCapacity(&numSectorsOfMemoryCapacity);
        RETURN_IF_FAILURE(result, "Error : Failed to Get Memory Capacity\n");
        NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

        uint8_t* pBuffer = g_DataBuffer;

        for (uint32_t byte = 0; byte < BufferSizeMax; byte++)
        {
            pBuffer[byte] = (byte % 2 == 0) ? 0xAA : 0x55;
        }

        nn::os::Tick totalTick(0);
        nn::os::Tick minTick(0);
        nn::os::Tick maxTick(0);
        for (uint32_t checkCount = 0; checkCount < NumCheckCountMax; checkCount++)
        {
            srand(checkCount);
            // システム領域を破壊しないようにある程度のオフセットを付ける
            // Sequential アクセス時も一定以下の Chunk サイズのときはアライメントを守ってアクセスする
            uint32_t sectorIndex = CheckStartingSector + (isRandomAccess ? (rand() % (numSectorsOfMemoryCapacity - CheckStartingSector)) & ~(RandomAlignment / SectorSize - 1)
                                    : ((chunkSize < RandomAlignment) ? RandomAlignment : chunkSize) * checkCount / SectorSize);

            nn::os::Tick tick(0);
            result = GetWriteTick(pBuffer, &tick, sectorIndex, chunkSize);
            RETURN_IF_FAILURE(result, "Error : SectorIndex = 0x%08X, ChunkSize = 0x%08zX\n", sectorIndex, chunkSize);
            totalTick += tick;

            //NN_LOG("%" PRId64 " (usec)\n", ConvertToTimeSpan(tick).GetMicroSeconds());

            minTick = (checkCount == 0 || tick < minTick) ? tick : minTick;
            maxTick = (checkCount == 0 || maxTick < tick) ? tick : maxTick;
        }

        nn::TimeSpan timeSpan = ConvertToTimeSpan(totalTick);
        pOutMeasurements->speed = ((static_cast<int64_t>(chunkSize) * NumCheckCountMax * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds()) / 1024; // X KByte/s
        pOutMeasurements->iops = (static_cast<int64_t>(NumCheckCountMax) * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds();
        pOutMeasurements->min = ConvertToTimeSpan(minTick).GetMicroSeconds();
        pOutMeasurements->max = ConvertToTimeSpan(maxTick).GetMicroSeconds();

        NN_LOG("%.1f (KB) %" PRId64 " (KB/sec) %" PRId64 " (IOPS) %" PRId64 " (usec) %" PRId64 " (usec)\n",
               chunkSize / 1024.0, pOutMeasurements->speed, pOutMeasurements->iops, pOutMeasurements->min, pOutMeasurements->max);

        return result;
    }

    void LogMeasurements(const size_t* chunkSizeTable, uint32_t numChunkTable, const Measurements* measurements) NN_NOEXCEPT
    {
        NN_LOG("Chunk(KB) KB/sec IOPS min(usec) max(usec)\n");
        for (uint32_t index = 0; index < numChunkTable; index++)
        {
            NN_LOG("%.1f %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "\n",
                   chunkSizeTable[index] / 1024.0,
                   measurements[index].speed, measurements[index].iops,
                   measurements[index].min, measurements[index].max);
        }
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
}

extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    NN_LOG("Test MMC Access\n");

    std::unique_ptr<nn::fs::IStorage> storage;
    nn::Result result = nn::fs::OpenBisPartition(&storage, TestPartitionId);
    ABORT_IF_FAILURE(result, "nn::fs::OpenBisPartition");
    g_pStorage = std::move(storage);

    const size_t ChunkSizeTable[] =
    {
        512, (1 * 1024), (2 * 1024), (4 * 1024), (8 * 1024), (16 * 1024),
        (32 * 1024), (64 * 1024), (128 * 1024), (256 * 1024), (512 * 1024),
        (1 * 1024 * 1024), (2 * 1024 * 1024), (4 * 1024 * 1024), (8 * 1024 * 1024),
    };
    const uint32_t NumChunkTable = sizeof(ChunkSizeTable) / sizeof(ChunkSizeTable[0]);

    #ifdef CHECK_SEQUENTIAL_WRITE
        Measurements SequentialWriteMeasurements[NumChunkTable];
    #endif
    #ifdef CHECK_SEQUENTIAL_READ
        Measurements SequentialReadMeasurements[NumChunkTable];
    #endif
    #ifdef CHECK_RANDOM_WRITE
        Measurements RandomWriteMeasurements[NumChunkTable];
    #endif
    #ifdef CHECK_RANDOM_READ
        Measurements RandomReadMeasurements[NumChunkTable];
    #endif

    #ifdef SELECT_AGING_MODE
        NN_LOG("/// SDMMC AGING CHECKER ///\n");
        uint32_t loop = 0;
        loop_bgn:
        NN_LOG("===== %d loop =====\n", loop++);
    #else
        NN_LOG("/// SDMMC SPEED CHECKER ///\n");
    #endif

    #ifdef CHECK_SEQUENTIAL_WRITE
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Sequential Write ##)\n", ChunkSizeTable[index]);
            result = MeasureWriteSpeed(&SequentialWriteMeasurements[index], ChunkSizeTable[index], false);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    #ifdef CHECK_SEQUENTIAL_READ
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Sequential Read ##)\n", ChunkSizeTable[index]);
            result = MeasureReadSpeed(&SequentialReadMeasurements[index], ChunkSizeTable[index], false);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    #ifdef CHECK_RANDOM_WRITE
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Random Write ##)\n", ChunkSizeTable[index]);
            result = MeasureWriteSpeed(&RandomWriteMeasurements[index], ChunkSizeTable[index], true);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    #ifdef CHECK_RANDOM_READ
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Random Read ##)\n", ChunkSizeTable[index]);
            result = MeasureReadSpeed(&RandomReadMeasurements[index], ChunkSizeTable[index], true);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    end:
    #ifdef CHECK_SEQUENTIAL_WRITE
        NN_LOG("--- Sequential Write ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, SequentialWriteMeasurements);
    #endif

    #ifdef CHECK_SEQUENTIAL_READ
        NN_LOG("--- Sequential Read ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, SequentialReadMeasurements);
    #endif

    #ifdef CHECK_RANDOM_WRITE
        NN_LOG("--- Random Write ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, RandomWriteMeasurements);
    #endif

    #ifdef CHECK_RANDOM_READ
        NN_LOG("--- Random Read ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, RandomReadMeasurements);
    #endif

    NN_LOG("[[ %s ]]\n", result.IsFailure() ? "FAILED !!" : "PASSED !!");
    if (result.IsFailure())
    {
        NN_LOG("Module:%d Description:%d Inner:%d\n",
               result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    #ifdef SELECT_AGING_MODE
        goto loop_bgn;
    #endif
}

