﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include <cinttypes>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。
// 注意：sdmmc の NN_SDMMC_ISSUE_COMMAND_FOR_DEBUG, NN_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG を有効にする必要があります。

namespace
{
    #if 0   // MMC (NAND データが壊れる可能性があるので要注意)
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else   // SD カード
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
            #else
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
            #endif
        #endif
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)
    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];

    const uint32_t TestSectorIndex = 1;
    const uint32_t TestNumSectors = 9;
    const size_t DataBufferSize = nn::util::align_up(nn::sdmmc::SectorSize * TestNumSectors, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;

    const char* GetPortName(nn::sdmmc::Port port)
    {
        switch (port)
        {
        case nn::sdmmc::Port_Mmc0:
            return "Port_Mmc0";
        case nn::sdmmc::Port_SdCard0:
            return "Port_SdCard0";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void SetData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            pCurrentData[i] = i;
        }
    }

    void CheckData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            NN_ABORT_UNLESS(pCurrentData[i] == i);
        }
    }

    void TestAbortCommand() NN_NOEXCEPT
    {
        // 便宜上、2GB 以上を High Capacity としてアドレスアクセスする（動作しないデバイスもありうる）
        uint32_t numSectorsOfMemoryCapacity = 0;
        nn::Result result = nn::sdmmc::GetDeviceMemoryCapacity(&numSectorsOfMemoryCapacity, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);
        bool isHighCapacity = true;
        if ((numSectorsOfMemoryCapacity / 2 / 1024 / 1024) < 2) // 2GB 未満
        {
            isHighCapacity = false;
        }

        uint16_t rca;
        result = nn::sdmmc::GetDeviceRca(&rca, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());

        // バッファクリア
        std::memset(g_pDataBuffer, 0xA5, DataBufferSize);
        nn::sdmmc::DataTransfer dataTransfer;
        dataTransfer.pDataBuffer = g_pDataBuffer;
        dataTransfer.dataBufferSize = DataBufferSize;
        dataTransfer.blockSize = nn::sdmmc::SectorSize;
        dataTransfer.numBlocks = TestNumSectors;

        // CMD25 発行後に即時 CMD12 で中断する
        int64_t abortTimingNanoSeconds = 0;
        uint32_t abortCommandIndex = 12;
        uint32_t abortCommandArgument = 0;
        NN_LOG("SetAbortCommandForDebug CMD%u 0x%08X %" PRId64 "(ns)\n", abortCommandIndex, abortCommandArgument, abortTimingNanoSeconds);
        nn::sdmmc::SetAbortCommandForDebug(TestPort, abortCommandIndex, abortCommandArgument, abortTimingNanoSeconds);
        uint32_t commandIndex = 25;
        uint32_t commandArgument = TestSectorIndex;
        if (!isHighCapacity)
        {
            commandArgument *= nn::sdmmc::SectorSize;
        }
        dataTransfer.isRead = false;
        uint32_t response;
        NN_LOG("IssueCommandForDebug CMD%u 0x%08X\n", commandIndex, commandArgument);
        result = nn::sdmmc::IssueCommandForDebug(&response, TestPort, commandIndex, commandArgument, &dataTransfer, true);
        if (!nn::sdmmc::ResultAbortCommandIssued::Includes(result))
        {
            // 想定外の result
            NN_LOG("unexpected result (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return;
        }

        // 念のため、Device Status クリア
        NN_LOG("ClearAbortCommandForDebug\n");
        nn::sdmmc::ClearAbortCommandForDebug(TestPort);
        commandIndex = 13;
        commandArgument = rca << 16;
        NN_LOG("IssueCommandForDebug CMD%u 0x%08X\n", commandIndex, commandArgument);
        result = nn::sdmmc::IssueCommandForDebug(&response, TestPort, commandIndex, commandArgument, nullptr, false);
        if (result.IsFailure())
        {
            NN_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("Response 0x%08X\n", response);
        }

        // CMD25 発行から 500us 後に CMD12 で中断する
        abortTimingNanoSeconds = 500 * 1000;
        NN_LOG("SetAbortCommandForDebug CMD%u 0x%08X %" PRId64 "(ns)\n", abortCommandIndex, abortCommandArgument, abortTimingNanoSeconds);
        nn::sdmmc::SetAbortCommandForDebug(TestPort, abortCommandIndex, abortCommandArgument, abortTimingNanoSeconds);
        NN_LOG("Write %s, sectorIndex=%u, numSectors=%u ...\n", GetPortName(TestPort), TestSectorIndex, TestNumSectors);
        result = nn::sdmmc::Write(TestPort, TestSectorIndex, TestNumSectors, g_pDataBuffer, DataBufferSize);
        if (!nn::sdmmc::ResultAbortCommandIssued::Includes(result))
        {
            // 想定外の result
            NN_LOG("unexpected result (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return;
        }

        // 念のため、Device Status クリア
        NN_LOG("ClearAbortCommandForDebug\n");
        nn::sdmmc::ClearAbortCommandForDebug(TestPort);
        commandIndex = 13;
        commandArgument = rca << 16;
        NN_LOG("CMD%u 0x%08X\n", commandIndex, commandArgument);
        result = nn::sdmmc::IssueCommandForDebug(&response, TestPort, commandIndex, commandArgument, nullptr, false);
        if (result.IsFailure())
        {
            NN_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            NN_LOG("Response 0x%08X\n", response);
        }

        // 以降、中断しないまま、正常に通信できるか確認
    }

    void TestReadWrite() NN_NOEXCEPT
    {
        SetData(g_pDataBuffer, nn::sdmmc::SectorSize * TestNumSectors);
        NN_LOG("Write %s, sectorIndex=%u, numSectors=%u ...\n", GetPortName(TestPort), TestSectorIndex, TestNumSectors);
        nn::Result result = nn::sdmmc::Write(TestPort, TestSectorIndex, TestNumSectors, g_pDataBuffer, DataBufferSize);
        if (result.IsFailure())
        {
            NN_LOG("nn::sdmmc::Write is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
            return;
        }
        NN_LOG("Done.\n");
        std::memset(g_pDataBuffer, 0xA5, DataBufferSize);
        NN_LOG("Read  %s, sectorIndex=%u, numSectors=%u ...\n", GetPortName(TestPort), TestSectorIndex, TestNumSectors);
        result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, TestSectorIndex, TestNumSectors);
        if (result.IsFailure())
        {
            NN_LOG("nn::sdmmc::Read is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
            return;
        }
        NN_LOG("Done.\n");
        NN_LOG("Verify ...\n");
        CheckData(g_pDataBuffer, nn::sdmmc::SectorSize * TestNumSectors);
        NN_LOG("Done.\n");
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo info;
        nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
            reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
    #endif
    if (TestPort == nn::sdmmc::Port_Mmc0)
    {
        nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }
    else
    {
        nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }

    uintptr_t dataBufferAddress;
    nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    result = nn::sdmmc::Activate(TestPort);
    if (result.IsSuccess())
    {
        TestAbortCommand();
        TestReadWrite();
    }
    else
    {
        NN_LOG("nn::sdmmc::Activate is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    nn::sdmmc::Deactivate(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&info);

        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}
