﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/erpt.h>
#include <nn/nn_Log.h>
#include <cinttypes>

namespace
{
    void LogData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint8_t* pCurrentData =  reinterpret_cast<uint8_t*>(pData);
        uint8_t* pDataEnd = pCurrentData + dataSize;
        while (true)
        {
            for (uint32_t i = 0; i < 0x10; i++)
            {
                NN_LOG("%02X ", *pCurrentData);
                pCurrentData++;
                if (pCurrentData >= pDataEnd)
                {
                    NN_LOG("\n");
                    return;
                }
            }
            NN_LOG("\n");
        }
    }

    void LogSdCardSpeedMode(nn::fs::SdCardSpeedMode speedMode) NN_NOEXCEPT
    {
        switch(speedMode)
        {
        case nn::fs::SdCardSpeedMode_Identification:
            NN_LOG("Identification");
            break;
        case nn::fs::SdCardSpeedMode_DefaultSpeed:
            NN_LOG("DefaultSpeed");
            break;
        case nn::fs::SdCardSpeedMode_HighSpeed:
            NN_LOG("HighSpeed");
            break;
        case nn::fs::SdCardSpeedMode_Sdr12:
            NN_LOG("Sdr12");
            break;
        case nn::fs::SdCardSpeedMode_Sdr25:
            NN_LOG("Sdr25");
            break;
        case nn::fs::SdCardSpeedMode_Sdr50:
            NN_LOG("Sdr50");
            break;
        case nn::fs::SdCardSpeedMode_Sdr104:
            NN_LOG("Sdr104");
            break;
        case nn::fs::SdCardSpeedMode_Ddr50:
            NN_LOG("Ddr50");
            break;
        case nn::fs::SdCardSpeedMode_Unknown:
            NN_LOG("Unknown");
            break;
        default:
            NN_LOG("Unknown: %d", speedMode);
            break;
        }
    }
}

extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);

    nn::Result result = nn::fs::MountSdCardForDebug("sdcard");
    if (result.IsSuccess())
    {
        // 本テストアプリを実行する前に挿入した SD カードに対して初期化を走らせておく
        nn::fs::Unmount("sdcard");
    }
    else
    {
        NN_LOG("nn::fs::MountSdCardForDebug() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    uint8_t cid[nn::fs::SdCardCidSize];
    result = nn::fs::GetSdCardCid(cid, sizeof(cid));
    if (result.IsSuccess())
    {
        NN_LOG("CID: ");
        LogData(cid, sizeof(cid));

        uint8_t mid = nn::fs::GetMidFromSdCardCid(cid);
        NN_LOG("  MID - 0x%02X\n", mid);

        char oid[nn::fs::SdCardCidOidSize + 1]; // + 1 : 終端 NULL 文字分
        nn::fs::GetOidFromSdCardCid(oid, sizeof(oid), cid);
        oid[nn::fs::SdCardCidOidSize] = 0;
        NN_LOG("  OID - \"%s\"\n", oid);

        char pnm[nn::fs::SdCardCidPnmSize + 1]; // + 1 : 終端 NULL 文字分
        nn::fs::GetPnmFromSdCardCid(pnm, sizeof(pnm), cid);
        pnm[nn::fs::SdCardCidPnmSize] = 0;
        NN_LOG("  PNM - \"%s\"\n", pnm);

        uint8_t prv = nn::fs::GetPrvFromSdCardCid(cid);
        NN_LOG("  PRV - 0x%02X\n", prv);

        uint32_t psn = nn::fs::GetPsnFromSdCardCid(cid);
        NN_LOG("  PSN - 0x%08X\n", psn);

        uint8_t mdtYearCode = nn::fs::GetMdtYearCodeFromSdCardCid(cid);
        NN_LOG("  MDT year code - %u (%u)\n", mdtYearCode, mdtYearCode + 2000);

        uint8_t mdtMonthCode = nn::fs::GetMdtMonthCodeFromSdCardCid(cid);
        NN_LOG("  MDT mounth code - %u\n", mdtMonthCode);
    }
    else
    {
        NN_LOG("nn::fs::GetSdCardCid() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    nn::fs::SdCardSpeedMode speedMode;
    result = nn::fs::GetSdCardSpeedMode(&speedMode);
    if (result.IsSuccess())
    {
        NN_LOG("Speed Mode: ");
        LogSdCardSpeedMode(speedMode);
        NN_LOG("\n");
    }
    else
    {
        NN_LOG("nn::fs::GetSdCardSpeedMode() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    int64_t size;
    result = nn::fs::GetSdCardUserAreaSize(&size);
    if (result.IsSuccess())
    {
        NN_LOG("User Area: %" PRId64 " (byte)\n", size);
        NN_LOG("  %" PRId64 " (KB), %" PRId64 " (sectors)\n", size / 1024, size / 512);
    }
    else
    {
        NN_LOG("nn::fs::GetSdCardUserAreaSize() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
    result = nn::fs::GetSdCardProtectedAreaSize(&size);
    if (result.IsSuccess())
    {
        NN_LOG("Protected Area: %" PRId64 " (byte)\n", size);
        NN_LOG("  %" PRId64 " (KB), %" PRId64 " (sectors)\n", size / 1024, size / 512);
    }
    else
    {
        NN_LOG("nn::fs::GetSdCardProtectedAreaSize() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    nn::fs::StorageErrorInfo storageErrorInfo;
    char logBuffer[nn::erpt::ArrayBufferLength];
    size_t logSize = 0;
    result = nn::fs::GetAndClearSdCardErrorInfo(&storageErrorInfo, &logSize, logBuffer, sizeof(logBuffer));
    if (result.IsSuccess())
    {
        NN_LOG("The number of activation failures: %u\n", storageErrorInfo.numActivationFailures);
        NN_LOG("The number of activation error corrections: %u\n", storageErrorInfo.numActivationErrorCorrections);
        NN_LOG("The number of read/write failuress: %u\n", storageErrorInfo.numReadWriteFailures);
        NN_LOG("The number of read/write error corrections: %u\n", storageErrorInfo.numReadWriteErrorCorrections);

        if (logSize > 0)
        {
            NN_LOG("Error Log: %s\n", logBuffer);
            NN_LOG("Error Log length: %zu\n", logSize);
        }
    }
    else
    {
        NN_LOG("nn::fs::GetAndClearSdCardErrorInfo() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}
