﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/erpt.h>
#include <nn/nn_Log.h>

namespace
{
    void LogData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint8_t* pCurrentData =  reinterpret_cast<uint8_t*>(pData);
        uint8_t* pDataEnd = pCurrentData + dataSize;
        while (true)
        {
            for (uint32_t i = 0; i < 0x10; i++)
            {
                NN_LOG("%02X ", *pCurrentData);
                pCurrentData++;
                if (pCurrentData >= pDataEnd)
                {
                    NN_LOG("\n");
                    return;
                }
            }
            NN_LOG("\n");
        }
    }

    void LogMmcSpeedMode(nn::fs::MmcSpeedMode speedMode) NN_NOEXCEPT
    {
        switch(speedMode)
        {
        case nn::fs::MmcSpeedMode_Identification:
            NN_LOG("Identification");
            return;
        case nn::fs::MmcSpeedMode_LegacySpeed:
            NN_LOG("LegacySpeed");
            return;
        case nn::fs::MmcSpeedMode_HighSpeed:
            NN_LOG("HighSpeed");
            return;
        case nn::fs::MmcSpeedMode_Hs200:
            NN_LOG("Hs200");
            return;
        case nn::fs::MmcSpeedMode_Hs400:
            NN_LOG("Hs400");
            return;
        case nn::fs::MmcSpeedMode_Unknown:
            NN_LOG("Unknown");
            return;
        default:
            NN_LOG("Unknown: %d", speedMode);
            return;
        }
    }
}

extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);

    uint8_t cid[nn::fs::MmcCidSize];
    nn::Result result = nn::fs::GetMmcCid(cid, sizeof(cid));
    if (result.IsSuccess())
    {
        NN_LOG("CID: ");
        LogData(cid, sizeof(cid));
    }
    else
    {
        NN_LOG("nn::fs::GetMmcCid() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    nn::fs::MmcSpeedMode speedMode;
    result = nn::fs::GetMmcSpeedMode(&speedMode);
    if (result.IsSuccess())
    {
        NN_LOG("Speed Mode: ");
        LogMmcSpeedMode(speedMode);
        NN_LOG("\n");
    }
    else
    {
        NN_LOG("nn::fs::GetMmcSpeedMode() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    uint8_t mmcExtendedCsd[nn::fs::MmcExtendedCsdSize];
    result = nn::fs::GetMmcExtendedCsd(mmcExtendedCsd, sizeof(mmcExtendedCsd));
    if (result.IsSuccess())
    {
        NN_LOG("MMC PRE_EOL_INFO: %u\n", mmcExtendedCsd[nn::fs::MmcExtendedCsdOffsetReEolInfo]);
        NN_LOG("MMC DEVICE_LIFE_TIME_EST_TYP_A: %u\n", mmcExtendedCsd[nn::fs::MmcExtendedCsdOffsetDeviceLifeTimeEstTypA]);
        NN_LOG("MMC DEVICE_LIFE_TIME_EST_TYP_B: %u\n", mmcExtendedCsd[nn::fs::MmcExtendedCsdOffsetDeviceLifeTimeEstTypB]);
    }
    else
    {
        NN_LOG("nn::fs::MmcExtendedCsd() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    uint32_t mmcPatrolCount;
    result = nn::fs::GetMmcPatrolCount(&mmcPatrolCount);
    if (result.IsSuccess())
    {
        NN_LOG("MMC Patrol Count: %u\n", mmcPatrolCount);
    }
    else
    {
        // 本体起動から 18 秒以内は取得できないので、断念
        NN_LOG("nn::fs::GetMmcPatrolCount() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }

    nn::fs::StorageErrorInfo storageErrorInfo;
    char logBuffer[nn::erpt::ArrayBufferLength];
    size_t logSize = 0;
    result = nn::fs::GetAndClearMmcErrorInfo(&storageErrorInfo, &logSize, logBuffer, sizeof(logBuffer));
    if (result.IsSuccess())
    {
        NN_LOG("The number of activation failures: %u\n", storageErrorInfo.numActivationFailures);
        NN_LOG("The number of activation error corrections: %u\n", storageErrorInfo.numActivationErrorCorrections);
        NN_LOG("The number of read/write failuress: %u\n", storageErrorInfo.numReadWriteFailures);
        NN_LOG("The number of read/write error corrections: %u\n", storageErrorInfo.numReadWriteErrorCorrections);

        if (logSize > 0)
        {
            NN_LOG("Error Log: %s\n", logBuffer);
            NN_LOG("Error Log length: %zu\n", logSize);
        }
    }
    else
    {
        NN_LOG("nn::fs::GetAndClearMmcErrorInfo() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
}
