﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SdmmcCommon.h"
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#include <nn/sdmmc/sdmmc_GcAsic.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os/os_MemoryHeap.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

void SetupDeviceAddressSpace(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceName deviceName) NN_NOEXCEPT
{
    // デバイスアドレス空間を 32bit アドレス空間で構築する
    nn::Result result = nn::dd::CreateDeviceAddressSpace(pDas, 0x100000000ull);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot create DeviceAddressSpace.");

    // 指定したデバイスに、構築したデバイスアドレス空間を関連付ける
    result = nn::dd::AttachDeviceAddressSpace(pDas, deviceName);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot attach DeviceAddressSpace to device.");
}

void CleanDeviceAddressSpace(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceName deviceName) NN_NOEXCEPT
{
    // 指定されたデバイスへの関連付けを解除する
    nn::dd::DetachDeviceAddressSpace(pDas, deviceName);

    // デバイスアドレス空間オブジェクトを破棄する
    nn::dd::DestroyDeviceAddressSpace(pDas);
}

nn::dd::DeviceVirtualAddress MapDeviceAddressSpaceAligned(nn::dd::DeviceAddressSpaceType* pDas, nn::dd::DeviceAddressSpaceMapInfo* pInfo,
    uintptr_t alignedBufferAddress, size_t alignedBufferSize, nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset) NN_NOEXCEPT
{
    static_assert(nn::dd::DeviceAddressSpaceMemoryRegionAlignment >= nn::sdmmc::BufferDeviceVirtualAddressAlignment, "");
    nn::dd::DeviceVirtualAddress deviceVirtualAddress;
    if (deviceVirtualAddressOffset == 0)
    {
        // NULL を避ける
        deviceVirtualAddress = nn::dd::DeviceAddressSpaceMemoryRegionAlignment;
    }
    else
    {
        // deviceVirtualAddressOffset までを避ける
        deviceVirtualAddress = nn::util::align_up(deviceVirtualAddressOffset, nn::dd::DeviceAddressSpaceMemoryRegionAlignment);
    }

    // 4KB アラインの 4KB のバッファは 1 回でデバイスアドレス空間にメモリをマップする
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    NN_ABORT_UNLESS(currentProcessHandle != nn::os::InvalidNativeHandle);
    nn::dd::InitializeDeviceAddressSpaceMapInfo(pInfo, pDas, currentProcessHandle,
        alignedBufferAddress, alignedBufferSize,
        deviceVirtualAddress,
        nn::dd::MemoryPermission_ReadWrite);
    size_t mappedSize;
    nn::Result result = nn::dd::MapNextDeviceAddressSpaceRegion(&mappedSize, pInfo);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot get MapNextDeviceAddressSpaceRegion.");
    NN_ABORT_UNLESS(mappedSize == alignedBufferSize);

    return deviceVirtualAddress;
}

void UnmapDeviceAddressSpaceAligned(nn::dd::DeviceAddressSpaceMapInfo* pInfo) NN_NOEXCEPT
{
    // デバイスアドレス空間からメモリを切り離す
    nn::dd::UnmapDeviceAddressSpaceRegion(pInfo);
}

nn::dd::DeviceVirtualAddress MapDeviceAddressSpaceAligned(nn::dd::DeviceAddressSpaceType* pDas,
    uintptr_t alignedBufferAddress, size_t alignedBufferSize, nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset) NN_NOEXCEPT
{
    // AllocateMemoryBlock() で確保した大きなデータバッファは下位 22bit 一致でデバイスアドレス空間にメモリをマップする
    static_assert(nn::os::MemoryBlockUnitSize >= nn::sdmmc::BufferDeviceVirtualAddressAlignment, "");
    const size_t DeviceAddressSpaceAligned = (0x1U << 22);
    static_assert(DeviceAddressSpaceAligned >= nn::os::MemoryBlockUnitSize, "");
    const uint64_t DeviceAddressSpaceAlignedMask = DeviceAddressSpaceAligned - 1;
    nn::dd::DeviceVirtualAddress deviceVirtualAddress;
    if (deviceVirtualAddressOffset == 0)
    {
        // NULL を避ける
        deviceVirtualAddress = DeviceAddressSpaceAligned;
    }
    else
    {
        // deviceVirtualAddressOffset までを避ける
        deviceVirtualAddress = nn::util::align_up(deviceVirtualAddressOffset, DeviceAddressSpaceAligned);
    }
    deviceVirtualAddress += (alignedBufferAddress & DeviceAddressSpaceAlignedMask);

    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    NN_ABORT_UNLESS(currentProcessHandle != nn::os::InvalidNativeHandle);
    nn::Result result = nn::dd::MapDeviceAddressSpaceAligned(pDas, currentProcessHandle,
        alignedBufferAddress, alignedBufferSize,
        deviceVirtualAddress,
        nn::dd::MemoryPermission_ReadWrite);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot map memory to DeviceAddressSpace.");

    return deviceVirtualAddress;
}

void UnmapDeviceAddressSpaceAligned(nn::dd::DeviceAddressSpaceType* pDas,
    uintptr_t alignedBufferAddress, size_t alignedBufferSize, nn::dd::DeviceVirtualAddress deviceVirtualAddress) NN_NOEXCEPT
{
    // デバイスアドレス空間からメモリを切り離す
    auto currentProcessHandle = nn::dd::GetCurrentProcessHandle();
    NN_ABORT_UNLESS(currentProcessHandle != nn::os::InvalidNativeHandle);
    nn::dd::UnmapDeviceAddressSpace(pDas, currentProcessHandle, alignedBufferAddress, alignedBufferSize, deviceVirtualAddress);
}

void LogSpeedMode(nn::sdmmc::SpeedMode speedMode) NN_NOEXCEPT
{
    switch (speedMode)
    {
    case nn::sdmmc::SpeedMode_MmcIdentification:
        NN_LOG("MMC Identification\n");
        break;
    case nn::sdmmc::SpeedMode_MmcLegacySpeed:
        NN_LOG("MMC Legacy Speed\n");
        break;
    case nn::sdmmc::SpeedMode_MmcHighSpeed:
        NN_LOG("MMC High Speed\n");
        break;
    case nn::sdmmc::SpeedMode_MmcHs200:
        NN_LOG("MMC HS200\n");
        break;
    case nn::sdmmc::SpeedMode_MmcHs400:
        NN_LOG("MMC HS400\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardIdentification:
        NN_LOG("SD Card Identification\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardDefaultSpeed:
        NN_LOG("SD Card Default Speed\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardHighSpeed:
        NN_LOG("SD Card High Speed\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr12:
        NN_LOG("SD Card SDR12\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr25:
        NN_LOG("SD Card SDR25\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr50:
        NN_LOG("SD Card SDR50\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardSdr104:
        NN_LOG("SD Card SDR104\n");
        break;
    case nn::sdmmc::SpeedMode_SdCardDdr50:
        NN_LOG("SD Card DDR50\n");
        break;
    case nn::sdmmc::SpeedMode_GcAsicFpgaSpeed:
        NN_LOG("GC ASIC FPGA\n");
        break;
    case nn::sdmmc::SpeedMode_GcAsicSpeed:
        NN_LOG("GC ASIC\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void LogBusStatus(nn::sdmmc::Port port) NN_NOEXCEPT
{
    nn::sdmmc::HostBusStatus hostBusStatus;
    nn::sdmmc::GetHostBusStatus(&hostBusStatus, port);

    NN_LOG("Host Bus Power: ");
    switch (hostBusStatus.busPower)
    {
    case nn::sdmmc::BusPower_Off:
        NN_LOG("OFF\n");
        break;
    case nn::sdmmc::BusPower_1_8V:
        NN_LOG("1.8V\n");
        break;
    case nn::sdmmc::BusPower_3_3V:
        NN_LOG("3.3V\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_LOG("Host Bus Width: ");
    switch (hostBusStatus.busWidth)
    {
    case nn::sdmmc::BusWidth_1Bit:
        NN_LOG("1 bit\n");
        break;
    case nn::sdmmc::BusWidth_4Bit:
        NN_LOG("4 bit\n");
        break;
    case nn::sdmmc::BusWidth_8Bit:
        NN_LOG("8 bit\n");
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_LOG("Host Clock: %u (kHz)\n", hostBusStatus.deviceClockFrequencyKHz);

    nn::sdmmc::SpeedMode hostSpeedMode;
    nn::sdmmc::BusWidth hostBusWidth;
    nn::Result result = nn::sdmmc::CheckConnection(&hostSpeedMode, &hostBusWidth, port);
    if (result.IsFailure())
    {
        NN_LOG("nn::sdmmc::CheckConnection is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        return;
    }

    NN_LOG("Host Speed Mode:   ");
    LogSpeedMode(hostSpeedMode);

    if (hostBusStatus.busWidth != hostBusWidth)
    {
        NN_LOG("nn::sdmmc::CheckConnection returns hostBusWidth: %d\n", hostBusWidth);
        return;
    }

    nn::sdmmc::SpeedMode deviceSpeedMode;
    result = nn::sdmmc::GetDeviceSpeedMode(&deviceSpeedMode, port);;
    if (result.IsFailure())
    {
        NN_LOG("nn::sdmmc::GetDeviceSpeedMode is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        return;
    }

    NN_LOG("Device Speed Mode: ");
    LogSpeedMode(deviceSpeedMode);
}
