﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include "hid_CalcUtil.h"
#include "hid_Gyroscope.h"

namespace
{

// LSM6DS3 の Typ値
const nn::xcd::SensorState TypicalOrigin      = { 0, 0, 0 };
const nn::xcd::SensorState TypicalSensitivity = { 13371, 13371, 13371 };

const nn::xcd::SensorState               DefaultLastAngularVelocityCount  = { 0, 0, 0 };
const nn::util::Float3                   DefaultStableCountSum            = {{{ 0.0f, 0.0f, 0.0f }}};
const nn::xcd::SensorState               DefaultStableCountTimes          = { 0, 0, 0 };
const bool                               DefaultEnableZeroDrift           = true;
const bool                               DefaultEnableZeroPlay            = false;
const float                              DefaultZeroPlayRadius            = 0.0f;
const nn::hid::detail::ZeroDriftMode     DefaultZeroDriftMode             = nn::hid::detail::ZeroDriftMode::ZeroDriftMode_Standard;

const nn::util::Float3                   DefaultMagnification             = {{{ 1.0f, 1.0f, 1.0f}}};
const nn::util::Float3                   DefaultDirectionMagnification    = {{{ 1.0f, 1.0f, 1.0f}}};

const nn::xcd::GyroscopeFsr              DefaultFsr                       = nn::xcd::GyroscopeFsr_2000dps;

void ApproximateByFactor( float* pOut, const float& target, const float& factor )
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    // factor が 1.0f に近いほど pOut は target に近づく
    *pOut += ( target - *pOut ) * factor;
}

void ApproximateByFactor( int32_t* pOut, const float& target, const float& factor )
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    // factor が 1.0f に近いほど pOut は target に近づく
    *pOut += static_cast<int32_t>(( target - *pOut ) * factor);
}

void CalcZeroPlay( float* pOutValue, const float& playRadius )
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if( -playRadius < *pOutValue &&
        *pOutValue  < playRadius )
    {
        *pOutValue = 0.0f;
    }
}

float GetRevisePower(float angularVelocity,
                     float samplingInterval,
                     float magnification)
{
    float       f1;
    float       v1;

    v1 = magnification * angularVelocity;
    f1 = samplingInterval * 360.0f * v1;
    f1 = f1 * f1 * 0.0001f + 1.0f;
    f1 *= samplingInterval * 2 * nn::hid::detail::Pi * v1;

    return f1;
}

float GetFullScaleRangeValue(const nn::xcd::GyroscopeFsr& gyroFsr)
{
    float gyroFsrValue = 1.0f;

    // gyroFsrValue = GyroscopeFsr / 360[dps]
    switch (gyroFsr)
    {
    case nn::xcd::GyroscopeFsr_250dps:
        gyroFsrValue = 0.6944f;
        break;
    case nn::xcd::GyroscopeFsr_500dps:
        gyroFsrValue = 1.3889f;
        break;
    case nn::xcd::GyroscopeFsr_1000dps:
        gyroFsrValue = 2.7778f;
        break;
    case nn::xcd::GyroscopeFsr_2000dps:
        gyroFsrValue = 5.5556f;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    return gyroFsrValue;
;
}

}

namespace nn { namespace hid { namespace detail {

Gyroscope::Gyroscope() NN_NOEXCEPT :
    m_LastAngularVelocityCount(DefaultLastAngularVelocityCount),
    m_StableCountSum(DefaultStableCountSum),
    m_StableCountTimes(DefaultStableCountTimes),
    m_EnableZeroDrift(DefaultEnableZeroDrift),
    m_EnableZeroPlay(DefaultEnableZeroPlay),
    m_ZeroPlayRadius(DefaultZeroPlayRadius),
    m_ZeroDriftMode(DefaultZeroDriftMode),
    m_Magnification(DefaultMagnification),
    m_DirectionMagnification(DefaultDirectionMagnification),
    m_Fsr(DefaultFsr)
{
    SetCalibrationValue(TypicalOrigin, TypicalSensitivity);
}

void Gyroscope::GetState(nn::util::Float3* pOutState, const nn::xcd::SensorState& count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    nn::xcd::SensorState angularVelocityCount; // 最新の角速度カウント

    // 静止判定を行い内部パラメータを更新
    UpdateGyroParameter( &m_StableCountTimes.x, &m_StableCountSum.x, count.x, m_LastAngularVelocityCount.x, m_Scale.x );
    UpdateGyroParameter( &m_StableCountTimes.y, &m_StableCountSum.y, count.y, m_LastAngularVelocityCount.y, m_Scale.y );
    UpdateGyroParameter( &m_StableCountTimes.z, &m_StableCountSum.z, count.z, m_LastAngularVelocityCount.z, m_Scale.z );

    // 静止期間が長いほど静止時の角速度カウント平均に現在の角速度カウントを近づける
    ReviseAngularVelocityCountByStability(&angularVelocityCount, count);

    // 静止期間が長いほどゼロ点を現在の角速度カウントに近づける
    if(m_EnableZeroDrift)
    {
        UpdateZeroCount(angularVelocityCount);
    }

    // 内部で確保された最新の角速度カウント値を更新する。
    UpdateLastAngularVelocityCount(angularVelocityCount);

    // 更新されたゼロ点を使って角速度値に変換する
    ConvertToAngularVelocity( pOutState, angularVelocityCount );

    // 遊び補正
    ReviseAngularVelocityByZeroPlay(pOutState);

    // 値域を制限するためクランプ
    const float AngularVelocityMax = GetFullScaleRangeValue(m_Fsr);
    const nn::util::Float3 Clamp = {{{ AngularVelocityMax, AngularVelocityMax, AngularVelocityMax }}};
    nn::hid::detail::ClampVector(pOutState, Clamp);
}

void Gyroscope::GetNoInputCount( nn::xcd::SensorState* pOutCount ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    pOutCount->x = static_cast<int32_t>(m_ZeroCount.x);
    pOutCount->y = static_cast<int32_t>(m_ZeroCount.y);
    pOutCount->z = static_cast<int32_t>(m_ZeroCount.z);
}

void Gyroscope::SetScaleValue() NN_NOEXCEPT
{
    const float Delta        = 936.0f - 0.0f; // 936dpsスケール - ゼロ点オフセット
    const float ConvertScale = 1.0f / 360.0f; // 360dps を 1.0 とする単位系に変換するための係数

    float fsrScale;

    // TORIAEZU: Fsr による係数, 2000dps のFSRを基準とした CAL 値が取得される
    switch (m_Fsr)
    {
    case nn::xcd::GyroscopeFsr_250dps:
        fsrScale = 0.125f;
        break;
    case nn::xcd::GyroscopeFsr_500dps:
        fsrScale = 0.250f;
        break;
    case nn::xcd::GyroscopeFsr_1000dps:
        fsrScale = 0.500f;
        break;
    case nn::xcd::GyroscopeFsr_2000dps:
        fsrScale = 1.000f;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    m_Scale.x = Delta / ( m_SensitivityCount.x - m_ZeroCount.x ) * ConvertScale * fsrScale;
    m_Scale.y = Delta / ( m_SensitivityCount.y - m_ZeroCount.y ) * ConvertScale * fsrScale;
    m_Scale.z = Delta / ( m_SensitivityCount.z - m_ZeroCount.z ) * ConvertScale * fsrScale;
}

void Gyroscope::SetSensorFsr(const nn::xcd::GyroscopeFsr& gyroFsr) NN_NOEXCEPT
{
    m_Fsr = gyroFsr;
    SetScaleValue();
}

//!< 6軸センサーのキャリブレーションを実行します。
void Gyroscope::SetCalibrationValue( const nn::xcd::SensorState& gyroscopeOrigin, const nn::xcd::SensorState& gyroscopeSensitivity ) NN_NOEXCEPT
{
    m_ZeroCount.x  = gyroscopeOrigin.x;
    m_ZeroCount.y  = gyroscopeOrigin.y;
    m_ZeroCount.z  = gyroscopeOrigin.z;

    m_SensitivityCount.x = gyroscopeSensitivity.x;
    m_SensitivityCount.y = gyroscopeSensitivity.y;
    m_SensitivityCount.z = gyroscopeSensitivity.z;

    SetScaleValue();
}

//!< 角速度を計算する際に掛ける倍率を各方向ごとに設定します。
void Gyroscope::SetMagnification(const nn::util::Float3& magnification) NN_NOEXCEPT
{
    m_Magnification = magnification;
}

//!< direction を内部で計算する際に使用する角速度の値にかける倍率を設定します。
void Gyroscope::SetDirectionMagnification(const nn::util::Float3& magnification) NN_NOEXCEPT
{
    m_DirectionMagnification = magnification;
}

void Gyroscope::ControlZeroDrift( bool enable ) NN_NOEXCEPT
{
    m_EnableZeroDrift = enable;
}

bool Gyroscope::IsZeroDriftEnabled() const NN_NOEXCEPT
{
    return m_EnableZeroDrift;
}

void Gyroscope::SetZeroDriftMode( const ZeroDriftMode& mode ) NN_NOEXCEPT
{
    m_ZeroDriftMode = mode;
}

void Gyroscope::GetZeroDriftMode( ZeroDriftMode* pOutMode ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMode);

    *pOutMode = m_ZeroDriftMode;
}

void Gyroscope::ResetZeroDriftMode() NN_NOEXCEPT
{
    SetZeroDriftMode(DefaultZeroDriftMode);
}

void Gyroscope::ControlZeroPlay( bool enable ) NN_NOEXCEPT
{
    m_EnableZeroPlay = enable;
}

bool Gyroscope::IsZeroPlayEnabled() const NN_NOEXCEPT
{
    return m_EnableZeroPlay;
}

void Gyroscope::SetZeroPlayParameters( const float& radius ) NN_NOEXCEPT
{
    m_ZeroPlayRadius = radius;
}

void Gyroscope::GetZeroPlayParameters( float* pOutRadius ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutRadius);

    *pOutRadius = m_ZeroPlayRadius;
}

void Gyroscope::ResetZeroPlayParameters() NN_NOEXCEPT
{
    SetZeroPlayParameters(DefaultZeroPlayRadius);
}

void Gyroscope::UpdateGyroParameter(int32_t* pStableCountTimes,
                                    float* pStableCountSum,
                                    const int32_t& currentCount,
                                    const int32_t& previousCount,
                                    const float& scale) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStableCountTimes);
    NN_SDK_REQUIRES_NOT_NULL(pStableCountSum);

    const float ZeroDriftCountRange = ZeroDrift[m_ZeroDriftMode].radius / scale;  //!< 前回サンプルとの差がこの値以下なら静止と判定する
    const int   StableCountTimesMax = ZeroDrift[m_ZeroDriftMode].count;   //!< 平均を取る回数の最大値
    const int   DeltaCount          = currentCount - previousCount;
    const bool  IsStable            = ( -ZeroDriftCountRange < DeltaCount && DeltaCount < ZeroDriftCountRange ); // ゼロ点付近であれば角速度が静止していると見做す

    if( IsStable )
    {
        if( *pStableCountTimes < StableCountTimesMax )
        {
            (*pStableCountSum) += currentCount;
            (*pStableCountTimes)++;
        }
    }
    else
    {
        // 状態のクリア
        *pStableCountSum = currentCount;
        *pStableCountTimes = 1;
    }
}

void Gyroscope::UpdateZeroCount(const nn::xcd::SensorState& count) NN_NOEXCEPT
{
    ApproximateByFactor( &m_ZeroCount.x, count.x, static_cast<float>( m_StableCountTimes.x / ZeroDrift[m_ZeroDriftMode].count ) );
    ApproximateByFactor( &m_ZeroCount.y, count.y, static_cast<float>( m_StableCountTimes.y / ZeroDrift[m_ZeroDriftMode].count ) );
    ApproximateByFactor( &m_ZeroCount.z, count.z, static_cast<float>( m_StableCountTimes.z / ZeroDrift[m_ZeroDriftMode].count ) );
}

void Gyroscope::UpdateLastAngularVelocityCount(const nn::xcd::SensorState& angularVelocity) NN_NOEXCEPT
{
    m_LastAngularVelocityCount = angularVelocity;
}

void Gyroscope::ConvertToAngularVelocity(nn::util::Float3* pOut, const nn::xcd::SensorState& count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    pOut->x = ( count.x - m_ZeroCount.x ) * m_Scale.x * m_Magnification.x;
    pOut->y = ( count.y - m_ZeroCount.y ) * m_Scale.y * m_Magnification.y;
    pOut->z = ( count.z - m_ZeroCount.z ) * m_Scale.z * m_Magnification.z;
}

void Gyroscope::ReviseAngularVelocityByZeroPlay( nn::util::Float3* pOut ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    if( m_EnableZeroPlay )
    {
        CalcZeroPlay( &pOut->x, m_ZeroPlayRadius );
        CalcZeroPlay( &pOut->y, m_ZeroPlayRadius );
        CalcZeroPlay( &pOut->z, m_ZeroPlayRadius );
    }
}

void Gyroscope::ReviseAngularVelocityCountByStability(nn::xcd::SensorState* pOut, const nn::xcd::SensorState& count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    *pOut = count;
    ApproximateByFactor( &pOut->x, static_cast<float>( m_StableCountSum.x / m_StableCountTimes.x), static_cast<float>( m_StableCountTimes.x / ZeroDrift[m_ZeroDriftMode].count ) );
    ApproximateByFactor( &pOut->y, static_cast<float>( m_StableCountSum.y / m_StableCountTimes.y), static_cast<float>( m_StableCountTimes.y / ZeroDrift[m_ZeroDriftMode].count ) );
    ApproximateByFactor( &pOut->z, static_cast<float>( m_StableCountSum.z / m_StableCountTimes.z), static_cast<float>( m_StableCountTimes.z / ZeroDrift[m_ZeroDriftMode].count ) );
}

void Gyroscope::UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& angularVelocity, const float& samplingInterval ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);

    // __VPADCalcGyroDir を参照
    nn::hid::DirectionState   d1;
    float f1;
    d1 = *pOutDirection ;

    // 方向の回転を直角方向へのベクトル加算で行っている為に、
    // 角速度の応じて加算ベクトルを大きくする近似処理を行っている。
    // 近似式はオリジナル

    //----- Ｘ方向回転処理
    f1 = GetRevisePower(angularVelocity.x,
                        samplingInterval,
                        m_DirectionMagnification.x);

    pOutDirection->y.x += f1 * d1.z.x ;
    pOutDirection->y.y += f1 * d1.z.y ;
    pOutDirection->y.z += f1 * d1.z.z ;

    pOutDirection->z.x -= f1 * d1.y.x ;
    pOutDirection->z.y -= f1 * d1.y.y ;
    pOutDirection->z.z -= f1 * d1.y.z ;

    //----- Ｙ方向回転処理
    f1 = GetRevisePower(angularVelocity.y,
                        samplingInterval,
                        m_DirectionMagnification.y);

    pOutDirection->z.x += f1 * d1.x.x ;
    pOutDirection->z.y += f1 * d1.x.y ;
    pOutDirection->z.z += f1 * d1.x.z ;

    pOutDirection->x.x -= f1 * d1.z.x ;
    pOutDirection->x.y -= f1 * d1.z.y ;
    pOutDirection->x.z -= f1 * d1.z.z ;

    //----- Ｚ方向回転処理
    f1 = GetRevisePower(angularVelocity.z,
                        samplingInterval,
                        m_DirectionMagnification.z);

    pOutDirection->x.x += f1 * d1.y.x ;
    pOutDirection->x.y += f1 * d1.y.y ;
    pOutDirection->x.z += f1 * d1.y.z ;

    pOutDirection->y.x -= f1 * d1.x.x ;
    pOutDirection->y.y -= f1 * d1.x.y ;
    pOutDirection->y.z -= f1 * d1.x.z ;

    //----- 正規化
    nn::hid::detail::NormalizeDirectionState( pOutDirection, 2.999f ) ;

}

void Gyroscope::GetZeroCountForDev(nn::util::Float3* pOutZeroCount ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutZeroCount);

    *pOutZeroCount = m_ZeroCount;
}

}}} // namespace nn::hid::detail
