﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       6軸センサーの計算のために用いるユーティリティ関数
 */

#pragma once

#include <cmath>

#include <nn/hid/hid_SixAxisSensor.h>

namespace nn { namespace hid { namespace detail {

const float Pi = 3.141592f; //!< 円周率の定義

/**
 * @brief       平方根を計算します
 *
 * @details     引数指定された値の平方根を返します。
 *
 * @param[in]   input                     負でない浮動小数点値
 *
 * @return      input の平方根を返します。
 *
 * @pre
 *              - input >= 0
 */
NN_FORCEINLINE float Sqrt( float input )
{
    NN_SDK_ASSERT(input >= 0);
    return std::sqrt(input);
}

/**
 * @brief       ベクトルをクランプします
 *
 * @details     引数指定された pOutVector をクランプします。
 *
 * @param[out]  pOutVector                クランプされるベクトル
 * @param[in]   clamp                     クランプ範囲
 *
 * @pre
 *              - pOutVector != nullptr
 */
void ClampVector( nn::util::Float3* pOutVector, const nn::util::Float3& clamp );

/**
 * @brief       ベクトルを正規化します
 *
 * @details     引数指定された pOutVector を正規化します。
 *              pOutVector のユークリッドノルムが 0.0f の場合は何もしません。
 *
 * @param[out]  pOutVector                正規化されるベクトル
 *
 * @return      正規化前の入力ベクトルのユークリッドノルムを返します。
 *
 * @pre
 *              - pOutVector != nullptr
 */
float Normalize( nn::util::Float3* pOutVector );

/**
 * @brief       DirectionState を擬似的に正規直交化します。
 *
 * @details     DirectionState の各軸ベクトルの重みづけを均等にしつつ正規直交化します。
 *
 * @param[out]  pOutDir                正規直交化される姿勢行列
 * @param[in]   Threshold              得られる姿勢行列の正規直交らしさの閾値。3.0f に近いほど正規直交系に近づく
 *
 * @pre
 *              - pOutDir != nullptr
 *              - pOutDir に指定される各軸ベクトルは右手系をなすこと
 *              - Threshold < 3.0f
 */
void NormalizeDirectionState( nn::hid::DirectionState *pOutDir, const float& Threshold );

/**
 * @brief       vec1 が vec2 に向く行列を作る。(TODO: 演算の意味の理解とリファクタリング)
 */
void MakeVecDir( nn::hid::DirectionState* pOutDir, const nn::util::Float3& vec1, const nn::util::Float3& vec2 );

/**
 * @brief       行列積を求めます。
 *
 * @details     Dir1 × Dir2 の結果を pOutDir に格納します。
 *
 * @param[out]  pOutDir                演算結果を格納する姿勢行列(= Dir1 × Dir2)
 * @param[in]   Dir1, Dir2             演算に利用する姿勢行列
 *
 * @pre
 *              - pOutDir != nullptr
 */
void  MultiplyDirection( nn::hid::DirectionState* pOutDir, const nn::hid::DirectionState& Dir1, const nn::hid::DirectionState& Dir2 );

/**
 * @brief       基本方向ベクトルに現在の方向ベクトルを近づけます。
 *
 * @details     方向ベクトルの変位(*pOutVec - lastVec) と補正力(revisePower) に応じて、
 *              現在の方向ベクトル(*pOutVec) を基本方向ベクトル (baseVec) に近づけます。
 *
 * @param[out]  pOutVec                最新サンプルの方向ベクトル
 * @param[in]   lastVec                前回サンプルの方向ベクトル
 * @param[in]   baseVec                基本方向ベクトル
 * @param[in]   revisePower            近づける強さ
 *
 * @return      補正のかかり具合を返します
 *
 * @pre
 *              - pOutVec != nullptr
 */
float ApproximateDirectionAxis(nn::util::Float3 *pOutVec,
                               const nn::util::Float3& lastVec,
                               const nn::util::Float3& baseVec,
                               const float& revisePower);

}}} // namespace nn::hid::detail
