﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

// ------------------------------------------------------------------------------------------------
// Build flags for tests to run
// ------------------------------------------------------------------------------------------------
#define RUN_BASIC
#define RUN_PEER_AND_NAME_VALIDATION
#define RUN_WITH_SESSION_CACHE
#define RUN_DOWNLOAD

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
//#define PRINT_INFO
//#define USE_WORKER_THREAD
//#define NO_RESOLVER

// ------------------------------------------------------------------------------------------------
// Macros
// ------------------------------------------------------------------------------------------------
#ifdef PRINT_INFO
#define MY_INFO_LOG NN_LOG
#else
#define MY_INFO_LOG(...)
#endif
#define BREAK_ON_FAILURE(result, isSuccess, ...) \
    if(result.IsFailure()) \
    { \
        NN_LOG("Failed (Description:%d) -  " __VA_ARGS__, result.GetDescription()); \
        isSuccess = false; \
        break; \
    }

#define START_PERF_TIMER(startTick) \
    do { \
        startTick = nn::os::GetSystemTick(); \
    } while (NN_STATIC_CONDITION(false))

#define STOP_PERF_TIMER(startTick, ...) \
    do { \
        nn::os::Tick endTick = nn::os::GetSystemTick(); \
        uint64_t milliSec = nn::os::ConvertToTimeSpan(endTick - startTick).GetMilliSeconds(); \
        uint64_t nanoSec = nn::os::ConvertToTimeSpan(endTick - startTick).GetNanoSeconds(); \
        NN_LOG("Perf: %8llu msec (%llu nsec) - " __VA_ARGS__, milliSec, nanoSec); \
    } while (NN_STATIC_CONDITION(false))

// ------------------------------------------------------------------------------------------------
// Grobal parameters
// ------------------------------------------------------------------------------------------------
namespace
{
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

const char* g_pFileName       = "10MB.bin";
const char* g_pHostName       = ServerName;
uint16_t    g_TestPortNumber  = ServerPort_Normal;
const char* g_pCaCert         = g_pTestCaCert;
uint32_t    g_CaCertSize      = sizeof(g_pTestCaCert);
const char* g_pPkcs12Data     = reinterpret_cast<const char*>(g_pTestClientPki);
const char* g_pPkcs12Password = g_pTestClientPkiPassword;
uint32_t    g_Pkcs12DataSize  = g_pTestClientPkiSize;
uint32_t    g_PasswordSize    = g_TestClientPkiPasswordLength;
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

int SetupTcpSocket(const char* pInHostName, uint16_t portNum)
{
    int            tcpSocket;
    nn::socket::InAddr inetAddr;

    memset(&inetAddr, 0x00, sizeof(inetAddr));

    if(pInHostName[0] == '\0')
    {
        NN_LOG("Host name is not set\n");
        return -1;
    }

    tcpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if(tcpSocket < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d)\n", errno);
        return -1;
    }
    MY_INFO_LOG("Created TCP socket (sockfd: %d).\n", tcpSocket);

    int buffSize = 1024 * 256;
    nn::socket::SockLenT optLen   = sizeof(buffSize);
    int rval = nn::socket::SetSockOpt(tcpSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &buffSize, optLen);
    if (rval < 0)
    {
        NN_LOG("Failed to set RCVBUF (errno:%d)\n", errno);
        return -1;
    }

    MY_INFO_LOG("Resolving %s\n", pInHostName);
    nn::socket::HostEnt *pHostEnt = nn::socket::GetHostEntByName(pInHostName);
    if(pHostEnt == nullptr)
    {
        NN_LOG("Failed to resolve host name (errno:%d)\n", errno);
        return -1;
    }

    // Just pick the first one
    memcpy(&inetAddr, pHostEnt->h_addr_list[0], sizeof(nn::socket::InAddr));

    nn::socket::SockAddrIn serverAddr;
    serverAddr.sin_addr.S_addr = inetAddr.S_addr;
    serverAddr.sin_family      = nn::socket::Family::Af_Inet;
    serverAddr.sin_port        = nn::socket::InetHtons(portNum);

    rval = nn::socket::Connect(
        tcpSocket,
        (nn::socket::SockAddr*)&serverAddr,
        sizeof(serverAddr));
    if(rval < 0)
    {
        NN_LOG("Failed to create TCP socket (errno: %d).\n", errno);
        return -1;
    }
    MY_INFO_LOG("Established TCP connection (addr:0x%x on port :%d).\n",
         serverAddr.sin_addr.S_addr, nn::socket::InetNtohs(serverAddr.sin_port));

    return tcpSocket;
}


#if defined(USE_WORKER_THREAD)
// ------------------------------------------------------------------------------------------------
// Threads
// ------------------------------------------------------------------------------------------------
namespace
{
const uint32_t NumberOfThreads = 1;
const size_t   ThreadStackSize = 1024 * 32;

NN_OS_ALIGNAS_THREAD_STACK uint8_t g_ThreadStack[ThreadStackSize];
nn::os::ThreadType                 g_ThreadTid;
nn::os::Event                      g_Done(nn::os::EventClearMode_ManualClear);
nn::os::Event                      g_CacheCreated(nn::os::EventClearMode_ManualClear);

void WorkerFunction(void* arg)
{
    bool                isSuccess = true;
    nn::Result          result = nn::ResultSuccess();
    nn::ssl::Context    sslContext;
    nn::ssl::Connection sslConnection;
    int                 socketFd = -1;

    //---------------------------------------------------------------------------------------------
    //  Perform SSL handshake
    //---------------------------------------------------------------------------------------------
    socketFd = SetupTcpSocket(g_pHostName, g_TestPortNumber);
    ASSERT_TRUE(socketFd >= 0);
    do
    {
        result = sslContext.Create(nn::ssl::Context::SslVersion_Auto);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create context\n");

        result = sslConnection.Create(&sslContext);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create connection\n");

        result = sslConnection.SetSocketDescriptor(socketFd);
        BREAK_ON_FAILURE(result, isSuccess, "SetSocketDescriptor\n");

        uint32_t hostNameLen = strlen(g_pHostName);
        result = sslConnection.SetHostName(g_pHostName, hostNameLen);
        BREAK_ON_FAILURE(result, isSuccess, "SetHostName\n");

        result = sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        BREAK_ON_FAILURE(result, isSuccess, "SetOption\n");
        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        BREAK_ON_FAILURE(result, isSuccess, "SetVerifyOption\n");

        result = sslConnection.DoHandshake();
        BREAK_ON_FAILURE(result, isSuccess, "DoHandshake\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    MY_INFO_LOG("Worker thread is signaling!\n");
    g_CacheCreated.Signal();
    MY_INFO_LOG("Worker thread is waiting....\n");
    g_Done.Wait();
    g_Done.Clear();

    //---------------------------------------------------------------------------------------------
    //  Cleanup
    //---------------------------------------------------------------------------------------------
    do
    {
        result = sslConnection.Destroy();
        BREAK_ON_FAILURE(result, isSuccess, "Connection::Destroy\n");

        result = sslContext.Destroy();
        BREAK_ON_FAILURE(result, isSuccess, "Context::Destroy\n");
    } while (NN_STATIC_CONDITION(false));
    MY_INFO_LOG("Worker thread is existing....\n");
}
}
#endif // USE_WORKER_THREAD

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    nn::os::Tick osTick;
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());

    START_PERF_TIMER(osTick);
    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
    STOP_PERF_TIMER(osTick, "ssl::Initialize\n");
}

#if defined(RUN_BASIC)
TEST(Basic, Success)
{
    bool                isSuccess = true;
    nn::Result          result = nn::ResultSuccess();
    nn::ssl::Context    sslContext;
    nn::ssl::Connection sslConnection;
    int                 socketFd = -1;
    nn::os::Tick        osTick;

    //---------------------------------------------------------------------------------------------
    //  Context setup
    //---------------------------------------------------------------------------------------------
    do
    {
        result = sslContext.Create(nn::ssl::Context::SslVersion_Auto);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create context\n");

        nn::ssl::CertStoreId serverCertStoreId;
        START_PERF_TIMER(osTick);
        result = sslContext.ImportServerPki(
            &serverCertStoreId,
            g_pCaCert,
            g_CaCertSize,
            nn::ssl::CertificateFormat_Pem);
        STOP_PERF_TIMER(osTick, "ImportServerPki\n");
        BREAK_ON_FAILURE(result, isSuccess, "ImportServerPki\n");

        nn::ssl::CertStoreId clientCertStoreId;
        START_PERF_TIMER(osTick);
        result = sslContext.ImportClientPki(
            &clientCertStoreId,
            g_pPkcs12Data,
            g_pPkcs12Password,
            g_Pkcs12DataSize,
            g_PasswordSize);
        STOP_PERF_TIMER(osTick, "ImportClientPki\n");
        BREAK_ON_FAILURE(result, isSuccess, "ImportClientPki\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    socketFd = SetupTcpSocket(g_pHostName, g_TestPortNumber);
    ASSERT_TRUE(socketFd >= 0);

    //---------------------------------------------------------------------------------------------
    //  Connection setup
    //---------------------------------------------------------------------------------------------
    do
    {
        result = sslConnection.Create(&sslContext);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create connection\n");

        result = sslConnection.SetSocketDescriptor(socketFd);
        BREAK_ON_FAILURE(result, isSuccess, "SetSocketDescriptor\n");

        uint32_t hostNameLen = static_cast<uint32_t>(strlen(g_pHostName));
        result = sslConnection.SetHostName(g_pHostName, hostNameLen);
        BREAK_ON_FAILURE(result, isSuccess, "SetHostName\n");

        result = sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        BREAK_ON_FAILURE(result, isSuccess, "SetOption\n");

        result = sslConnection.SetVerifyOption(
            nn::ssl::Connection::VerifyOption::VerifyOption_None);
        BREAK_ON_FAILURE(result, isSuccess, "SetVerifyOption\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  SSL handshake
    //---------------------------------------------------------------------------------------------
    do
    {
        START_PERF_TIMER(osTick);
        result = sslConnection.DoHandshake();
        STOP_PERF_TIMER(osTick, "DoHandshake without session cache\n");
        BREAK_ON_FAILURE(result, isSuccess, "DoHandshake\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  Flush session cache
    //---------------------------------------------------------------------------------------------
    do
    {
        START_PERF_TIMER(osTick);
        result = sslConnection.FlushSessionCache();
        STOP_PERF_TIMER(osTick, "Connection::FlushSessionCache\n");
        BREAK_ON_FAILURE(result, isSuccess, "Connection::FlushSessionCache\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  Cleanup
    //---------------------------------------------------------------------------------------------
    do
    {
        START_PERF_TIMER(osTick);
        result = sslConnection.Destroy();
        STOP_PERF_TIMER(osTick, "Connection::Destroy\n");
        BREAK_ON_FAILURE(result, isSuccess, "Connection::Destroy\n");

        START_PERF_TIMER(osTick);
        result = sslContext.Destroy();
        STOP_PERF_TIMER(osTick, "Context::Destroy\n");
        BREAK_ON_FAILURE(result, isSuccess, "Context::Destroy\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);
} // NOLINT(impl/function_size)
#endif // RUN_BASIC

#if defined(RUN_PEER_AND_NAME_VALIDATION)
TEST(DoHandshakeWithPeerCaHostNameValidation, Success)
{
    bool                isSuccess = true;
    nn::Result          result = nn::ResultSuccess();
    nn::ssl::Context    sslContext;
    nn::ssl::Connection sslConnection;
    int                 socketFd = -1;
    nn::os::Tick        osTick;

    //---------------------------------------------------------------------------------------------
    //  Setup
    //---------------------------------------------------------------------------------------------
    socketFd = SetupTcpSocket(g_pHostName, g_TestPortNumber);
    ASSERT_TRUE(socketFd >= 0);
    do
    {
        result = sslContext.Create(nn::ssl::Context::SslVersion_Auto);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create context\n");

        nn::ssl::CertStoreId serverCertStoreId;
        result = sslContext.ImportServerPki(
            &serverCertStoreId,
            g_pCaCert,
            g_CaCertSize,
            nn::ssl::CertificateFormat_Pem);
        BREAK_ON_FAILURE(result, isSuccess, "ImportServerPki\n");

        result = sslConnection.Create(&sslContext);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create connection\n");

        result = sslConnection.SetSocketDescriptor(socketFd);
        BREAK_ON_FAILURE(result, isSuccess, "SetSocketDescriptor\n");

        uint32_t hostNameLen = static_cast<uint32_t>(strlen(g_pHostName));
        result = sslConnection.SetHostName(g_pHostName, hostNameLen);
        BREAK_ON_FAILURE(result, isSuccess, "SetHostName\n");

        result = sslConnection.SetVerifyOption(
            nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa |
            nn::ssl::Connection::VerifyOption::VerifyOption_HostName);
        BREAK_ON_FAILURE(result, isSuccess, "SetVerifyOption\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  SSL handshake
    //---------------------------------------------------------------------------------------------
    do
    {
        START_PERF_TIMER(osTick);
        result = sslConnection.DoHandshake();
        STOP_PERF_TIMER(osTick, "DoHandshake (peerCa and HostName validations) without session cache\n");
        BREAK_ON_FAILURE(result, isSuccess, "DoHandshake\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  Cleanup
    //---------------------------------------------------------------------------------------------
    do
    {
        result = sslConnection.FlushSessionCache();
        BREAK_ON_FAILURE(result, isSuccess, "Connection::FlushSessionCache\n");

        result = sslConnection.Destroy();
        BREAK_ON_FAILURE(result, isSuccess, "Connection::Destroy\n");
        result = sslContext.Destroy();
        BREAK_ON_FAILURE(result, isSuccess, "Context::Destroy\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);
} // NOLINT(impl/function_size)
#endif // RUN_PEER_AND_NAME_VALIDATION


#if defined(RUN_WITH_SESSION_CACHE)
TEST(SessionCacheEnabled, Success)
{
    int                  i = 0;
    bool                 isSuccess = true;
    nn::Result           result = nn::ResultSuccess();
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;
    int                  socketFd = -1;
    nn::os::Tick         osTick;

    for (i = 0; i < 2; i++)
    {
        pSslContext = new nn::ssl::Context();
        ASSERT_TRUE(pSslContext != nullptr);

        pSslConnection = new nn::ssl::Connection();
        ASSERT_TRUE(pSslConnection != nullptr);

        socketFd = SetupTcpSocket(g_pHostName, g_TestPortNumber);
        ASSERT_TRUE(socketFd >= 0);
        do
        {
            result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
            BREAK_ON_FAILURE(result, isSuccess, "Failed to create context\n");
            result = pSslConnection->Create(pSslContext);
            BREAK_ON_FAILURE(result, isSuccess, "Failed to create connection\n");
            result = pSslConnection->SetSocketDescriptor(socketFd);
            BREAK_ON_FAILURE(result, isSuccess, "SetSocketDescriptor\n");
            uint32_t hostNameLen = static_cast<uint32_t>(strlen(g_pHostName));
            result = pSslConnection->SetHostName(g_pHostName, hostNameLen);
            BREAK_ON_FAILURE(result, isSuccess, "SetHostName\n");
            result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
            BREAK_ON_FAILURE(result, isSuccess, "SetOption\n");
            result = pSslConnection->SetVerifyOption(
                nn::ssl::Connection::VerifyOption::VerifyOption_None);
            BREAK_ON_FAILURE(result, isSuccess, "SetVerifyOption\n");
        } while (NN_STATIC_CONDITION(false));
        ASSERT_TRUE(isSuccess);

        do
        {
            START_PERF_TIMER(osTick);
            result = pSslConnection->DoHandshake();
            if (i > 0)
            {
                STOP_PERF_TIMER(osTick, "DoHandshake with session cache\n");
            }
            else
            {
                STOP_PERF_TIMER(osTick, "DoHandshake without session cache\n");
            }
            BREAK_ON_FAILURE(result, isSuccess, "DoHandshake\n");
        } while (NN_STATIC_CONDITION(false));
        ASSERT_TRUE(isSuccess);

        do
        {
            result = pSslConnection->Destroy();
            BREAK_ON_FAILURE(result, isSuccess, "Connection::Destroy\n");
            result = pSslContext->Destroy();
            BREAK_ON_FAILURE(result, isSuccess, "Context::Destroy\n");
        } while (NN_STATIC_CONDITION(false));
        ASSERT_TRUE(isSuccess);

        delete pSslConnection;
        delete pSslContext;

        pSslConnection = nullptr;
        pSslContext = nullptr;
        socketFd = -1;
    }
} // NOLINT(impl/function_size)
#endif // RUN_WITH_SESSION_CACHE

#ifdef RUN_DOWNLOAD
TEST(Download, Success)
{
    bool                isSuccess = true;
    nn::Result          result = nn::ResultSuccess();
    nn::ssl::Context    sslContext;
    nn::ssl::Connection sslConnection;
    int                 socketFd = -1;
    nn::os::Tick        osTick;

    //---------------------------------------------------------------------------------------------
    //  Context setup
    //---------------------------------------------------------------------------------------------
    do
    {
        result = sslContext.Create(nn::ssl::Context::SslVersion_Auto);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create context\n");

        nn::ssl::CertStoreId serverCertStoreId;
        START_PERF_TIMER(osTick);
        result = sslContext.ImportServerPki(
            &serverCertStoreId,
            g_pCaCert,
            g_CaCertSize,
            nn::ssl::CertificateFormat_Pem);
        STOP_PERF_TIMER(osTick, "ImportServerPki\n");
        BREAK_ON_FAILURE(result, isSuccess, "ImportServerPki\n");

        nn::ssl::CertStoreId clientCertStoreId;
        START_PERF_TIMER(osTick);
        result = sslContext.ImportClientPki(
            &clientCertStoreId,
            g_pPkcs12Data,
            g_pPkcs12Password,
            g_Pkcs12DataSize,
            g_PasswordSize);
        STOP_PERF_TIMER(osTick, "ImportClientPki\n");
        BREAK_ON_FAILURE(result, isSuccess, "ImportClientPki\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    socketFd = SetupTcpSocket(g_pHostName, g_TestPortNumber);
    ASSERT_TRUE(socketFd >= 0);

    //---------------------------------------------------------------------------------------------
    //  Connection setup
    //---------------------------------------------------------------------------------------------
    do
    {
        result = sslConnection.Create(&sslContext);
        BREAK_ON_FAILURE(result, isSuccess, "Failed to create connection\n");

        result = sslConnection.SetSocketDescriptor(socketFd);
        BREAK_ON_FAILURE(result, isSuccess, "SetSocketDescriptor\n");

        uint32_t hostNameLen = static_cast<uint32_t>(strlen(g_pHostName));
        result = sslConnection.SetHostName(g_pHostName, hostNameLen);
        BREAK_ON_FAILURE(result, isSuccess, "SetHostName\n");

        result = sslConnection.SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true);
        BREAK_ON_FAILURE(result, isSuccess, "SetOption\n");

        result = sslConnection.SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
        BREAK_ON_FAILURE(result, isSuccess, "SetVerifyOption\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  SSL handshake
    //---------------------------------------------------------------------------------------------
    do
    {
        START_PERF_TIMER(osTick);
        result = sslConnection.DoHandshake();
        STOP_PERF_TIMER(osTick, "DoHandshake without session cache\n");
        BREAK_ON_FAILURE(result, isSuccess, "DoHandshake\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  SSL Write
    //---------------------------------------------------------------------------------------------
    do
    {
        int sentBytes        = 0;
        char httpReqBuff[128] = {0};
        sprintf(httpReqBuff, "GET /%s HTTP/1.0\r\nHost: %s\r\n\r\n", g_pFileName, g_pHostName);
        START_PERF_TIMER(osTick);
        result = sslConnection.Write(httpReqBuff, &sentBytes, static_cast<uint32_t>(strlen(httpReqBuff)));
        STOP_PERF_TIMER(osTick, "Write with small HTTP request\n");
        BREAK_ON_FAILURE(result, isSuccess, "Write\n");
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  SSL Read
    //---------------------------------------------------------------------------------------------
    do
    {
        uint64_t  receivedTotal = 0;
        int  receivedBytes = 0;
        const int maxReadSize   = 1024 * 1024 * 1;
        char*     pReadBuffer   = new char[maxReadSize];

        ASSERT_TRUE(pReadBuffer != nullptr);

        START_PERF_TIMER(osTick);
        do {
            result = sslConnection.Read(
                pReadBuffer,
                &receivedBytes,
                maxReadSize);
            BREAK_ON_FAILURE(result, isSuccess, "Read\n");
            if(receivedBytes == 0)
            {
                MY_INFO_LOG("Connection closed by the server.\n");
                break;
            }
            receivedTotal += receivedBytes;
        } while (NN_STATIC_CONDITION(true));
        delete[] pReadBuffer;
        ASSERT_TRUE(isSuccess);

        nn::os::Tick endTick = nn::os::GetSystemTick();
        uint64_t milliSec = nn::os::ConvertToTimeSpan(endTick - osTick).GetMilliSeconds();
        uint64_t nanoSec = nn::os::ConvertToTimeSpan(endTick - osTick).GetNanoSeconds();

        if(milliSec == 0)
            milliSec = 1;

        uint64_t bytePerSec = (receivedTotal * 1000) / milliSec;
        uint64_t kbPerSec   = bytePerSec / 1024;
        NN_LOG("Perf: %8llu msec (%llu nsec) - Read %d bytes ",
            milliSec, nanoSec, receivedTotal);
        NN_LOG("[%lluKB/sec][%.2fMB/sec]\n",
            kbPerSec,
            (kbPerSec > 1024)?(static_cast<float>(static_cast<float>(kbPerSec) / 1024)):(0.0));
    } while (NN_STATIC_CONDITION(false));
    ASSERT_TRUE(isSuccess);

    //---------------------------------------------------------------------------------------------
    //  Cleanup
    //---------------------------------------------------------------------------------------------
    do
    {
        START_PERF_TIMER(osTick);
        result = sslConnection.Destroy();
        STOP_PERF_TIMER(osTick, "Connection::Destroy\n");
        BREAK_ON_FAILURE(result, isSuccess, "Connection::Destroy\n");

        START_PERF_TIMER(osTick);
        result = sslContext.Destroy();
        STOP_PERF_TIMER(osTick, "Context::Destroy\n");
        BREAK_ON_FAILURE(result, isSuccess, "Context::Destroy\n");
    } while (NN_STATIC_CONDITION(false));

    ASSERT_TRUE(isSuccess);
} // NOLINT(impl/function_size)
#endif // RUN_DOWNLOAD

TEST(FinalizeTest, Success)
{
    nn::os::Tick        osTick;

    START_PERF_TIMER(osTick);
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    STOP_PERF_TIMER(osTick, "ssl::Finalize\n");

    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
