﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>

// ------------------------------------------------------------------------------------------------
// Build flags for tests to run
// ------------------------------------------------------------------------------------------------
#define IMPORT_SERVER_PKI
#define VERIFY_PEER
#define VERIFY_HOST

// TODO: Remove this when time validation is supported on the hardware
#ifndef NN_BUILD_TARGET_PLATFORM_NX
#define VERIFY_DATE
#endif

// ------------------------------------------------------------------------------------------------
// Grobal parameters
// ------------------------------------------------------------------------------------------------
namespace
{
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

//-------------------------------------------------------------------------------------------------
//  Tests
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());

    ASSERT_TRUE(nn::ssl::Initialize().IsSuccess());
}

TEST(BlockingNoVerify, Success)
{
    nn::ssl::Connection::VerifyOption verifyOption =
        nn::ssl::Connection::VerifyOption::VerifyOption_None;
    SimpleHttpsClient httpsClient(
        false, // Blocking
        ServerName,
        ServerPort_Normal);
#ifdef NO_RESOLVER
    httpsClient.SetIpAddress(ServerIpAddress);
#endif
    EXPECT_TRUE(httpsClient.Initialize(
        nn::ssl::Context::SslVersion::SslVersion_Auto,
        verifyOption));
    EXPECT_TRUE(httpsClient.PerformSslHandshake(false));
    EXPECT_TRUE(httpsClient.SendHttpGetOverSsl());
    EXPECT_TRUE(httpsClient.ReceiveAllHttpData());
    httpsClient.Finalize();
}

TEST(Blocking, Success)
{
    nn::ssl::Connection::VerifyOption verifyOption =
        nn::ssl::Connection::VerifyOption::VerifyOption_None;

    verifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa |
                   nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
#if defined(VERIFY_DATE)
    verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;
#endif

    SimpleHttpsClient httpsClient(
        false, // Blocking
        ServerName,
        ServerPort_Normal);
#ifdef NO_RESOLVER
    httpsClient.SetIpAddress(ServerIpAddress);
#endif
    EXPECT_TRUE(httpsClient.Initialize(
        nn::ssl::Context::SslVersion::SslVersion_Auto,
        verifyOption));
    EXPECT_TRUE(httpsClient.ImportServerPki(
        g_pTestCaCert,
        sizeof(g_pTestCaCert),
        nn::ssl::CertificateFormat_Pem));
    EXPECT_TRUE(httpsClient.PerformSslHandshake(false));
    EXPECT_TRUE(httpsClient.SendHttpGetOverSsl());
    EXPECT_TRUE(httpsClient.ReceiveAllHttpData());
    httpsClient.Finalize();
}

TEST(NonBlocking, Success)
{
    nn::ssl::Connection::VerifyOption verifyOption =
        nn::ssl::Connection::VerifyOption::VerifyOption_None;

    verifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa |
                   nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
#if defined(VERIFY_DATE)
    verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;
#endif

    SimpleHttpsClient httpsClient(
        true, // Non-blocking
        ServerName,
        ServerPort_Normal);
#ifdef NO_RESOLVER
    httpsClient.SetIpAddress(ServerIpAddress);
#endif
    EXPECT_TRUE(httpsClient.Initialize(
        nn::ssl::Context::SslVersion::SslVersion_Auto,
        verifyOption));
    EXPECT_TRUE(httpsClient.ImportServerPki(
        g_pTestCaCert,
        sizeof(g_pTestCaCert),
        nn::ssl::CertificateFormat_Pem));
    EXPECT_TRUE(httpsClient.PerformSslHandshake(false));
    EXPECT_TRUE(httpsClient.SendHttpGetOverSsl());
    EXPECT_TRUE(httpsClient.ReceiveAllHttpData());
    httpsClient.Finalize();
}

TEST(FinalizeTest, Success)
{
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
