﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/ssl.h>
#include <nn/socket.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>
#include <Common/testClientPkiNoPwd.h>
// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
//#define NO_RESOLVER

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
namespace
{
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    EXPECT_TRUE(nn::ssl::Initialize().IsSuccess());
}

TEST(OnlyWithRootCa, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));

    nn::Result verifyResult;
    EXPECT_TRUE(pSslConnection->GetVerifyCertError(&verifyResult).IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult));

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(OnlyWithIntermediate, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult;
    EXPECT_TRUE(pSslConnection->GetVerifyCertError(&verifyResult).IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult));

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(WithIntermediateAndRoot, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(WithDirectServerCert, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestCaCert,
            sizeof(g_pTestCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(WithDirectCert, GetCert)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestCaCert,
            sizeof(g_pTestCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    char buf[4096];
    uint32_t certSize = 0;
    uint32_t certCount = 0;
    result = pSslConnection->SetServerCertBuffer(buf, sizeof(buf));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake(&certSize, &certCount);
    EXPECT_TRUE(result.IsSuccess());

    EXPECT_TRUE(certCount == 1);
    EXPECT_TRUE(certSize > 0);

    nn::ssl::Connection::ServerCertDetail detail;
    result = pSslConnection->GetServerCertDetail(&detail, buf, 0);
    EXPECT_FALSE(result.IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultNoServerChain::Includes(result));

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}

TEST(WithDirectCert, GetChainOldApi)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestCaCert,
            sizeof(g_pTestCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_GetServerCertChain, true);
    EXPECT_TRUE(result.IsSuccess());

    char buf[4096];
    uint32_t certSize = 0;
    uint32_t certCount = 0;
    result = pSslConnection->SetServerCertBuffer(buf, sizeof(buf));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake(&certSize, &certCount);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(certCount == 1);
    EXPECT_TRUE(certSize > 0);

    nn::ssl::Connection::ServerCertDetail detail;
    result = pSslConnection->GetServerCertDetail(&detail, buf, 0);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(detail.dataSize > 0);
    EXPECT_TRUE(detail.pDerData > &buf[0]);
    EXPECT_TRUE(detail.pDerData < &buf[detail.dataSize]);

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(WithDirectCert, GetChainNewApi)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestCaCert,
            sizeof(g_pTestCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_GetServerCertChain, true);
    EXPECT_TRUE(result.IsSuccess());

    char buf[4096];
    uint32_t certSize = 0;
    uint32_t certCount = 0;
    result = pSslConnection->DoHandshake(&certSize, &certCount, buf, sizeof(buf));
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(certCount == 1);
    EXPECT_TRUE(certSize > 0);

    nn::ssl::Connection::ServerCertDetail detail;
    result = pSslConnection->GetServerCertDetail(&detail, buf, 0);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(detail.dataSize > 0);
    EXPECT_TRUE(detail.pDerData > &buf[0]);
    EXPECT_TRUE(detail.pDerData < &buf[detail.dataSize]);

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(NoCrossContext1, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext1 = new nn::ssl::Context();
    nn::ssl::Context*    pSslContext2 = new nn::ssl::Context();

    result = pSslContext1->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection1 = new nn::ssl::Connection();;
    result = pSslConnection1->Create(pSslContext1);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection2 = new nn::ssl::Connection();;
    result = pSslConnection2->Create(pSslContext2);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd1 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    int socketFd2 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection1->SetSocketDescriptor(socketFd1);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection1->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->SetSocketDescriptor(socketFd2);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult2;
    EXPECT_TRUE(pSslConnection2->GetVerifyCertError(&verifyResult2).IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult2));

    result = pSslConnection2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(NoCrossContext2, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext1 = new nn::ssl::Context();
    nn::ssl::Context*    pSslContext2 = new nn::ssl::Context();

    result = pSslContext1->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection1 = new nn::ssl::Connection();;
    result = pSslConnection1->Create(pSslContext1);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection2 = new nn::ssl::Connection();;
    result = pSslConnection2->Create(pSslContext2);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd1 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    int socketFd2 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection1->SetSocketDescriptor(socketFd1);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection1->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->SetSocketDescriptor(socketFd2);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult2;
    EXPECT_TRUE(pSslConnection2->GetVerifyCertError(&verifyResult2).IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult2));

    result = pSslConnection2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(NoCrossContext3, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext1 = new nn::ssl::Context();
    nn::ssl::Context*    pSslContext2 = new nn::ssl::Context();

    result = pSslContext1->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection1 = new nn::ssl::Connection();;
    result = pSslConnection1->Create(pSslContext1);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection2 = new nn::ssl::Connection();;
    result = pSslConnection2->Create(pSslContext2);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd1 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    int socketFd2 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection1->SetSocketDescriptor(socketFd1);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    //  Should FAIL because context#1 doesn't have the root CA
    result = pSslConnection1->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult1;
    EXPECT_TRUE(pSslConnection1->GetVerifyCertError(&verifyResult1).IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult1));

    result = pSslConnection2->SetSocketDescriptor(socketFd2);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    //  Should SUCCEED because context#2 has the root CA and context#1 has
    //  made the intermediate signing CA "known" to the certstore.
    result = pSslConnection2->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(DupInstallCertCrossContext, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext1 = new nn::ssl::Context();
    nn::ssl::Context*    pSslContext2 = new nn::ssl::Context();

    result = pSslContext1->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext1);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    //  Install the cert in context#2 first, should get re-used behind
    //  the scenes within SSL.
    result = pSslContext2->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext1->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(DupInstallDeleteCertCrossContext, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext1 = new nn::ssl::Context();
    nn::ssl::Context*    pSslContext2 = new nn::ssl::Context();

    result = pSslContext1->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection1 = new nn::ssl::Connection();;
    result = pSslConnection1->Create(pSslContext1);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection2 = new nn::ssl::Connection();;
    result = pSslConnection2->Create(pSslContext2);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId1;
    nn::ssl::CertStoreId certStoreId2;
    nn::ssl::CertStoreId certStoreId3;
    nn::ssl::CertStoreId certStoreId4;

    result = pSslContext1->ImportServerPki(
            &certStoreId1,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    //  Install the cert in context#2 first, should get re-used behind
    //  the scenes within SSL.
    result = pSslContext2->ImportServerPki(
            &certStoreId2,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext1->ImportServerPki(
            &certStoreId3,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    //  Don't forget the intermediate on context#2 so it will work
    result = pSslContext2->ImportServerPki(
            &certStoreId4,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    int socketFd2 = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection1->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection1->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection1->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    //  Remove shared cert from context#1.  Make sure connection#2 can still
    //  succeed to handshake.
    result = pSslContext1->RemovePki(certStoreId3);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->SetSocketDescriptor(socketFd2);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection2->DoHandshake();
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->RemovePki(certStoreId2);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext1->RemovePki(certStoreId1);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext2->RemovePki(certStoreId4);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext2->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext1->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(VerifyCertRemove, Success)
{
    nn::Result result;

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();
    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    nn::ssl::CertStoreId certStoreId;
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestSigningCaCert,
            sizeof(g_pTestSigningCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->ImportServerPki(
            &certStoreId,
            g_pTestRootCaCert,
            sizeof(g_pTestRootCaCert),
            nn::ssl::CertificateFormat_Pem);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslContext->RemovePki(certStoreId);
    EXPECT_TRUE(result.IsSuccess());

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
#ifndef NO_RESOLVER
        ServerName,
        0
#else
        nullptr,
        ServerIpAddress
#endif
    );

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultVerifyCertFailed::Includes(result));
    nn::Result verifyResult;
    EXPECT_TRUE(pSslConnection->GetVerifyCertError(&verifyResult).IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSslErrorUnkownCa::Includes(verifyResult));

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
}


TEST(FinalizeTest, Success)
{
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
