﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// ------------------------------------------------------------------------------------------------
// Macros
// ------------------------------------------------------------------------------------------------
#ifdef NN_BUILD_CONFIG_OS_WIN
#define MY_SNPRINTF _snprintf
#else
#define MY_SNPRINTF snprintf
#endif

#define EXPECT_TRUE_AND_LOG(CONDITION, RESULT) \
do                                             \
{                                              \
    if( (CONDITION) == false )                 \
    {                                          \
        NN_LOG("\n * Error Line: %d. Failing Condition: '%s'. Result Description: %d\n\n", __LINE__, #CONDITION, RESULT.GetDescription()); \
        FAIL();                                \
    }                                          \
} while(NN_STATIC_CONDITION(false))

#define CHECK_HANDSHAKE_RESULT(CONDITION, RESULT, SSL_CONNECTION_PTR)  \
do                                                                     \
{                                                                      \
    if( (CONDITION) == false )                                         \
    {                                                                  \
        NN_LOG("\n * Error Line: %d. Failing Condition: '%s'. Result Description: %d\n\n", __LINE__, #CONDITION, RESULT.GetDescription()); \
        if( nn::ssl::ResultVerifyCertFailed::Includes(RESULT) )        \
        {                                                              \
            nn::Result verifyResult = nn::ResultSuccess();             \
            nn::Result tempResult = SSL_CONNECTION_PTR->GetVerifyCertError(&verifyResult); \
            if(tempResult.IsSuccess())                                 \
            {                                                          \
                NN_LOG(" * Verify Result Description: %d\n\n", verifyResult.GetDescription()); \
            }                                                          \
            else                                                       \
            {                                                          \
                NN_LOG("\n * Error: Failed to GetVerifyCertError(). Desc: %d\n\n", tempResult.GetDescription()); \
            }                                                          \
        }                                                              \
        FAIL();                                                        \
    }                                                                  \
} while(NN_STATIC_CONDITION(false))

// ------------------------------------------------------------------------------------------------
// Test result params
// ------------------------------------------------------------------------------------------------
typedef struct TestParam
{
    bool     isDone;
    uint32_t id;
    char     fromName[32];
    char     apiName[128];
} TestParam;

namespace
{
    // ------------------------------------------------------------------------------------------------
    // Grobal functions
    // ------------------------------------------------------------------------------------------------
    void SetTestDoneFlag(TestParam* pTestArray, int testIdCount, uint32_t id, bool condition)
    {
        if (condition == false)
        {
            return;
        }

        for (int i = 0; i < testIdCount; i++)
        {
            if (pTestArray[i].id == id)
            {
                pTestArray[i].isDone = true;
                break;
            }
        }
    }

    void DumpTestResult(TestParam* pTestArray, int testIdCount)
    {
        bool isOk = true;

        NN_LOG("[Test Result]\n");
        for (int i = 0; i < testIdCount; i++)
        {
            NN_LOG("|%s|ID: %d|From: %s\t|%s\n",
                (pTestArray[i].isDone == true)?"OK":"NG",
                pTestArray[i].id,
                pTestArray[i].fromName,
                pTestArray[i].apiName);

            if (pTestArray[i].isDone == false)
            {
                isOk = false;
            }
        }
        NN_LOG("------------------\n");
        NN_LOG("| Result: %s\n", (isOk)?"OK":"NG");
        NN_LOG("------------------\n");

        EXPECT_TRUE(isOk == true);
    }

    void SetupSsl(
        nn::ssl::Context* pInSslContext,
        nn::ssl::Connection* pInSslConnection,
        int socketFd,
        const char* pInHostName)
    {
        nn::Result result;

        EXPECT_TRUE(pInSslContext != nullptr);
        EXPECT_TRUE(pInSslConnection != nullptr);
        result = pInSslContext->Create(nn::ssl::Context::SslVersion_Auto);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        result = pInSslConnection->Create(pInSslContext);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        result = pInSslConnection->SetSocketDescriptor(socketFd);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        result = pInSslConnection->SetHostName(pInHostName, static_cast<uint32_t>(strlen(pInHostName)));
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        result = pInSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    }

    void CleanupSsl(
        nn::ssl::Context* pInSslContext,
        nn::ssl::Connection* pInSslConnection)
    {
        nn::Result result;

        EXPECT_TRUE(pInSslContext != nullptr);
        EXPECT_TRUE(pInSslConnection != nullptr);

        result = pInSslConnection->Destroy();
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        delete pInSslConnection;
        result = pInSslContext->Destroy();
        EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
        delete pInSslContext;
    }
}
