﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Base.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <cstring>
#include <nn/svc/svc_Base.h>
#include <nn/ro/detail/ro_NroHeader.h>

#include <nn/ro.h>

#include "util_TestNro.h"
#include "util_TestResult.h"
#include "util_TestFsManager.h"

TestNro::TestNro() NN_NOEXCEPT
    : m_Name(nullptr)
    , m_Nro(nullptr)
    , m_Bss(nullptr)
    , m_Nrr(nullptr)
    , m_ImageSize(0)
    , m_BufferSize(0)
    , m_NrrSize(0)
    , m_Allocator(nullptr)
{
    m_Module._state = nn::ro::Module::State_Unloaded;
    m_Info._state = nn::ro::RegistrationInfo::State_Unregistered;
}

TestNro::TestNro(const char* name, nn::mem::StandardAllocator* allocator) NN_NOEXCEPT
    : m_Name(name)
    , m_Nro(nullptr)
    , m_Bss(nullptr)
    , m_Nrr(nullptr)
    , m_ImageSize(0)
    , m_BufferSize(0)
    , m_NrrSize(0)
    , m_Allocator(allocator)
{
    m_Module._state = nn::ro::Module::State_Unloaded;
    m_Info._state = nn::ro::RegistrationInfo::State_Unregistered;
}

void TestNro::SetUp(const char* name, nn::mem::StandardAllocator* allocator) NN_NOEXCEPT
{
    m_Name = name;
    m_Allocator = allocator;
}

TestNro::~TestNro() NN_NOEXCEPT
{
    Unload();
}

nn::Result TestNro::Load(int flag, const char* nrrName) NN_NOEXCEPT
{
    TestFsManager& fs = TestFsManager::GetInstance();

    m_Nrr = m_Allocator->Allocate(MaxFileSize, DefaultAlign);
    m_NrrSize = fs.ReadAll(m_Nrr, MaxFileSize, (nrrName == nullptr) ? NrrName : nrrName);

    auto result = nn::ro::RegisterModuleInfo(&m_Info, m_Nrr);
    NN_ASSERT_RESULT_SUCCESS(result);

    m_Nro = m_Allocator->Allocate(MaxFileSize, DefaultAlign);
    m_ImageSize = fs.ReadAll(m_Nro, MaxFileSize, m_Name);

    result = nn::ro::GetBufferSize(&m_BufferSize, m_Nro);
    NN_ASSERT_RESULT_SUCCESS(result);

    if (m_BufferSize != 0)
    {
        m_Bss = m_Allocator->Allocate(m_BufferSize, DefaultAlign);
    }

    nn::svc::MemoryInfo info;
    nn::svc::PageInfo pageInfo;
    result = nn::svc::QueryMemory(&info, &pageInfo, reinterpret_cast<uintptr_t>(m_Nro));
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT_EQUAL(info.state, nn::svc::MemoryState_Normal);
    NN_ASSERT_EQUAL(info.permission, nn::svc::MemoryPermission_ReadWrite);
    NN_ASSERT_EQUAL(info.attribute, 0);

    if (m_Bss != nullptr)
    {
        result = nn::svc::QueryMemory(&info, &pageInfo, reinterpret_cast<uintptr_t>(m_Bss));
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT_EQUAL(info.state, nn::svc::MemoryState_Normal);
        NN_ASSERT_EQUAL(info.permission, nn::svc::MemoryPermission_ReadWrite);
        NN_ASSERT_EQUAL(info.attribute, 0);
    }

    result = nn::ro::LoadModule(&m_Module, m_Nro, m_Bss, m_BufferSize, flag);

    if (result.IsSuccess())
    {
        nn::svc::QueryMemory(&info, &pageInfo, reinterpret_cast<uintptr_t>(m_Nro));
        NN_ASSERT_RESULT_SUCCESS(result);
        NN_ASSERT_EQUAL(info.state, nn::svc::MemoryState_Normal);
        NN_ASSERT_EQUAL(info.permission, nn::svc::MemoryPermission_None);
        NN_ASSERT_EQUAL(info.attribute, nn::svc::MemoryAttribute_Locked);

        if (m_Bss != nullptr)
        {
            nn::svc::QueryMemory(&info, &pageInfo, reinterpret_cast<uintptr_t>(m_Bss));
            NN_ASSERT_RESULT_SUCCESS(result);
            NN_ASSERT_EQUAL(info.state, nn::svc::MemoryState_Normal);
            NN_ASSERT_EQUAL(info.permission, nn::svc::MemoryPermission_None);
            NN_ASSERT_EQUAL(info.attribute, nn::svc::MemoryAttribute_Locked);
        }
    }

    return result;

}

void TestNro::Unload() NN_NOEXCEPT
{
    if (m_Module._state == nn::ro::Module::State_Loaded)
    {
        nn::ro::UnloadModule(&m_Module);
    }

    if (m_Info._state == nn::ro::RegistrationInfo::State_Registered)
    {
        nn::ro::UnregisterModuleInfo(&m_Info);
    }

    if (m_Bss)
    {
        m_Allocator->Free(m_Bss);
        m_Bss = nullptr;
    }
    if (m_Nro)
    {
        m_Allocator->Free(m_Nro);
        m_Nro = nullptr;
    }
    if (m_Nrr)
    {
        m_Allocator->Free(m_Nrr);
        m_Nrr = nullptr;
    }
}

nn::Result TestNro::FindSymbol(uintptr_t* pOutValue, const char* name) const NN_NOEXCEPT
{
    return nn::ro::LookupModuleSymbol(pOutValue, &m_Module, name);
}
