﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/psc/psc_Types.h>
#include <nn/psm/psm_SystemProcessApi.h>

#include <nnt/nntest.h>

#include "testPsm_ChargerDriverStub.h"
#include "testPsm_Common.h"
#include "testPsm_SplStub.h"

namespace nnt { namespace psm {
namespace {

struct PsmHiZPolicyTestSetting
{
    //! USB 電源関連の状態
    ::nnt::psm::UsbPowerState usbPowerState;

    //! コントローラーへの給電要求
    bool isControllerPowerSupplyAcquired;

    //! 充電完了通知がある
    bool isChargeTerminated;

    //! Hi-Z への遷移の期待値
    bool hiZTransition;
};

const PsmHiZPolicyTestSetting PsmHiZPolicyTestSettings[] =
{
    { ::nnt::psm::UsbPowerState::Sdp500Ma, false, true,  true  },
    { ::nnt::psm::UsbPowerState::Sdp500Ma, false, false, false },
    { ::nnt::psm::UsbPowerState::Sdp500Ma, true,  true,  false },
    { ::nnt::psm::UsbPowerState::Inactive, false, true,  false },
};

class PsmHiZPolicyTest : public ::testing::TestWithParam<PsmHiZPolicyTestSetting>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmHiZPolicyTestName, PsmHiZPolicyTest, ::testing::ValuesIn(PsmHiZPolicyTestSettings));

} // namespace

// 1. FullAwake にします。
// 2. 十分な給電能力のある USB の対向を設定します。
// 3. コントローラの給電要求を明示的に止めます。
// 4. チャージャーの満充電フラグを立てます。
// 5. SleepReady にします。
// 6. SPL のスタブで Hi-Z 遷移要求が来ていることを確認します。

TEST_P(PsmHiZPolicyTest, Sequence)
{
    TransitPmState(::nn::psc::PmState_FullAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);

    ChangeUsbPowerState(GetParam().usbPowerState);
    if ( GetParam().isControllerPowerSupplyAcquired )
    {
        ::nn::psm::AcquireControllerPowerSupply();
    }
    else
    {
        ::nn::psm::ReleaseControllerPowerSupply();
    }
    if ( GetParam().isChargeTerminated )
    {
        driver::detail::SetStatus(::nn::psm::driver::detail::ChargerDriver::Status::ChargeTermination);
    }
    else
    {
        driver::detail::SetStatus(::nn::psm::driver::detail::ChargerDriver::Status::FastCharge);
    }

    TransitPmState(::nn::psc::PmState_SleepReady);

    if ( GetParam().hiZTransition )
    {
        EXPECT_EQ(1, ::nnt::spl::GetIsChargerHiZModeEnabled());
        driver::detail::SetHiZCondition(true);

        TransitPmState(::nn::psc::PmState_MinimumAwake);

        EXPECT_EQ(0, ::nnt::spl::GetIsChargerHiZModeEnabled());
        EXPECT_EQ(false, driver::detail::GetHiZCondition());
    }
    else
    {
        EXPECT_EQ(0, ::nnt::spl::GetIsChargerHiZModeEnabled());

        TransitPmState(::nn::psc::PmState_MinimumAwake);

        EXPECT_EQ(0, ::nnt::spl::GetIsChargerHiZModeEnabled());
        EXPECT_EQ(false, driver::detail::GetHiZCondition());
    }

    TransitPmState(::nn::psc::PmState_FullAwake);
    ChangeUsbPowerState(UsbPowerState::Inactive);
}

}} // namespace nnt::psm
