﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/gpio/gpio.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntrusiveList.h>

namespace {

int g_StubInitialized = 0;

struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

class SessionContext : public nn::util::IntrusiveListBaseNode<SessionContext>
{
public:
    SessionContext() NN_NOEXCEPT
    : m_InterruptEnabled(false)
    , m_Name(static_cast<nn::gpio::GpioPadName>(0))
    , m_InterruptMode(nn::gpio::InterruptMode_LowLevel)
    , m_Direction(nn::gpio::Direction_Input)
    , m_Value(nn::gpio::GpioValue_Low)
    , m_InterruptStatus(nn::gpio::InterruptStatus_Inactive)
    {
    }

    void Initialize(nn::gpio::GpioPadName padName) NN_NOEXCEPT
    {
        m_InterruptEnabled = false;
        m_Name = padName;

        m_InterruptMode   = nn::gpio::InterruptMode_LowLevel;
        m_Direction       = nn::gpio::Direction_Input;
        m_Value           = nn::gpio::GpioValue_Low;
        m_InterruptStatus = nn::gpio::InterruptStatus_Inactive;

        nn::os::CreateSystemEvent(&m_SystemEvent, nn::os::EventClearMode_ManualClear, true);
    }

    void Finalize() NN_NOEXCEPT
    {
        nn::os::DestroySystemEvent(&m_SystemEvent);
    }

public:
    bool m_InterruptEnabled;
    nn::gpio::GpioPadName m_Name;

    nn::gpio::InterruptMode   m_InterruptMode;
    nn::gpio::Direction       m_Direction;
    nn::gpio::GpioValue       m_Value;
    nn::gpio::InterruptStatus m_InterruptStatus;

    nn::os::SystemEventType m_SystemEvent;
};

nn::util::IntrusiveList<SessionContext, nn::util::IntrusiveListBaseNodeTraits<SessionContext>> g_SessionContextList;

SessionContext* FindSessionContextInList(SessionContext* pSessionContext) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( &(*itr) == pSessionContext )
        {
            return &(*itr);
        }
    }

    return nullptr;
}

SessionContext* FindSessionContextInList(nn::gpio::GpioPadName name) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( (*itr).m_Name == name )
        {
            return &(*itr);
        }
    }

    return nullptr;
}

const size_t MaxSessionContexts = 30;

SessionContext* AddSessionContextToList(nn::gpio::GpioPadName name) NN_NOEXCEPT
{
    NN_ASSERT(g_SessionContextList.size() < MaxSessionContexts)

    // 簡単にするため同一 GpioPadName の SessionContext は存在しないものとする。
    NN_ASSERT(FindSessionContextInList(name) == nullptr);

    SessionContext* pSessionContext = new SessionContext();
    pSessionContext->Initialize(name);

    g_SessionContextList.push_front(*pSessionContext);

    return pSessionContext;
}

bool RemoveSessionContextFromList(SessionContext* pSessionContext) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( &(*itr) == pSessionContext )
        {
            (*itr).Finalize();
            g_SessionContextList.erase(itr);
            delete pSessionContext;
            return true;
        }
    }

    return false;
}

SessionContext* FindSessionContextInList(nn::gpio::GpioPadSession* pSession) NN_NOEXCEPT
{
    return FindSessionContextInList(static_cast<SessionContext*>(pSession->_sessionHandle));
}

void AddSessionContextToList(nn::gpio::GpioPadSession* pOutSession, nn::gpio::GpioPadName name) NN_NOEXCEPT
{
    pOutSession->_sessionHandle = static_cast<void*>(AddSessionContextToList(name));
    pOutSession->_pEvent = nullptr;

    NN_ASSERT_NOT_NULL(pOutSession->_sessionHandle);
}

void RemoveSessionContextFromList(nn::gpio::GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pSession);

    NN_ASSERT(RemoveSessionContextFromList(static_cast<SessionContext*>(pSession->_sessionHandle)));
}

void BindEvent(nn::os::SystemEventType* pEvent, nn::gpio::GpioPadSession* pSession) NN_NOEXCEPT
{
    SessionContext* pSessionContext = FindSessionContextInList(pSession);
    NN_ASSERT_NOT_NULL(pSessionContext);

    NN_ASSERT(pSession->_pEvent == nullptr);
    NN_ASSERT_NOT_NULL(pEvent);

    nn::os::NativeHandle handle = nn::os::GetReadableHandleOfSystemEvent(&(pSessionContext->m_SystemEvent));
    nn::os::AttachReadableHandleToSystemEvent(pEvent, handle, false, nn::os::EventClearMode_ManualClear);
    pSession->_pEvent = pEvent;
}

void UnbindEvent(nn::gpio::GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(FindSessionContextInList(pSession));
    NN_ASSERT_NOT_NULL(pSession->_pEvent);

    nn::os::DestroySystemEvent(pSession->_pEvent);
    pSession->_pEvent = nullptr;
}

} // namespace

namespace nn { namespace gpio {

void Initialize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    if ( g_StubInitialized == 0 )
    {
        NN_ASSERT(g_SessionContextList.empty());
    }

    g_StubInitialized++;
}

void Finalize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized > 0);

    g_StubInitialized--;

    if ( g_StubInitialized == 0 )
    {
        NN_ASSERT(g_SessionContextList.empty());
    }
}

void OpenSession(GpioPadSession* pOutSession, GpioPadName pad) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pOutSession);

    AddSessionContextToList(pOutSession, pad);
}

void CloseSession(GpioPadSession* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    RemoveSessionContextFromList(pSession);
}

void SetDirection(GpioPadSession* pSession, Direction direction) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    SessionContext* pSessionContext = FindSessionContextInList(pSession);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->m_Direction = direction;
}

void SetInterruptMode(GpioPadSession* pSession, InterruptMode mode) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    SessionContext* pSessionContext = FindSessionContextInList(pSession);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->m_InterruptMode = mode;
}

void SetInterruptEnable(GpioPadSession* pSession, bool enable) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    SessionContext* pSessionContext = FindSessionContextInList(pSession);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->m_InterruptEnabled = enable;
}

void ClearInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    NN_UNUSED(pSession);
}

void SetValue(GpioPadSession* pSession, GpioValue value) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    SessionContext* pSessionContext = FindSessionContextInList(pSession);
    NN_ASSERT_NOT_NULL(pSessionContext);

    pSessionContext->m_Value = value;
}

Result BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);
    NN_ASSERT_NOT_NULL(pEvent);

    BindEvent(pEvent, pSession);

    NN_RESULT_SUCCESS;
}

void UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);
    NN_ASSERT_NOT_NULL(pSession);

    UnbindEvent(pSession);
}

}} // namespace nn::gpio

namespace nnt { namespace gpio {

bool CheckSessionContext(nn::gpio::GpioPadName padName, nn::gpio::Direction direction, nn::gpio::GpioValue value) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);

    SessionContext* pSessionContext = FindSessionContextInList(padName);

    NN_ASSERT_NOT_NULL(pSessionContext);

    if ( !((pSessionContext->m_Direction == direction) && (pSessionContext->m_Value == value)) )
    {
        NN_LOG("name:      %d\n",    padName);
        NN_LOG("direction: expected(%d), actual(%d)\n", direction, pSessionContext->m_Direction);
        NN_LOG("value:     expected(%d), actual(%d)\n", value, pSessionContext->m_Value);
    }

    return (pSessionContext->m_Direction == direction) && (pSessionContext->m_Value == value);
}

void Signal(nn::gpio::GpioPadName padName) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT(g_StubInitialized);

    SessionContext* pSessionContext = FindSessionContextInList(padName);

    nn::os::SignalSystemEvent(&(pSessionContext->m_SystemEvent));
}

}} // namespace nnt::gpio

