﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/psc.h>

#include <nn/psm/driver/psm.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_Internal.h>

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
#include <nn/psm/psm_ManufactureInternal.h>
#include <nn/psm/psm_PsmManufactureServer.h>
#endif

#include <nn/psm/psm_PsmServer.h>
#include <nn/psm/psm_SystemProcessApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/usb/pd/usb_PdType.h>
#include <nn/usb/usb_PmTypes.h>

#include <nnt/nntest.h>

#include "testPsm_ChargerDriverStub.h"
#include "testPsm_Common.h"
#include "testPsm_FuelGaugeDriverStub.h"
#include "testPsm_GpioStub.h"
#include "testPsm_UsbPdStub.h"
#include "testPsm_UsbPmStub.h"

namespace nnt { namespace psm {

namespace {

// 確実な同期のためにハンドリング完了イベントへのポインタを使用する。
::nn::os::EventType* g_pEventPsc;
::nn::os::EventType* g_pEventFuelGauge;
::nn::os::EventType* g_pEventFuelGaugeTimer;
::nn::os::EventType* g_pEventUsbPd;
::nn::os::EventType* g_pEventUsbPm;
::nn::os::EventType* g_pEventEnoughPowerChargeEmulation;
::nn::psc::PmModule* g_pPmModule;

::nn::sf::UnmanagedServiceObject<::nn::psm::IPsmServer, ::nn::psm::PsmServer> g_Server;

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
::nn::sf::UnmanagedServiceObject<::nn::psm::IPsmManufactureServer, ::nn::psm::PsmManufactureServer> g_ManufactureServer;
#endif

::nn::psc::PmState g_CurrentState = ::nn::psc::PmState_FullAwake;

struct PreviousStatus
{
    int batteryVoltageMilliVolt;
    int averageCurrentMilliAmpere;
    double batteryChargePercentage;
    double voltageFuelGaugePercentage;

    //! すべての前状態を無効な値（負数）にする
    void Reset() NN_NOEXCEPT
    {
        batteryVoltageMilliVolt = -1;
        averageCurrentMilliAmpere = -1;
        batteryChargePercentage = -1.0;
        voltageFuelGaugePercentage = -1.0;
    }
}
g_PreviousStatus;

void InitializeEventPointers() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetPscEventForTest(&g_pEventPsc));
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetFuelGaugeEventForTest(&g_pEventFuelGauge));
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetFuelGaugeTimerEventForTest(&g_pEventFuelGaugeTimer));
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetUsbPdEventForTest(&g_pEventUsbPd));
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetUsbPmEventForTest(&g_pEventUsbPm));
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetEnoughPowerChargeEmulationEventForTest(&g_pEventEnoughPowerChargeEmulation));
    NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::psm::driver::GetPmModuleForTest(&g_pPmModule));
}

void TimedWaitAndClearEvent(::nn::os::EventType* pEvent) NN_NOEXCEPT
{
    // PSM ライブラリ内部の 10 ミリ秒ごとの TimerEvent を待つ為に 10 ミリ秒以上の十分長い値を設定します。
    EXPECT_TRUE(::nn::os::TimedWaitEvent(pEvent, ::nn::TimeSpan::FromMilliSeconds(2000)));
    ::nn::os::ClearEvent(pEvent);
}

void WaitAndClearEvent(::nn::os::EventType* pEvent) NN_NOEXCEPT
{
    ::nn::os::WaitEvent(pEvent);
    ::nn::os::ClearEvent(pEvent);
}

void GeneratePowerDeliveryStatus(::nn::usb::pd::Status* pOutStatus, const UsbPowerState usbPowerState) NN_NOEXCEPT
{
    pOutStatus->Clear();

    uint32_t objectPosition = 0;
    uint32_t maximumOperatingCurrent = 0;
    uint32_t maximumCurrent = 0;
    uint32_t voltage = 0;

    bool active = false;
    ::nn::usb::pd::StatusError statusError = ::nn::usb::pd::StatusError_None;
    ::nn::usb::pd::StatusPowerRole statusPowerRole = ::nn::usb::pd::StatusPowerRole_Sink;
    ::nn::usb::pd::StatusRequest statusRequest = ::nn::usb::pd::StatusRequest_None;

    switch ( usbPowerState )
    {
    case UsbPowerState::He15V:
        objectPosition = 0x2;
        maximumOperatingCurrent = 3000; // Ilim should be 1200
        maximumCurrent = 0;
        voltage = 15000;
        active = true;
        break;
    case UsbPowerState::He13V:
        objectPosition = 0x2;
        maximumOperatingCurrent = 3000; // Ilim should be 1500
        maximumCurrent = 0;
        voltage = 13000;
        active = true;
        break;
    case UsbPowerState::Lt13V:
        objectPosition = 0x2;
        maximumOperatingCurrent = 3000; // Ilim should be 2000
        maximumCurrent = 0;
        voltage = 12000; // 13V 未満としてとりあえず 12V を設定する。
        active = true;
        break;
    case UsbPowerState::Dcp1500Ma:
    case UsbPowerState::Cdp1500Ma:
    case UsbPowerState::Sdp500Ma:
    case UsbPowerState::Apple500Ma:
    case UsbPowerState::Apple1000Ma:
    case UsbPowerState::Apple2000Ma:
        objectPosition = 0x1;
        maximumOperatingCurrent = 0;
        maximumCurrent = 0; // 0, 1500, 3000 のいずれかが PD によって設定される。DCP, CDP, SDP, Apple900 は実機で挙動を確認した結果 0 だったので 0 を設定。
        voltage = 5000;
        active = true;
        break;
    case UsbPowerState::StartOutput:
        active = false;
        statusPowerRole = ::nn::usb::pd::StatusPowerRole_Sink;
        statusRequest = ::nn::usb::pd::StatusRequest_PowerOutput;
        break;
    case UsbPowerState::ActiveOutput:
        active = true;
        statusPowerRole = ::nn::usb::pd::StatusPowerRole_Source;
        break;
    case UsbPowerState::EndOutput:
        active = false;
        statusPowerRole = ::nn::usb::pd::StatusPowerRole_Source;
        statusRequest = ::nn::usb::pd::StatusRequest_PowerInput;
        break;
    case UsbPowerState::ErrorOverVoltage:
        statusError = ::nn::usb::pd::StatusError_OverVoltage;
        break;
    case UsbPowerState::ErrorCradleUsbHubUndetected:
        statusError = ::nn::usb::pd::StatusError_CradleUsbHubUndetected;
        break;
    case UsbPowerState::Inactive:
        break;
    } // NOLINT(style/switch_default)

    pOutStatus->m_CurrentRdo.Set<::nn::usb::pd::Rdo::ObjectPosition>(objectPosition);
    pOutStatus->m_CurrentRdo.Set<::nn::usb::pd::Rdo::MaximumOperatingCurrent>(maximumOperatingCurrent / ::nn::usb::pd::RdoMilliAmpereUnit);
    pOutStatus->m_CurrentPdo.Set<::nn::usb::pd::Pdo::MaximumCurrent>(maximumCurrent / ::nn::usb::pd::PdoMilliAmpereUnit);
    pOutStatus->m_CurrentPdo.Set<::nn::usb::pd::Pdo::Voltage>(voltage / ::nn::usb::pd::PdoMilliVoltUnit);

    pOutStatus->m_Data.Set<::nn::usb::pd::Status::Active>(active);
    pOutStatus->m_Data.Set<::nn::usb::pd::Status::Error>(static_cast<::nn::Bit8>(statusError));
    pOutStatus->m_Data.Set<::nn::usb::pd::Status::PowerRole>(static_cast<::nn::Bit8>(statusPowerRole));
    pOutStatus->m_Data.Set<::nn::usb::pd::Status::Request>(static_cast<::nn::Bit8>(statusRequest));
}

void GeneratePowerState(::nn::usb::UsbPowerState* pOutState, ::nn::usb::pd::Status& status, const UsbPowerState usbPowerState) NN_NOEXCEPT
{
    pOutState->role = ::nn::usb::UsbPowerRole_Unknown;
    pOutState->charger = ::nn::usb::UsbChargerType_Unknown;
    pOutState->voltage = 0;
    pOutState->current = 0;
    pOutState->pdo.Clear();
    pOutState->rdo.Clear();

    switch ( usbPowerState )
    {
    case UsbPowerState::He15V:
    case UsbPowerState::He13V:
    case UsbPowerState::Lt13V:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Pd;
        pOutState->pdo = status.m_CurrentPdo;
        pOutState->rdo = status.m_CurrentRdo;
        break;
    case UsbPowerState::Dcp1500Ma:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Dcp;
        pOutState->voltage = 5000;
        pOutState->current = 1500;
        break;
    case UsbPowerState::Cdp1500Ma:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Cdp;
        pOutState->voltage = 5000;
        pOutState->current = 1500;
        break;
    case UsbPowerState::Sdp500Ma:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Sdp;
        pOutState->voltage = 5000;
        pOutState->current = 500;
        break;
    case UsbPowerState::Apple500Ma:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Apple500ma;
        pOutState->voltage = 5000;
        pOutState->current = 500;
        break;
    case UsbPowerState::Apple1000Ma:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Apple1000ma;
        pOutState->voltage = 5000;
        pOutState->current = 900;
        break;
    case UsbPowerState::Apple2000Ma:
        pOutState->role = ::nn::usb::UsbPowerRole_Sink;
        pOutState->charger = ::nn::usb::UsbChargerType_Apple2000ma;
        pOutState->voltage = 5000;
        pOutState->current = 2000;
        break;
    case UsbPowerState::StartOutput:
        // 未実装。
        break;
    case UsbPowerState::ActiveOutput:
        // 未実装。
        break;
    case UsbPowerState::EndOutput:
        // 未実装。
        break;
    case UsbPowerState::ErrorOverVoltage:
        // 未実装。
        break;
    case UsbPowerState::ErrorCradleUsbHubUndetected:
        // 未実装。
        break;
    case UsbPowerState::Inactive:
        break;
    } // NOLINT(style/switch_default)
}

bool IsActiveState(::nn::psc::PmState pmState) NN_NOEXCEPT
{
    return (pmState == ::nn::psc::PmState_FullAwake || pmState == ::nn::psc::PmState_MinimumAwake);
}

void TransitPmStateInternal(::nn::psc::PmState pmState, bool stopOnActiveState) NN_NOEXCEPT
{
    auto nextState = g_CurrentState;

    while ( g_CurrentState != pmState && !(stopOnActiveState && IsActiveState(g_CurrentState)) )
    {
        switch ( g_CurrentState )
        {
        case ::nn::psc::PmState_FullAwake:
            if ( pmState == ::nn::psc::PmState_ShutdownReady )
            {
                nextState = ::nn::psc::PmState_ShutdownReady;
            }
            else
            {
                nextState = ::nn::psc::PmState_MinimumAwake;
            }
            break;
        case ::nn::psc::PmState_MinimumAwake:
            if ( pmState == ::nn::psc::PmState_FullAwake )
            {
                nextState = ::nn::psc::PmState_FullAwake;
            }
            else if ( pmState == ::nn::psc::PmState_ShutdownReady )
            {
                nextState = ::nn::psc::PmState_ShutdownReady;
            }
            else
            {
                nextState = ::nn::psc::PmState_SleepReady;
            }
            break;
        case ::nn::psc::PmState_SleepReady:
            nextState = ::nn::psc::PmState_EssentialServicesSleepReady;
            break;
        case ::nn::psc::PmState_EssentialServicesSleepReady:
            nextState = ::nn::psc::PmState_EssentialServicesAwake;
            break;
        case ::nn::psc::PmState_EssentialServicesAwake:
            nextState = ::nn::psc::PmState_MinimumAwake;
            break;
        case ::nn::psc::PmState_ShutdownReady:
            nextState = ::nn::psc::PmState_FullAwake;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NN_LOG("State transition: %d -> %d\n", g_CurrentState, nextState);
        if ( g_CurrentState == ::nn::psc::PmState_ShutdownReady && nextState == ::nn::psc::PmState_FullAwake )
        {
            ::nnt::psm::Finalize();
            ::nnt::psm::Initialize();
        }
        else
        {
            ChangePmState(nextState);
        }

        g_CurrentState = nextState;
    }
}

} // namespace

void Initialize() NN_NOEXCEPT
{
    ::nn::psm::driver::Initialize();
    ::nn::psm::InitializeWith(g_Server.GetShared());

#if defined(PSM_DETAIL_MANUFACTURE_BUILD)
    ::nn::psm::InitializeForManufactureWith(g_ManufactureServer.GetShared());
#endif

    InitializeEventPointers();
    g_PreviousStatus.Reset();
}

void Finalize() NN_NOEXCEPT
{
    ::nn::psm::Finalize();
    ::nn::psm::driver::Finalize();
}

void CheckVdd50Condition(Vdd50Condition vdd50Condition) NN_NOEXCEPT
{
    ::nn::gpio::GpioValue vdd50AValue = ::nn::gpio::GpioValue_Low;
    ::nn::gpio::GpioValue vdd50BValue = ::nn::gpio::GpioValue_Low;

    switch ( vdd50Condition )
    {
    case Vdd50Condition_None:
        vdd50AValue = ::nn::gpio::GpioValue_Low;
        vdd50BValue = ::nn::gpio::GpioValue_Low;
        break;
    case Vdd50Condition_Vdd50A:
        vdd50AValue = ::nn::gpio::GpioValue_High;
        vdd50BValue = ::nn::gpio::GpioValue_Low;
        break;
    case Vdd50Condition_Vdd50B:
        vdd50AValue = ::nn::gpio::GpioValue_Low;
        vdd50BValue = ::nn::gpio::GpioValue_High;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 裏口。セッションを開くことなく Mutex を取って現在の状態を確認する。
    EXPECT_EQ(true, ::nnt::gpio::CheckSessionContext(::nn::gpio::GpioPadName_Vdd50AEn, ::nn::gpio::Direction_Output, vdd50AValue));
    EXPECT_EQ(true, ::nnt::gpio::CheckSessionContext(::nn::gpio::GpioPadName_Vdd50BEn, ::nn::gpio::Direction_Output, vdd50BValue));
}

void ChangeBatteryTemperature(double temperature) NN_NOEXCEPT
{
    driver::detail::SetBatteryTemperature(temperature);

    // 閾値に深い意図はない。
    // 現在の PSM ライブラリの実装では上側の閾値でシグナルされても下側の閾値でシグナルされてもその後の挙動に変わりはない。
    if ( temperature < 14.0 )
    {
        driver::detail::SignalBatteryTemperatureTooLow();
    }
    else
    {
        driver::detail::SignalBatteryTemperatureTooHigh();
    }

    // FualGauge のハンドリングを待つ。
    TimedWaitAndClearEvent(g_pEventFuelGauge);
}

void ChangeBatteryVoltageMilliVolt(int batteryVoltageMilliVolt) NN_NOEXCEPT
{
    driver::detail::SetBatteryVoltageMilliVolt(batteryVoltageMilliVolt);

    if ( batteryVoltageMilliVolt != g_PreviousStatus.batteryVoltageMilliVolt )
    {
        // 非同期に定期的に更新されるので TimerEvent を 3 回待つ。
        WaitAndClearEvent(g_pEventFuelGaugeTimer); // 前回のシグナルのクリア。
        WaitAndClearEvent(g_pEventFuelGaugeTimer); // batteryVoltage 更新時 PSM のハンドラが batteryVoltage 取得からシグナルまでの間にいた可能性を考慮した Wait。
        WaitAndClearEvent(g_pEventFuelGaugeTimer); // 値が反映されたことを保証する Wait。

        g_PreviousStatus.batteryVoltageMilliVolt = batteryVoltageMilliVolt;
    }
}

void ChangeBatteryChargePercentage(double batteryChargePercentage) NN_NOEXCEPT
{
    driver::detail::SetBatteryChargePercentage(batteryChargePercentage);

    if ( batteryChargePercentage != g_PreviousStatus.batteryChargePercentage )
    {
        // 非同期に定期的に更新されるので TimerEvent を 3 回待つ。理由は上述。
        WaitAndClearEvent(g_pEventFuelGaugeTimer);
        WaitAndClearEvent(g_pEventFuelGaugeTimer);
        WaitAndClearEvent(g_pEventFuelGaugeTimer);

        g_PreviousStatus.batteryChargePercentage = batteryChargePercentage;
    }
}

void ChangeAverageCurrentMilliAmpere(int averageCurrentMilliAmpere) NN_NOEXCEPT
{
    driver::detail::SetAverageCurrentMilliAmpere(averageCurrentMilliAmpere);

    if ( averageCurrentMilliAmpere != g_PreviousStatus.averageCurrentMilliAmpere )
    {
        // 非同期に定期的に更新されるので TimerEvent を 3 回待つ。理由は上述。
        WaitAndClearEvent(g_pEventFuelGaugeTimer);
        WaitAndClearEvent(g_pEventFuelGaugeTimer);
        WaitAndClearEvent(g_pEventFuelGaugeTimer);

        g_PreviousStatus.averageCurrentMilliAmpere = averageCurrentMilliAmpere;
    }
}

void ChangeVoltageFuelGaugePercentage(double voltageFuelGaugePercentage) NN_NOEXCEPT
{
    driver::detail::SetVoltageFuelGaugePercentage(voltageFuelGaugePercentage);

    if ( voltageFuelGaugePercentage != g_PreviousStatus.voltageFuelGaugePercentage )
    {
        // 非同期に定期的に更新されるので TimerEvent を 3 回待つ。理由は上述。
        WaitAndClearEvent(g_pEventFuelGaugeTimer);
        WaitAndClearEvent(g_pEventFuelGaugeTimer);
        WaitAndClearEvent(g_pEventFuelGaugeTimer);

        g_PreviousStatus.voltageFuelGaugePercentage = voltageFuelGaugePercentage;
    }
}

void ChangeEnoughPowerChargeEmulation(bool enabled) NN_NOEXCEPT
{
    if ( enabled )
    {
        ::nn::psm::EnableEnoughPowerChargeEmulation();
    }
    else
    {
        ::nn::psm::DisableEnoughPowerChargeEmulation();
    }

    // FualGauge のハンドリングを待つ。
    TimedWaitAndClearEvent(g_pEventEnoughPowerChargeEmulation);
}

void ChangeUsbPowerState(UsbPowerState usbPowerState) NN_NOEXCEPT
{
    ::nn::usb::pd::Status status;
    ::nn::usb::UsbPowerState state;

    GeneratePowerDeliveryStatus(&status, usbPowerState);
    GeneratePowerState(&state, status, usbPowerState);

    // 設定した USB PD の Status を流し込む。
    if ( ::nnt::usb::pd::SetStatus(status) )
    {
        // USB PD のハンドリングを待つ。
        TimedWaitAndClearEvent(g_pEventUsbPd);
    }

    // 設定した USB PM の State を流し込む。
    ::nnt::usb::SetPmPowerState(state);
    ::nnt::usb::SignalPmPowerEvent();

    // USB PM のハンドリングを待つ。
    TimedWaitAndClearEvent(g_pEventUsbPm);
}

void ChangePmState(::nn::psc::PmState pmState) NN_NOEXCEPT
{
    ::nn::psc::PmFlagSet flags;

    // 特に指定するフラグは無いので Reset だけ行う。
    flags.Reset();

    g_pPmModule->Dispatch(pmState, flags);
    TimedWaitAndClearEvent(g_pEventPsc);
}

void TransitPmState(::nn::psc::PmState pmState) NN_NOEXCEPT
{
    TransitPmStateInternal(pmState, false);
}

void TransitPmState(::nn::psc::PmState pmState, bool stopOnActiveState) NN_NOEXCEPT
{
    // ActiveState にならない可能性がある場合強制的に遷移先を変更します。
    if ( stopOnActiveState && !IsActiveState(g_CurrentState) && !IsActiveState(pmState) )
    {
        if ( g_CurrentState == ::nn::psc::PmState_ShutdownReady )
        {
            TransitPmStateInternal(::nn::psc::PmState_FullAwake, false);
        }
        else
        {
            TransitPmStateInternal(::nn::psc::PmState_MinimumAwake, false);
        }
    }

    TransitPmStateInternal(pmState, stopOnActiveState);
}

}} // namespace nnt::psm
