﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_Manufacture.h>
#include <nn/psm/psm_System.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nnt/nntest.h>

TEST(PsmBasicTest, GetBatteryChargePercentage)
{
    ::nn::psm::Initialize();

    int percentage = ::nn::psm::GetBatteryChargePercentage();
    NN_LOG("PSM Test: battery charge percentage: %d %%\n", percentage);
    EXPECT_LE(1, percentage);
    EXPECT_LE(percentage, 100);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, GetRawBatteryChargePercentage)
{
    ::nn::psm::Initialize();

    double percentage = ::nn::psm::GetRawBatteryChargePercentage();
    NN_LOG("PSM Test: raw battery charge percentage: %f %%\n", percentage);
    EXPECT_LE(0.0f, percentage);
    EXPECT_LE(percentage, 100.0f);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, GetBatteryAgePercentage)
{
    ::nn::psm::Initialize();

    double percentage = ::nn::psm::GetBatteryAgePercentage();
    NN_LOG("PSM Test: batteryage percentage: %f %%\n", percentage);
    EXPECT_LE(0.0f, percentage);
    EXPECT_LE(percentage, 256.0f);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, GetChargerType)
{
    ::nn::psm::Initialize();

    auto chargerType = ::nn::psm::GetChargerType();
    EXPECT_TRUE(
        chargerType == ::nn::psm::ChargerType_Unconnected
        || chargerType == ::nn::psm::ChargerType_EnoughPower
        || chargerType == ::nn::psm::ChargerType_LowPower
        || chargerType == ::nn::psm::ChargerType_NotSupported);

    const char* chargerTypeName = nullptr;
    switch (chargerType)
    {
    case ::nn::psm::ChargerType_Unconnected:
        chargerTypeName = "Unconnected";
        break;
    case ::nn::psm::ChargerType_EnoughPower:
        chargerTypeName = "EnoughPower";
        break;
    case ::nn::psm::ChargerType_LowPower:
        chargerTypeName = "LowPower";
        break;
    case ::nn::psm::ChargerType_NotSupported:
        chargerTypeName = "NotSupported";
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_LOG("PSM Test: charger type: %s\n", chargerTypeName);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, GetBatteryVoltageState)
{
    ::nn::psm::Initialize();

    auto batteryVoltageState = ::nn::psm::GetBatteryVoltageState();
    EXPECT_TRUE(
        batteryVoltageState == ::nn::psm::BatteryVoltageState_ShutdownRequired
        || batteryVoltageState == ::nn::psm::BatteryVoltageState_SleepRequired
        || batteryVoltageState == ::nn::psm::BatteryVoltageState_BoostPerformanceModeProhibited
        || batteryVoltageState == ::nn::psm::BatteryVoltageState_Good);

    const char* batteryVoltageStateName = nullptr;
    switch (batteryVoltageState)
    {
    case ::nn::psm::BatteryVoltageState_ShutdownRequired:
        batteryVoltageStateName = "ShutdownRequired";
        break;
    case ::nn::psm::BatteryVoltageState_SleepRequired:
        batteryVoltageStateName = "SleepRequired";
        break;
    case ::nn::psm::BatteryVoltageState_BoostPerformanceModeProhibited:
        batteryVoltageStateName = "BoostPerformanceModeProhibited";
        break;
    case ::nn::psm::BatteryVoltageState_Good:
        batteryVoltageStateName = "Good";
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_LOG("PSM Test: battery voltage state: %s\n", batteryVoltageStateName);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, EnableBatteryCharging)
{
    ::nn::psm::Initialize();

    // 最後に状態を戻すために最初の状態を取得しておく
    bool isEnabled = ::nn::psm::IsBatteryChargingEnabled();

    ::nn::psm::EnableBatteryCharging();
    EXPECT_TRUE(::nn::psm::IsBatteryChargingEnabled() == true);

    ::nn::psm::DisableBatteryCharging();
    EXPECT_TRUE(::nn::psm::IsBatteryChargingEnabled() == false);

    if (isEnabled)
    {
        ::nn::psm::EnableBatteryCharging();
    }
    else
    {
        ::nn::psm::DisableBatteryCharging();
    }
    EXPECT_TRUE(::nn::psm::IsBatteryChargingEnabled() == isEnabled);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, OpenSession)
{
    ::nn::psm::Initialize();

    ::nn::psm::Session session;
    ::nn::psm::OpenSession(&session);
    ::nn::psm::CloseSession(&session);

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, BindEvent)
{
    ::nn::psm::Initialize();

    ::nn::psm::Session session;
    ::nn::psm::OpenSession(&session);

    ::nn::os::SystemEventType event;
    ::nn::psm::BindStateChangeEvent(&event, &session);
    ::nn::psm::UnbindStateChangeEvent(&session);

    ::nn::psm::CloseSession(&session);

    ::nn::psm::Finalize();
}

// システムの状態が破綻する可能性があるので通常時にはテストを無効化
TEST(PsmBasicTest, DISABLED_EnableFastBatteryCharging)
{
    ::nn::psm::Initialize();

    ::nn::psm::EnableFastBatteryCharging();

    ::nn::psm::DisableFastBatteryCharging();

    ::nn::psm::Finalize();
}

TEST(PsmBasicTest, GetChargerTypeChangeEvent)
{
    ::nn::psm::Initialize();

    ::nn::os::SystemEventType event;
    ::nn::psm::GetChargerTypeChangeEvent(&event);

    ::nn::psm::Finalize();
}
