﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/profiler.h>
#include <nn/profiler/profiler_Control.private.h>

#include <nnt/nntest.h>

#include "../../Common/testProfiler_Common.h"

namespace /*anonymous*/ {

const size_t ThreadStackSize = 16 * 1024;

NN_ALIGNAS(4096) char gProfilerBuffer[nn::profiler::MinimumBufferSize];

volatile bool gShouldExit = false;

struct ThreadDetails
{
    NN_OS_ALIGNAS_THREAD_STACK char stack[ThreadStackSize];
    nn::os::ThreadType thread;
    int64_t accumulator;
    int64_t count[3];
    int threadIdx;
    int coreAffinity;
};
ThreadDetails threads[4];

int64_t Fibonacci(int n)
{
    if (n == 0) { return 0; }
    if (n == 1) { return 1; }
    return (Fibonacci(n - 1) + Fibonacci(n - 2));
}

bool UseOutOfProcessProfiler()
{
    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    for (int i = 0; i < argc; ++i)
    {
        char* arg = argv[i];
        if (arg != nullptr)
        {
            if (strcmp(arg, "--out") == 0)
            {
                return true;
            }
            else if (strcmp(arg, "--in") == 0)
            {
                return false;
            }
        }
    }

    return false;
}

} // anonymous


namespace nn { namespace profiler {

bool IsPCConnected();

}} // namespace nn::profiler

void ThreadFunction(void* arg)
{
    ThreadDetails* details = static_cast<ThreadDetails*>(arg);
    details->accumulator = 0;
    details->count[0] = 0;
    details->count[1] = 0;
    details->count[2] = 0;

    while (!gShouldExit)
    {
        int64_t value = Fibonacci(23);
        details->accumulator += value;
        details->count[nn::os::GetCurrentCoreNumber()] += 1;

        if (details->threadIdx <= 2)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
        }
    }
}



TEST(ThreadMigration, Base)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    EXPECT_RESULT_SUCCESS(result);

    gShouldExit = false;
    for (int i = 0; i < 4; ++i)
    {
        threads[i].threadIdx = i;
        if (i < 3)
        {
            threads[i].coreAffinity = (1 << i);
        }
        else
        {
            threads[i].coreAffinity = 0x7;
        }
    }

    for (int i = 0; i < 4; ++i)
    {
        int idealCore = i % 3;
        result = nn::os::CreateThread(
            &threads[i].thread,
            ThreadFunction,
            &threads[i],
            &threads[i].stack,
            ThreadStackSize,
            nn::os::DefaultThreadPriority + (i / 3),
            idealCore);
        EXPECT_RESULT_SUCCESS(result);
        EXPECT_NE(0, (threads[i].coreAffinity & (1 << idealCore)));
        nn::os::SetThreadCoreMask(&threads[i].thread, idealCore, threads[i].coreAffinity);

        char threadName[nn::os::ThreadNameLengthMax];
        sprintf(threadName, "Thread: %d", i);
        nn::os::SetThreadName(&threads[i].thread, threadName);
    }

    NN_LOG("Waiting for PC to connect...\n");
    while (!nn::profiler::IsPCConnected())
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    for (int i = 0; i < 4; ++i)
    {
        nn::os::StartThread(&threads[i].thread);
    }

    uint32_t flags = nn::profiler::Flags_Callstack;
    if (UseOutOfProcessProfiler())
    {
        NN_LOG("Taking Out-of-Process profile\n");
        flags |= nn::profiler::Flags_OutOfProcessProfiling;
    }
    else
    {
        NN_LOG("Taking In-Process profile\n");
    }

    result = nn::profiler::SetProfileSettings(
        nn::profiler::Cores_All,
        flags,
        nn::profiler::PerformanceCounterGroup_Disabled,
        nn::profiler::SampleRate_ByTime100x);
    EXPECT_RESULT_SUCCESS(result);

    result = nn::profiler::StartProfiling();
    EXPECT_RESULT_SUCCESS(result);
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    result = nn::profiler::StopProfiling();
    EXPECT_RESULT_SUCCESS(result);

    //while (nn::profiler::GetProfilerStatus() != nn::profiler::ProfilerStatus_Profiling)
    //{
    //    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    //}
    //while (nn::profiler::GetProfilerStatus() > nn::profiler::ProfilerStatus_Active)
    //{
    //    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    //}

    gShouldExit = true;

    for (int i = 0; i < 4; ++i)
    {
        nn::os::WaitThread(&threads[i].thread);
    }

    for (int i = 0; i < 4; ++i)
    {
        nn::os::DestroyThread(&threads[i].thread);
    }

    int64_t totals[4];
    for (int i = 0; i < 4; ++i)
    {
        int64_t total;
        ThreadDetails *t = &threads[i];
        total = t->count[0] + t->count[1] + t->count[2];
        if (i < 3)
        {
            EXPECT_EQ(total, t->count[i]);
        }
        totals[i] = total;
        NN_LOG("Thread %d: %lld\n", i, total);
    }

    while (nn::profiler::IsPCConnected())
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    result = nn::profiler::Finalize();
    EXPECT_RESULT_SUCCESS(result);
} //NOLINT(impl/function_size)
