﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#define NN_BUILD_CONFIG_HTC_ENABLED

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/htcs.h>
#include <nn/profiler.h>

// Profiler private headers
// --
#include <nn/profiler/profiler_Control.private.h>
#include <nn/profiler/profiler_Module.private.h>
// --

#include <nnt/nntest.h>

#include "../../Common/testProfiler_Common.h"
#include "../../../../../Programs/Eris/Sources/Libraries/profiler/profiler_Main.h"
#include "../../../../../Programs/Eris/Sources/Processes/profiler/common/profiler_SampleRates.h"
#include "../../../../../Programs/Eris/Sources/Processes/profiler/common/profiler_SettingsFromThePcGui.h"


namespace nn { namespace profiler {

#include <nn/profiler/profiler_Result.private.h>

uint64_t RegisterStringId(const char* pMarkerName);

namespace detail
{
    nn::Result ForceAttachToInProcess();
}

}}


namespace /*anonymous*/ {

    NN_ALIGNAS(4096) char gProfilerBuffer[nn::profiler::MinimumBufferSize];

    void* HtcsAllocate(size_t size)
    {
        return malloc(size);
    }

    void HtcsFree(void* mem, size_t size)
    {
        NN_UNUSED(size);
        free(mem);
    }

    void TakeSimpleProfileCommon()
    {
        nn::Result result;

        result = nn::profiler::StartProfiling();
        EXPECT_RESULT_SUCCESS(result);
        EXPECT_EQ(nn::profiler::ProfilerStatus_Profiling, nn::profiler::GetProfilerStatus());

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));

        result = nn::profiler::StopProfiling();
        EXPECT_RESULT_SUCCESS(result);
        EXPECT_NE(nn::profiler::ProfilerStatus_Profiling, nn::profiler::GetProfilerStatus());
    }

    void TakeMultipleProfilesCommon()
    {
        nn::Result result;

        for (int i = 0; i < 3; ++i)
        {
            NN_LOG("Taking profile %d\n", i + 1);

            TakeSimpleProfileCommon();

            // There is some round trip time with the profiler process.
            // Give the process some time to cope, but it really shouldn't take that long.
            int waitCount = 0;
            while (nn::profiler::GetProfilerStatus() != nn::profiler::ProfilerStatus_Active && waitCount < 10)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
            EXPECT_LT(waitCount, 10);
        }
    }

    void ConditionalRecordsCommon()
    {
        nn::Result result;
        const uint64_t id = 0;
        const uint64_t data_u64 = 0;
        const double data_dbl = 0.0;

        result = nn::profiler::StartProfiling();
        EXPECT_RESULT_SUCCESS(result);

        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User1));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User2));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User3));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User4));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User5));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User6));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User7));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User8));
        EXPECT_RESULT_SUCCESS(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(nn::profiler::Heartbeats_MAX)));

        result = nn::profiler::RecordData(id, data_u64);
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::RecordData("Data-uint64_t", data_u64);
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::RecordData(nullptr, data_u64);
        EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
        result = nn::profiler::RecordData(id, data_dbl);
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::RecordData("Data-double", data_dbl);
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::RecordData(nullptr, data_dbl);
        EXPECT_RESULT(result, nn::profiler::ResultNullArgument);

        result = nn::profiler::EnterCodeBlock(id);
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::EnterCodeBlock("CodeBlock-manual");
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::EnterCodeBlock(nullptr);
        EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
        result = nn::profiler::ExitCodeBlock(id);
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::ExitCodeBlock("CodeBlock-manual");
        EXPECT_RESULT_SUCCESS(result);
        result = nn::profiler::ExitCodeBlock(nullptr);
        EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
        {
            nn::profiler::ScopedCodeBlock block1(id);
            nn::profiler::ScopedCodeBlock block2("CodeBlock-scoped");
            nn::profiler::ScopedCodeBlock block3(nullptr);
        }

        result = nn::profiler::StopProfiling();
        EXPECT_RESULT_SUCCESS(result);
    }

    void CheckProfilerSettings(uint32_t mask, uint32_t flags, uint32_t perf, uint32_t rate)
    {
        nn::profiler::SettingsFromThePcGui* settings = nn::profiler::GetProfilerSettingsPointer();
        EXPECT_EQ(settings->coreMask, mask);
        EXPECT_EQ(settings->flags, flags);
        EXPECT_EQ(settings->perf_counter_cycle, (perf != nn::profiler::PerformanceCounterGroup_Disabled) ? 1 : 0);
        EXPECT_EQ(settings->requested_time_between_samples_in_nanoseconds, nn::profiler::gSampleByTimeRates[rate]);
    }

    void SetupOutOfProcess()
    {
        nn::Result result;
        result = nn::profiler::detail::ForceAttachToInProcess();
        EXPECT_RESULT_SUCCESS(result);

        uint32_t cores = nn::profiler::Cores_All;
        uint32_t flags = nn::profiler::Flags_Callstack | nn::profiler::Flags_OutOfProcessProfiling;
        nn::profiler::PerformanceCounterGroup perf = nn::profiler::PerformanceCounterGroup_Disabled;
        nn::profiler::SampleRate rate = nn::profiler::SampleRate_ByTime100x;

        result = nn::profiler::SetProfileSettings(cores, flags, perf, rate);
        EXPECT_RESULT_SUCCESS(result);
        CheckProfilerSettings(cores, flags, perf, rate);
    }

} // anonymous


TEST(ApiValidation, BasicSetup)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    EXPECT_EQ(nn::profiler::ProfilerStatus_Active, nn::profiler::GetProfilerStatus());

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);

    EXPECT_EQ(nn::profiler::ProfilerStatus_Offline, nn::profiler::GetProfilerStatus());
}


TEST(ApiValidation, MultipleSetup)
{
    nn::Result result;

    for (int i = 0; i < 3; ++i)
    {
        NN_LOG("Initialization attempt %d\n", i + 1);

        result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
        ASSERT_RESULT_SUCCESS(result);

        EXPECT_EQ(nn::profiler::ProfilerStatus_Active, nn::profiler::GetProfilerStatus());

        result = nn::profiler::Finalize();
        ASSERT_RESULT_SUCCESS(result);

        EXPECT_EQ(nn::profiler::ProfilerStatus_Offline, nn::profiler::GetProfilerStatus());
    }
}


// HTCS not initialized before profiler
// Profiler should initialize and finalize HTCS
TEST(ApiValidation, HtcsNotInitialized)
{
    nn::Result result;
    bool isHtcsInitialized;

    isHtcsInitialized = nn::htcs::IsInitialized();
    ASSERT_FALSE(isHtcsInitialized);

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    isHtcsInitialized = nn::htcs::IsInitialized();
    EXPECT_TRUE(isHtcsInitialized);

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);

    isHtcsInitialized = nn::htcs::IsInitialized();
    EXPECT_FALSE(isHtcsInitialized);
}


// HTCS already initialized before profiler
// Profiler should not finalize HTCS
TEST(ApiValidation, HtcsAlreadyInitialized)
{
    nn::Result result;
    bool isHtcsInitialized;

    isHtcsInitialized = nn::htcs::IsInitialized();
    ASSERT_FALSE(isHtcsInitialized);

    nn::htcs::Initialize(HtcsAllocate, HtcsFree);
    isHtcsInitialized = nn::htcs::IsInitialized();
    ASSERT_TRUE(isHtcsInitialized);

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    isHtcsInitialized = nn::htcs::IsInitialized();
    EXPECT_TRUE(isHtcsInitialized);

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);

    isHtcsInitialized = nn::htcs::IsInitialized();
    EXPECT_TRUE(isHtcsInitialized);

    nn::htcs::Finalize();
    isHtcsInitialized = nn::htcs::IsInitialized();
    EXPECT_FALSE(isHtcsInitialized);
}


TEST(ApiValidation, CallsWithoutInitialization)
{
    nn::Result result;

    const uint64_t id = 0;
    const uint64_t data_u64 = 0;
    const double data_dbl = 0.0;

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_Api.h
    result = nn::profiler::Finalize();
    EXPECT_RESULT_SUCCESS(result);
    EXPECT_EQ(nn::profiler::GetProfilerStatus(), nn::profiler::ProfilerStatus_Offline);
    result = nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::RecordData(id, data_u64);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::RecordData("Data-uint64_t", data_u64);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::RecordData(nullptr, data_u64);
    EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
    result = nn::profiler::RecordData(id, data_dbl);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::RecordData("Data-double", data_dbl);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::RecordData(nullptr, data_dbl);
    EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
    ///////////////////////////////////////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_CodeBlocks.h
    result = nn::profiler::EnterCodeBlock(id);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::EnterCodeBlock("CodeBlock-manual");
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::EnterCodeBlock(nullptr);
    EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
    result = nn::profiler::ExitCodeBlock(id);
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::ExitCodeBlock("CodeBlock-manual");
    EXPECT_RESULT(result, nn::profiler::ResultSkippedRecord);
    result = nn::profiler::ExitCodeBlock(nullptr);
    EXPECT_RESULT(result, nn::profiler::ResultNullArgument);
    {
        nn::profiler::ScopedCodeBlock block1(id);
        nn::profiler::ScopedCodeBlock block2("CodeBlock-scoped");
        nn::profiler::ScopedCodeBlock block3(nullptr);
    }
    ///////////////////////////////////////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_Control.private.h
    result = nn::profiler::StartProfiling();
    EXPECT_RESULT(result, nn::profiler::ResultNotInitialized);
    result = nn::profiler::StopProfiling();
    EXPECT_RESULT(result, nn::profiler::ResultNotInitialized);
    result = nn::profiler::SetProfileSettings(nn::profiler::Cores_All, nn::profiler::Flags_Callstack, nn::profiler::PerformanceCounterGroup_Disabled, nn::profiler::SampleRate_ByTime100x);
    EXPECT_RESULT(result, nn::profiler::ResultNotInitialized);
    ///////////////////////////////////////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_Memory.h
    nn::profiler::RecordHeapRange(reinterpret_cast<void*>(0), reinterpret_cast<void*>(UINTPTR_MAX), "ApiValidation");

    nn::profiler::TrackMalloc(reinterpret_cast<void*>(1), 1);
    nn::profiler::TrackMalloc(reinterpret_cast<void*>(2), 1, "System");
    nn::profiler::TrackMalloc(reinterpret_cast<void*>(3), 1, "System", 1);
    nn::profiler::TrackCalloc(reinterpret_cast<void*>(4), 1, 1);
    nn::profiler::TrackCalloc(reinterpret_cast<void*>(5), 1, 1, "System");
    nn::profiler::TrackCalloc(reinterpret_cast<void*>(6), 1, 1, "System", 1);
    nn::profiler::TrackRealloc(reinterpret_cast<void*>(6), reinterpret_cast<void*>(6), 2);
    nn::profiler::TrackRealloc(reinterpret_cast<void*>(6), reinterpret_cast<void*>(6), 3, "System");
    nn::profiler::TrackRealloc(reinterpret_cast<void*>(6), reinterpret_cast<void*>(6), 4, "System", 1);
    nn::profiler::TrackFree(reinterpret_cast<void*>(1));
    nn::profiler::TrackFree(reinterpret_cast<void*>(2), "System");
    nn::profiler::TrackFree(reinterpret_cast<void*>(3));
    nn::profiler::TrackFree(reinterpret_cast<void*>(4));
    nn::profiler::TrackFree(reinterpret_cast<void*>(5));
    nn::profiler::TrackFree(reinterpret_cast<void*>(6));
    nn::profiler::TrackNew(reinterpret_cast<void*>(1), 1);
    nn::profiler::TrackNew(reinterpret_cast<void*>(2), 1, "System");
    nn::profiler::TrackNew(reinterpret_cast<void*>(3), 1, "System", 1);
    nn::profiler::TrackNew(reinterpret_cast<void*>(4), 1, "System", false);
    nn::profiler::TrackNew(reinterpret_cast<void*>(5), 1, "System", 1, true);
    nn::profiler::TrackDelete(reinterpret_cast<void*>(1));
    nn::profiler::TrackDelete(reinterpret_cast<void*>(2), "System");
    nn::profiler::TrackDelete(reinterpret_cast<void*>(3), "System");
    nn::profiler::TrackDelete(reinterpret_cast<void*>(4), "System", false);
    nn::profiler::TrackDelete(reinterpret_cast<void*>(5), "System", true);

    nn::profiler::TrackMalloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(1), 1);
    nn::profiler::TrackMalloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(2), 1, "System");
    nn::profiler::TrackMalloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(3), 1, "System", 1);
    nn::profiler::TrackCalloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(4), 1, 1);
    nn::profiler::TrackCalloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(5), 1, 1, "System");
    nn::profiler::TrackCalloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(6), 1, 1, "System", 1);
    nn::profiler::TrackRealloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(6), reinterpret_cast<void*>(6), 2);
    nn::profiler::TrackRealloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(6), reinterpret_cast<void*>(6), 3, "System");
    nn::profiler::TrackRealloc(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(6), reinterpret_cast<void*>(6), 4, "System", 1);
    nn::profiler::TrackFree(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(1));
    nn::profiler::TrackFree(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(2), "System");
    nn::profiler::TrackFree(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(3));
    nn::profiler::TrackFree(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(4));
    nn::profiler::TrackFree(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(5));
    nn::profiler::TrackFree(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(6));
    nn::profiler::TrackNew(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(1), 1);
    nn::profiler::TrackNew(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(2), 1, "System");
    nn::profiler::TrackNew(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(3), 1, "System", 1);
    nn::profiler::TrackNew(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(4), 1, "System", false);
    nn::profiler::TrackNew(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(5), 1, "System", 1, true);
    nn::profiler::TrackDelete(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(1));
    nn::profiler::TrackDelete(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(2), "System");
    nn::profiler::TrackDelete(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(3), "System");
    nn::profiler::TrackDelete(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(4), "System", false);
    nn::profiler::TrackDelete(nn::profiler::GenerateNextTrackingId(), reinterpret_cast<void*>(5), "System", true);
    ///////////////////////////////////////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_Module.private.h
    // Not currently tested here
    ///////////////////////////////////////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_Result.h
    // Nothing to test
    ///////////////////////////////////////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////////////////////////////////////
    // profiler_Types.h
    {
#define PROFILER_TYPES_H_TEST_STRING_LITERAL_STRING "ApiValidation"
        const char* pLiteral = PROFILER_TYPES_H_TEST_STRING_LITERAL_STRING;
        nn::profiler::StringLiteral strLiteral(PROFILER_TYPES_H_TEST_STRING_LITERAL_STRING);
        EXPECT_EQ(pLiteral, strLiteral.GetString());
        EXPECT_EQ(pLiteral, &strLiteral);
        EXPECT_EQ(*pLiteral, *(strLiteral.GetString()));
        EXPECT_EQ(strlen(pLiteral), strLiteral.GetLength());
#undef PROFILER_TYPES_H_TEST_STRING_LITERAL_STRING
    }
    ///////////////////////////////////////////////////////////////////////////////////////////////
} //NOLINT(impl/function_size)


TEST(ApiValidation, BadInitialization)
{
    nn::Result result;

    result = nn::profiler::Initialize(nullptr, 0);
    EXPECT_RESULT(result, nn::profiler::ResultNullArgument);

    result = nn::profiler::Initialize(gProfilerBuffer, 0);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    // This doesn't need to be done, but putting this here anyway to ensure spurious
    // calls to Finalize don't cause problems.
    result = nn::profiler::Finalize();
    EXPECT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, HeartbeatsNoInitialization)
{
    nn::Result result;

    for (int i = 0; i <= static_cast<int>(nn::profiler::Heartbeats_MAX); ++i)
    {
        EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(i)), nn::profiler::ResultSkippedRecord);
    }

    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main),  nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User1), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User2), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User3), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User4), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User5), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User6), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User7), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User8), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_MAX),   nn::profiler::ResultSkippedRecord);

    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(-1)), nn::profiler::ResultInvalidArgument);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(static_cast<int>(nn::profiler::Heartbeats_MAX) + 1)), nn::profiler::ResultInvalidArgument);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(INT_MAX)), nn::profiler::ResultInvalidArgument);
}

#if !defined(MINIMIZE_INITIALIZATION)

TEST(ApiValidation, MultipleInitialization)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    EXPECT_RESULT(result, nn::profiler::ResultAlreadyDone);

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, HeartbeatsWithInitialization)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    for (int i = 0; i <= static_cast<int>(nn::profiler::Heartbeats_MAX); ++i)
    {
        EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(i)), nn::profiler::ResultSkippedRecord);
    }

    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_Main), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User1), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User2), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User3), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User4), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User5), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User6), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User7), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User8), nn::profiler::ResultSkippedRecord);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(nn::profiler::Heartbeats_MAX)), nn::profiler::ResultSkippedRecord);

    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(-1)), nn::profiler::ResultInvalidArgument);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(static_cast<int>(nn::profiler::Heartbeats_MAX) + 1)), nn::profiler::ResultInvalidArgument);
    EXPECT_RESULT(nn::profiler::RecordHeartbeat(static_cast<nn::profiler::Heartbeats>(INT_MAX)), nn::profiler::ResultInvalidArgument);

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, TakeSimpleProfile)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    TakeSimpleProfileCommon();

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, TakeMultipleProfiles)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    TakeMultipleProfilesCommon();

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, InvalidProfileOperations)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    EXPECT_RESULT_SUCCESS(result);

    result = nn::profiler::StopProfiling();
    EXPECT_RESULT(result, nn::profiler::ResultAlreadyDone);

    result = nn::profiler::StartProfiling();
    EXPECT_RESULT_SUCCESS(result);
    result = nn::profiler::StartProfiling();
    EXPECT_RESULT(result, nn::profiler::ResultAlreadyDone);

    result = nn::profiler::StopProfiling();
    EXPECT_RESULT_SUCCESS(result);
    result = nn::profiler::StopProfiling();
    EXPECT_RESULT(result, nn::profiler::ResultAlreadyDone);

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, ConditionalRecords)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    ConditionalRecordsCommon();

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, MemoryApi)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    nn::profiler::RecordHeapRange(reinterpret_cast<void*>(0), reinterpret_cast<void*>(UINTPTR_MAX), "ApiValidation");

    // Loop enough times to ensure that the buffer to send data to the PC fills
    // That buffer is 8KB.
    // At least 8-bytes, the address, is sent with each message (even in 32-bit mode).
    // So attempt to track at least that many items.
    const size_t memBufferSize = 8 * 1024;
    for (int i = 0; i < memBufferSize / sizeof(uint64_t); ++i)
    {
        nn::profiler::TrackMalloc(reinterpret_cast<void*>(i), 1);
        nn::profiler::TrackFree(reinterpret_cast<void*>(i));
    }

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}

#endif


// Tests the private function nn::profiler::RegisterStringId
TEST(ApiValidation, RegisterStringId)
{
#define APIVALIDATION_REGISTERSTRINGID_STRING "ApiValidation"
    nn::Result result;
    uint64_t id;

    id = nn::profiler::RegisterStringId(nullptr);
    EXPECT_EQ(id, UINT64_MAX);

    id = nn::profiler::RegisterStringId(APIVALIDATION_REGISTERSTRINGID_STRING);
    EXPECT_EQ(id, UINT64_MAX);

#if !defined(MINIMIZE_INITIALIZATION)
    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    id = nn::profiler::RegisterStringId(nullptr);
    EXPECT_EQ(id, UINT64_MAX);

    id = nn::profiler::RegisterStringId(APIVALIDATION_REGISTERSTRINGID_STRING);
    EXPECT_NE(id, UINT64_MAX);

    {
        char dynamicStr[64];
        memset(dynamicStr, 0, sizeof(dynamicStr));
        GTEST_SNPRINTF_(dynamicStr, sizeof(dynamicStr), "%s", APIVALIDATION_REGISTERSTRINGID_STRING);
        uint64_t id2 = nn::profiler::RegisterStringId(dynamicStr);
        EXPECT_EQ(id, id2);
    }

    {
        char dynamicStr[64];
        memset(dynamicStr, 0, sizeof(dynamicStr));
        GTEST_SNPRINTF_(dynamicStr, sizeof(dynamicStr), "%s ", APIVALIDATION_REGISTERSTRINGID_STRING);
        uint64_t id2 = nn::profiler::RegisterStringId(dynamicStr);
        EXPECT_NE(id, id2);
    }

    {
        uint64_t id2 = nn::profiler::RegisterStringId(APIVALIDATION_REGISTERSTRINGID_STRING);
        EXPECT_EQ(id, id2);
    }

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
#endif
#undef APIVALIDATION_REGISTERSTRINGID_STRING
}


TEST(ApiValidation, SetProfilerSettings)
{
    nn::Result result;

    // Default settings from GUI
    uint32_t cores = nn::profiler::Cores_All;
    uint32_t flags = nn::profiler::Flags_Callstack;
    nn::profiler::PerformanceCounterGroup perf = nn::profiler::PerformanceCounterGroup_Disabled;
    nn::profiler::SampleRate rate = nn::profiler::SampleRate_ByTime100x;

    const uint32_t flagsAll =
        nn::profiler::Flags_Simple |
        nn::profiler::Flags_Callstack |
        nn::profiler::Flags_PerformanceCounters;

    result = nn::profiler::SetProfileSettings(cores, flags, perf, rate);
    EXPECT_RESULT(result, nn::profiler::ResultNotInitialized);

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, rate);
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, flags, perf, rate);

    result = nn::profiler::SetProfileSettings(0, flags, perf, rate);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(UINT32_MAX, flags, perf, rate);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, 0, perf, rate);
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, 0, perf, rate);

    result = nn::profiler::SetProfileSettings(cores, flagsAll, perf, rate);
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, flagsAll, perf, rate);

    result = nn::profiler::SetProfileSettings(cores, UINT32_MAX, perf, rate);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, flags, nn::profiler::PerformanceCounterGroup(0), rate);
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, flags, nn::profiler::PerformanceCounterGroup(0), rate);

    result = nn::profiler::SetProfileSettings(cores, flags, nn::profiler::PerformanceCounterGroup(-1), rate);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, flags, nn::profiler::PerformanceCounterGroup(INT32_MAX), rate);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, nn::profiler::SampleRate_ByTimeMin);
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, flags, perf, nn::profiler::SampleRate_ByTimeMin);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, nn::profiler::SampleRate_ByTimeMax);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, nn::profiler::SampleRate(0));
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, flags, perf, nn::profiler::SampleRate(0));

    result = nn::profiler::SetProfileSettings(cores, flags, perf, nn::profiler::SampleRate(-1));
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, nn::profiler::SampleRate(INT32_MAX));
    EXPECT_RESULT(result, nn::profiler::ResultInvalidArgument);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, rate);
    EXPECT_RESULT_SUCCESS(result);
    CheckProfilerSettings(cores, flags, perf, rate);

    result = nn::profiler::StartProfiling();
    EXPECT_RESULT_SUCCESS(result);

    result = nn::profiler::SetProfileSettings(cores, flags, perf, rate);
    EXPECT_RESULT(result, nn::profiler::ResultInvalidProfilerStatus);

    result = nn::profiler::StopProfiling();
    EXPECT_RESULT_SUCCESS(result);

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, Core3Initialization)
{
    int startingIdealCore;
    nn::Bit64 startingAffinityMask;
    auto thread = nn::os::GetCurrentThread();
    nn::os::GetThreadCoreMask(&startingIdealCore, &startingAffinityMask, thread);

    auto affinityMask = nn::os::GetThreadAvailableCoreMask();

    ASSERT_NE(affinityMask & (1 << 3), 0);

    nn::os::SetThreadCoreMask(thread, 3, 1 << 3);
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));

    auto currentCore = nn::os::GetCurrentCoreNumber();
    ASSERT_EQ(currentCore, 3);

    ASSERT_RESULT(nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer)), nn::profiler::ResultInitFailedCore3);

    nn::os::SetThreadCoreMask(thread, startingIdealCore, startingAffinityMask);
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
}


TEST(ApiValidation, TakeSimpleProfileOutOfProcess)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    SetupOutOfProcess();

    TakeSimpleProfileCommon();

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, TakeMultipleProfilesOutOfProcess)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    SetupOutOfProcess();

    TakeMultipleProfilesCommon();

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}


TEST(ApiValidation, ConditionalRecordsOutOfProcess)
{
    nn::Result result;

    result = nn::profiler::Initialize(gProfilerBuffer, sizeof(gProfilerBuffer));
    ASSERT_RESULT_SUCCESS(result);

    SetupOutOfProcess();

    ConditionalRecordsCommon();

    result = nn::profiler::Finalize();
    ASSERT_RESULT_SUCCESS(result);
}
