﻿param(
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]$SampleMode = [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]::InProcess,
    
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [string]$Platform,
    
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [string]$BuildType
)

[string]$filename = "nxcp-smoketest.$SampleMode"
if ([string]::IsNullOrEmpty($Platform) -eq $false)
{
    $filename = "$filename.$Platform"
}
if ([string]::IsNullOrEmpty($BuildType) -eq $false)
{
    $filename = "$filename.$BuildType"
}

[string]$profileName = [System.IO.Path]::Combine([System.IO.Path]::GetTempPath(), "NintendoCpuProfiler", "profiles", "$filename.nprof")
[System.IO.Directory]::CreateDirectory([System.IO.Path]::GetDirectoryName($profileName))

. "$([Nintendo.CPU.Profiler.Scripting.ScriptingCore]::ScriptDirectory)/testProfiler_Common.ps1"

write-output "Starting testProfiler_SmokeTest"

if (test-path $profileName)
{
    remove-item $profileName
}

function TabContentsEquivalent($left, $right)
{
    $leftLines = $left -split [Environment]::NewLine
    $rightLines = $right -split [Environment]::NewLine

    if ($leftLines.Count -ne $rightLines.Count)
    {
        return $false
    }

    $loopCount = [System.Math]::Min($leftLines.Count, 3)
    for ($i = 0; $i -lt $loopCount; ++$i)
    {
        if ($leftLines[$i] -ne $rightLines[$i])
        {
            return $false
        }
    }

    return $true
}

function SyncToDevKit()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    SyncWhenAvailable
    
    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }
    
    return $containedFailure
}

function C2748_CreateNewProfile()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    $samplingSettingsSet = profiler-sample Time 500
    if (!$samplingSettingsSet)
    {
        WriteError "Could not setting sampling settings"
        $containedFailure = $true
    }

    profiler-callstacks $true
    profiler-perfcounters Disabled
    profiler-nx-cores 0 1 2
    profiler-start Sampled
    sleep 5
    profiler-stop

    profiler-save $global:profileName

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Taking and/or saving profile generated a warning"
        $containedFailure = $true
    }

    if ($false -eq (test-path $profileName))
    {
        WriteError "Failed to save profile"
        $containedFailure = $true
    }

    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines.Count -le 3)
    {
        WriteError "No functions loaded"
        $containedFailure = $true
    }

    $infoTab = profiler-copy Info
    $lines = $infoTab -split [Environment]::NewLine
    if ($lines.Count -le 1)
    {
        WriteError "No info on Info Tab"
        $containedFailure = $true
    }

    # Reset
    #  nothing to do

    profiler-unsync

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }

    return $containedFailure
}

function C2749_LoadExistingProfile
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    profiler-open $global:profileName

    profiler-showtab Functions
    profiler-showtab Info

    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines.Count -le 3)
    {
        WriteError "No functions loaded"
        $containedFailure = $true
    }

    $infoTab = profiler-copy Info
    $lines = $infoTab -split [Environment]::NewLine
    if ($lines.Count -le 1)
    {
        WriteError "No info on Info Tab"
        $containedFailure = $true
    }

    # Reset
    #  nothing to do

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }

    return $containedFailure
}

function C2751_UnitRibbonBar
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    profiler-showtab Functions
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines.Count -le 3)
    {
        WriteError "No functions loaded"
        return $false
    }

    profiler-units Percent $false
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines[4].Contains("%") -eq $false)
    {
        WriteError "Units Percent does not show '%' in Functions Tab"
        $containedFailure = $true
    }

    profiler-units Time $false
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines[4].Contains("ms") -eq $false)
    {
        WriteError "Units Time does not show 'ms' in Functions Tab"
        $containedFailure = $true
    }

    profiler-units AvgTime $false
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines[4].Contains("ms") -eq $false)
    {
        WriteError "Units AvgTime does not show 'ms' in Functions Tab"
        $containedFailure = $true
    }

    profiler-units Samples $false
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines[4].Contains("ms") -or $lines[4].Contains("%"))
    {
        WriteError "Units Samples shows 'ms' or '%' in Functions Tab"
        $containedFailure = $true
    }

    profiler-units Percent $true
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines[4].Contains("%") -eq $false)
    {
        WriteError "Units Percent does not show '%' in Functions Tab"
        $containedFailure = $true
    }
    if ($lines[4].Contains("±") -eq $false)
    {
        WriteError "Units Error does not show '±' in Functions Tab"
        $containedFailure = $true
    }

    profiler-units Percent $false
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines[4].Contains("%") -eq $false)
    {
        WriteError "Units Percent does not show '%' in Functions Tab"
        $containedFailure = $true
    }
    if ($lines[4].Contains("±") -eq $true)
    {
        WriteError "Units Error should not show '±' in Functions Tab"
        $containedFailure = $true
    }

    # Reset
    profiler-units Percent $false

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }

    return $containedFailure
}

function C2752_AnalysisTools
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    profiler-showtab Functions
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines.Count -le 3)
    {
        WriteError "No functions loaded"
        return $false
    }

    profiler-showtab Functions
    profiler-select Functions -Top
    profiler-showtab SampleGraph
    profiler-analyze -SpikeDetection BadFrame
    profiler-analyze -Story Brief
    profiler-filter Functions -Core 1
    $filteredFunctionList = profiler-copy Functions
    if ($functionList -eq $filteredFunctionList)
    {
        WriteError "Filtered function list should not match unfiltered list"
        $containedFailure = $true
    }
    $lines = $filteredFunctionList -split [Environment]::NewLine
    if ($lines[3].Contains("Core"))
    {
        WriteError "Functions not filtered by core"
        $containedFailure = $true
    }

    # Reset
    profiler-samplegraph -DeselectAllFrames
    profiler-select Functions -DeselectAll
    profiler-filter Functions -Core All
    profiler-filter Functions -ShowSelected $false
    profiler-sort Functions Self
    $filteredFunctionList = profiler-copy Functions
    if ($false -eq (TabContentsEquivalent $functionList $filteredFunctionList))
    {
        WriteError "Function list still filtered"
        write-output "--"
        write-output "$functionList"
        write-output "--"
        write-output "$filteredFunctionList"
        write-output "--"
        $containedFailure = $true
    }

    return $containedFailure
}

function C2753_Filters
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    profiler-showtab Functions
    $functionList = profiler-copy Functions
    $lines = $functionList -split [Environment]::NewLine
    if ($lines.Count -le 3)
    {
        WriteError "No functions loaded"
        return $false
    }

    profiler-filter Functions -Negate $false -RegEx $false -FilterName "Fibonacci"
    $filteredFunctionList = profiler-copy Functions
    if (TabContentsEquivalent $functionList $filteredFunctionList)
    {
        WriteError "Filtered function list should not match unfiltered list"
        $containedFailure = $true
    }
    $lines = $filteredFunctionList -split [Environment]::NewLine
    if ($lines.Count -ne 5)
    {
        $lineCount = $lines.Count - 4
        write-output "--"
        write-output "lineCount = $lineCount"
        write-output "$lines"
        write-output "--"
        WriteError "There should only be a single function found for 'Fibonacci'"
        $containedFailure = $true
    }

    # Reset
    profiler-filter Functions -Negate $false -RegEx $false -FilterName ""
    $filteredFunctionList = profiler-copy Functions
    if ($false -eq (TabContentsEquivalent $functionList $filteredFunctionList))
    {
        WriteError "Function list still filtered"
        $containedFailure = $true
    }

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }

    return $containedFailure
}

function C2754_CodeCoverageWindow
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    $seenFunctions = profiler-copy CodeCoverage Seen
    $notSeenFunctions = profiler-copy CodeCoverage NotSeen
    if (TabContentsEquivalent $seenFunctions $notSeenFunctions)
    {
        WriteError "Functions Seen and Not Seen should not be the same"
        $containedFailure = $true
    }

    profiler-showtab CodeCoverage
    profiler-filter CodeCoverage -Negate $false -RegEx $false -FilterName "Tick"

    $filteredSeenFunctions = profiler-copy CodeCoverage Seen
    if (TabContentsEquivalent $seenFunctions $filteredSeenFunctions)
    {
        WriteError "Filtered functions Seen should be different than unfiltered"
        $containedFailure = $true
    }
    $lines = $filteredSeenFunctions -split [Environment]::NewLine
    $firstSpace = $lines[0].IndexOf(" ")
    if ($firstSpace -le 0)
    {
        WriteError "Could not find a space in the first line of filtered seen"
        $containedFailure = $true
    }
    elseif ($firstSpace -gt 1)
    {
        WriteError "More than a few instances of Tick found in the filtered seen results"
        $containedFailure = $true
    }

    $filteredNotSeenFunctions = profiler-copy CodeCoverage NotSeen
    $lines = $filteredNotSeenFunctions -split [Environment]::NewLine
    $firstSpace = $lines[0].IndexOf(" ")
    if ($firstSpace -le 0)
    {
        WriteError "Could not find a space in the first line of filtered not seen"
        $containedFailure = $true
    }
    elseif ($firstSpace -gt 3)
    {
        WriteError "More than a few hundred of Tick found in the filtered not seen results"
        $containedFailure = $true
    }

    # Reset
    profiler-filter CodeCoverage -Negate $false -RegEx $false -FilterName ""
    $filteredSeenFunctions = profiler-copy CodeCoverage Seen
    $filteredNotSeenFunctions = profiler-copy CodeCoverage NotSeen
    if ($false -eq (TabContentsEquivalent $seenFunctions $filteredSeenFunctions))
    {
        WriteError "After reset seen function list not equal"
        $containedFailure = $true
    }
    if ($false -eq (TabContentsEquivalent $notSeenFunctions $filteredNotSeenFunctions))
    {
        WriteError "After reset not seen function list not equal"
        $containedFailure = $true
    }

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }

    return $containedFailure
}

function C2755_CallTreeWindow
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count

    profiler-select Functions -DeselectAll

    $callTree = profiler-copy CallTree

    profiler-showtab CallTree
    profiler-treecontrol CallTree -Action ExpandAll
    $expandedCallTree = profiler-copy CallTree
    profiler-treecontrol CallTree -Action CollapseAll
    $collapsedCallTree = profiler-copy CallTree

    if ($callTree -ne $collapsedCallTree)
    {
        WriteError "Default and recollapsed tree should be equal"
        $containedFailure = $true
    }
    if ($callTree -eq $expandedCallTree)
    {
        WriteError "Default and expanded tree should be different"
        $containedFailure = $true
    }
    if ($collapsedCallTree -eq $expandedCallTree)
    {
        WriteError "Expand and recollapsed tree should be different"
        $containedFailure = $true
    }

    # Reset
    #  nothing to do

    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }

    return $containedFailure
}

# Set default state
#profiler-showtab Functions
#profiler-showtab Info
#profiler-select Functions -DeselectAll
#write-output "clearing Functions"
#profiler-filter Functions $false $false ""
#write-output "clearing Last"
#profiler-filter Last $false $false ""
#write-output "clearing Threads"
#profiler-filter Threads $false $false ""
#write-output "clearing Counters"
#profiler-filter Counters $false $false ""
#write-output "clearing CallTree"
#profiler-filter CallTree $false $false ""
#write-output "clearing CodeCoverage"
#profiler-filter CodeCoverage $false $false ""
#profiler-sort Functions Self

RunTest("SyncToDevKit")
RunTest("C2748_CreateNewProfile")
RunTest("C2749_LoadExistingProfile")
RunTest("C2751_UnitRibbonBar")
RunTest("C2752_AnalysisTools")
RunTest("C2753_Filters")
RunTest("C2754_CodeCoverageWindow")
RunTest("C2755_CallTreeWindow")

TestsFinished
