﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/prepo/detail/prepo_PlayReportGenerator.h>
#include <nn/prepo/detail/service/core/prepo_ReportBuffer.h>
#include <nn/prepo/detail/service/core/prepo_SystemInfo.h>

namespace
{
    int g_CountOfNormal;
    int g_CountOfNoCheck;
}

TEST(Initialize, Main)
{
    nn::time::Initialize();

    nn::prepo::detail::service::core::SystemInfo::SetOperationMode(0);
}

TEST(Add, Normal)
{
    nn::Bit8 data[nn::prepo::ReportBufferSizeMax];
    size_t dataSize;

    nn::prepo::detail::PlayReportGenerator::Initialize(&dataSize, data, sizeof (data));

    auto Add = [&](const char* key, auto value) -> nn::Result
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(
            &dataSize, key, value, data, sizeof (data), dataSize);
    };

    NNT_ASSERT_RESULT_SUCCESS(Add("x", 1.23));
    NNT_ASSERT_RESULT_SUCCESS(Add("y", 4.56));
    NNT_ASSERT_RESULT_SUCCESS(Add("z", 7.89));

    const char eventId[] = "dummy";
    const nn::ApplicationId appId = { 0x1234567890ABCDEF };

    nn::Bit8 sysInfo[nn::prepo::detail::service::core::SystemInfo::RequiredBufferSize];
    size_t sysInfoSize;

    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::detail::service::core::SystemInfo::Collect(
        &sysInfoSize, sysInfo, sizeof (sysInfo), appId, eventId));

    auto& buffer = nn::prepo::detail::service::core::ReportBuffer::GetInstance();

    for (int i = 0; i < 10000; i++)
    {
        int count;
        nn::os::GenerateRandomBytes(&count, sizeof (count));
        count &= 0x7FFFFFFF;
        count %= 400; // 追加しすぎて溢れないようにする。

        g_CountOfNormal = 0;
        g_CountOfNoCheck = 0;

        for (int j = 0; j < count; j++)
        {
            int random;
            nn::os::GenerateRandomBytes(&random, sizeof (random));
            random &= 0x7FFFFFFF;

            if (random % 2 == 0)
            {
                NNT_ASSERT_RESULT_SUCCESS(buffer.Add(nn::prepo::ReportCategory_Normal,
                    data, dataSize, sysInfo, sysInfoSize));
                g_CountOfNormal++;
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(buffer.Add(nn::prepo::ReportCategory_AntiPiracy,
                    data, dataSize, sysInfo, sysInfoSize));
                g_CountOfNoCheck++;
            }
        }

        NNT_ASSERT_RESULT_SUCCESS(buffer.Flush([](nn::prepo::ReportCategory category, void* data, size_t dataSize, int count) -> nn::Result
        {
            if (category == nn::prepo::ReportCategory_Normal)
            {
                NN_RESULT_THROW_UNLESS(count == g_CountOfNormal, nn::prepo::ResultInvalidReportData());
            }
            else
            {
                NN_RESULT_THROW_UNLESS(category == nn::prepo::ReportCategory_AntiPiracy, nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(count == g_CountOfNoCheck, nn::prepo::ResultInvalidReportData());
            }

            nn::prepo::detail::msgpack::InputStreamParam stream = {static_cast<nn::Bit8*>(data), dataSize};

            for (int i = 0; i < count; i++)
            {
                nn::prepo::detail::msgpack::AnyData any = {};

                NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_Map, nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(any.num == 2, nn::prepo::ResultInvalidReportData());

                NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_String, nn::prepo::ResultInvalidReportData());

                const char* pSysInfoKey = reinterpret_cast<const char*>(&stream.data[any.string.position]);
                NN_RESULT_THROW_UNLESS(nn::util::Strncmp(pSysInfoKey, "sys_info", static_cast<int>(any.string.length)) == 0, nn::prepo::ResultInvalidReportData());

                NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_Map, nn::prepo::ResultInvalidReportData());

                const int sysInfoCount = static_cast<int>(any.num);

                for (int j = 0; j < sysInfoCount; j++)
                {
                    // キー
                    NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                    NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_String, nn::prepo::ResultInvalidReportData());

                    // バリュー
                    NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                }

                NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_String, nn::prepo::ResultInvalidReportData());

                const char* pDataKey = reinterpret_cast<const char*>(&stream.data[any.string.position]);
                NN_RESULT_THROW_UNLESS(nn::util::Strncmp(pDataKey, "data", static_cast<int>(any.string.length)) == 0, nn::prepo::ResultInvalidReportData());

                NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_Map, nn::prepo::ResultInvalidReportData());

                const int dataCount = static_cast<int>(any.num);

                for (int j = 0; j < dataCount; j++)
                {
                    // キー
                    NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                    NN_RESULT_THROW_UNLESS(any.type == nn::prepo::detail::msgpack::AnyDataType_String, nn::prepo::ResultInvalidReportData());

                    // バリュー
                    NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::ReadCurrent(&any, &stream), nn::prepo::ResultInvalidReportData());
                }
            }
            NN_RESULT_THROW_UNLESS(stream.GetRemainSize() == 0, nn::prepo::ResultInvalidReportData());

            NN_RESULT_SUCCESS;
        }));
    }
}
