﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_Result.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/prepo/detail/prepo_ApiDetail.h>
#include <nn/prepo/detail/prepo_PlayReportGenerator.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <cstdlib>

#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>

#include "../Common/testPrepo_Common.h"

namespace
{
    int g_UserCount;
    nn::account::Uid g_Users[nn::account::UserCountMax] = {};
}

TEST(PlayReportGenerator, Initialize)
{
    nn::account::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&g_UserCount,
        g_Users, nn::account::UserCountMax));
    if (!(g_UserCount > 0))
    {
        NN_LOG("This test needs one or more users.\n");
        ASSERT_TRUE(g_UserCount > 0);
    }
}

TEST(PlayReportGenerator, Basic)
{
    nn::prepo::PlayReport report;
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));

    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidKey, report.Add("*", 1.23));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultDuplicatedKey, report.Add("Position:x", 1.23));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultDuplicatedKey, report.Add("POSITION:X", 1.23));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x123", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x12", 1.2));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x1", 1.0));
    report.Clear();

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));
    EXPECT_EQ(report.GetCount(), 3);
    report.Clear();

    nn::prepo::Any64BitId idDec = {1};
    nn::prepo::Any64BitId idHex = {0x1234567890ABCDEF};
    nn::prepo::Any64BitId idBeef = {0xbeefBEEFbeefBEEF};

    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:dec", idDec));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:hex", idHex));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:beef", idBeef));
    EXPECT_EQ(report.GetCount(), 3);

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("abcdefghijklmnopqrstuvwxyz_"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("0123456789"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("_"));

    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidEventId, report.SetEventId("ABCDEFGHIJKLMNOPQRSTUVWXYZ"));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidEventId, report.SetEventId(":"));

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("prepo_generator_basic"));
    NNT_EXPECT_RESULT_SUCCESS(report.Save());
}

TEST(PlayReportGenerator, UserSepcifying)
{
    nn::prepo::PlayReport report("prepo_generator_user_specify");
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x123", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x12", 1.2));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x1", 1.0));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));

    nn::prepo::Any64BitId idDec = {1};
    nn::prepo::Any64BitId idHex = {0x1234567890ABCDEF};
    nn::prepo::Any64BitId idBeef = {0xbeefBEEFbeefBEEF};

    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:dec", idDec));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:hex", idHex));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:beef", idBeef));

    EXPECT_EQ(report.GetCount(), 9);

    NNT_EXPECT_RESULT_SUCCESS(report.Save(g_Users[0]));
}

TEST(PlayReportGenerator, SystemBasic)
{
    nn::prepo::SystemPlayReport report("prepo_generator_sysbasic");
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));

    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidKey, report.Add("*", 1.23));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultDuplicatedKey, report.Add("Position:x", 1.23));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultDuplicatedKey, report.Add("POSITION:X", 1.23));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x123", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x12", 1.2));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x1", 1.0));
    report.Clear();

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));
    EXPECT_EQ(report.GetCount(), 3);
    report.Clear();

    nn::prepo::Any64BitId idDec = {1};
    nn::prepo::Any64BitId idHex = {0x1234567890ABCDEF};
    nn::prepo::Any64BitId idBeef = {0xbeefBEEFbeefBEEF};

    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:dec", idDec));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:hex", idHex));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:beef", idBeef));
    EXPECT_EQ(report.GetCount(), 3);

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("abcdefghijklmnopqrstuvwxyz_"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("0123456789"));
    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("_"));

    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidEventId, report.SetEventId("ABCDEFGHIJKLMNOPQRSTUVWXYZ"));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidEventId, report.SetEventId(":"));

    NNT_EXPECT_RESULT_SUCCESS(report.SetEventId("prepo_generator_basic"));

    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultInvalidApplicationId, report.Save());

    nn::ApplicationId applicationId = {0x0123456789abcdef};
    NNT_EXPECT_RESULT_SUCCESS(report.SetApplicationId(applicationId)); // Application ID を自己申告する。

    NNT_EXPECT_RESULT_SUCCESS(report.Save());
}

TEST(PlayReportGenerator, SystemUserSepcifying)
{
    nn::prepo::SystemPlayReport report("prepo_generator_sysusr_specify");
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x123", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x12", 1.2));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x1", 1.0));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));

    nn::prepo::Any64BitId idDec = {1};
    nn::prepo::Any64BitId idHex = {0x1234567890ABCDEF};
    nn::prepo::Any64BitId idBeef = {0xbeefBEEFbeefBEEF};

    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:dec", idDec));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:hex", idHex));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("id:beef", idBeef));

    EXPECT_EQ(report.GetCount(), 9);

    nn::ApplicationId applicationId = {0x0123456789abcdef};
    NNT_EXPECT_RESULT_SUCCESS(report.SetApplicationId(applicationId));

    NNT_EXPECT_RESULT_SUCCESS(report.Save(g_Users[0]));
}

TEST(PlayReportGenerator, Boundary)
{
    nn::prepo::PlayReport report("1234567890123456789012345678901");

    for (int i = 1; i <= nn::prepo::EventIdLengthMax; i++)
    {
        char eventId[nn::prepo::EventIdLengthMax + 1] = {};

        for (int n = 0; n < i; n++)
        {
            eventId[n] = '1';
        }

        NNT_EXPECT_RESULT_SUCCESS(report.SetEventId(eventId));
    }

    // 3 バイトは MAP() で消費
    // x: 3 バイト消費
    // y: 5 バイト消費
    static char buffer[11];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("x", static_cast<int64_t>(127)));
    NNT_EXPECT_RESULT_SUCCESS(report.Add("y", static_cast<int64_t>(128)));

    EXPECT_EQ(report.GetCount(), 2);

    // Clear
    NNT_EXPECT_RESULT_SUCCESS(report.Save());

    // 8 バイト消費
    NNT_EXPECT_RESULT_SUCCESS(report.Add("abcabc", static_cast<int64_t>(127)));

    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultOutOfResource, report.Add("position:x", 1.23));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultOutOfResource, report.Add("position:y", 0.00));
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultOutOfResource, report.Add("position:z", -5.55));

    EXPECT_EQ(report.GetCount(), 1);

    static char bufferMin[nn::prepo::PlayReport::BufferSizeMin];

    report.SetBuffer(bufferMin, sizeof (bufferMin));

    // 3 バイトキーバリュー追加
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultOutOfResource, report.Add("n", static_cast<int64_t>(0)));

    static char bufferMax[nn::prepo::PlayReport::BufferSizeMax];

    report.SetBuffer(bufferMax, sizeof (bufferMax));

    for (int i = 0; i < nn::prepo::KeyValueCountMax; i++)
    {
        char key[nn::prepo::KeyLengthMax + 1] = {};
        char value[nn::prepo::StringValueLengthMax + 1] = {};

        nn::util::SNPrintf(key, sizeof (key), "key_00000000000000000000000000000000000000000000000000000000%03d", i);
        nn::util::SNPrintf(value, sizeof (value), "value_000000000000000000000000000000000000000000000000000000%03d", i);

        NNT_EXPECT_RESULT_SUCCESS(report.Add(key, value));
    }
    EXPECT_EQ(report.GetCount(), nn::prepo::KeyValueCountMax);

    // 3 バイトキーバリュー追加
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultKeyValueCountLimitReached, report.Add("n", static_cast<int64_t>(0)));
    // 4 バイトキーバリュー追加
    NNT_EXPECT_RESULT_FAILURE(nn::prepo::ResultKeyValueCountLimitReached, report.Add("nn", static_cast<int64_t>(0)));
}

TEST(PlayReportGenerator, Verifier)
{
    size_t length;

    EXPECT_TRUE(nn::prepo::detail::VerifyEventId(&length, "1"));
    EXPECT_TRUE(nn::prepo::detail::VerifyEventId(&length, "abc"));
    EXPECT_TRUE(nn::prepo::detail::VerifyEventId(&length, "_"));
    EXPECT_TRUE(nn::prepo::detail::VerifyEventId(&length, "abc_1"));

    EXPECT_FALSE(nn::prepo::detail::VerifyEventId(&length, ""));
    EXPECT_FALSE(nn::prepo::detail::VerifyEventId(&length, "!"));
    EXPECT_FALSE(nn::prepo::detail::VerifyEventId(&length, "a:b"));

    EXPECT_TRUE(nn::prepo::detail::VerifyEventId(&length, "1234567890123456789012345678901"));
    EXPECT_FALSE(nn::prepo::detail::VerifyEventId(&length, "12345678901234567890123456789012"));

    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, "1"));
    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, "abc"));
    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, "_"));
    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, ":"));
    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, "abc:1"));
    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, "abc_:1"));

    EXPECT_FALSE(nn::prepo::detail::VerifyKey(&length, ""));
    EXPECT_FALSE(nn::prepo::detail::VerifyKey(&length, "!"));

    EXPECT_TRUE(nn::prepo::detail::VerifyKey(&length, "123456789012345678901234567890123456789012345678901234567890123"));
    EXPECT_FALSE(nn::prepo::detail::VerifyKey(&length, "1234567890123456789012345678901234567890123456789012345678901234"));

    EXPECT_TRUE(nn::prepo::detail::VerifyStringValue(&length, ""));
    EXPECT_TRUE(nn::prepo::detail::VerifyStringValue(&length, "123456789012345678901234567890123456789012345678901234567890123"));
    EXPECT_FALSE(nn::prepo::detail::VerifyStringValue(&length, "1234567890123456789012345678901234567890123456789012345678901234"));
}

template <size_t BufferSize>
class DebugPlayReport
{
public:
    //
    nn::Bit8 buffer[BufferSize];
    //
    size_t position;

public:
    DebugPlayReport() NN_NOEXCEPT
    {
        Clear();
    }

    void Clear() NN_NOEXCEPT
    {
        nn::prepo::detail::PlayReportGenerator::Initialize(&position, buffer, BufferSize);
    }

    nn::Result Add(const char* key, int64_t value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }

    nn::Result Add(const char* key, const nn::prepo::Any64BitId& value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }

    nn::Result Add(const char* key, double value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }

    nn::Result Add(const char* key, const char* value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(&position, key, value, buffer, BufferSize, position);
    }
};

TEST(PlayReportGenerator, DataCorruption)
{
    DebugPlayReport<2048> report;

    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));
    EXPECT_TRUE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    report.buffer[0] = 0x01;

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(report.buffer, report.position));

    nn::Bit8 random[256];

    for (int i = 0; i < sizeof (random); i++)
    {
        random[i] = static_cast<nn::Bit8>(std::rand() % 0xFF);
    }

    EXPECT_FALSE(nn::prepo::detail::VerifyReport(random, sizeof (random)));
}

//　HOSTFS 利用時のみ実行できるテストです。
// テストを実行するときは、prepo_FileSystem.cpp 冒頭の ENABLE_HOSTFS を 1 に設定してください。
#if 0
TEST(PlayReportGenerator, CheckFileFormat)
{
    nn::prepo::PlayReport report("fallen_event");
    char buffer[1024];

    report.SetBuffer(buffer, sizeof (buffer));

    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::ClearStorage());
    NN_UTIL_SCOPE_EXIT
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::prepo::ClearStorage());
    };

    for (int64_t i = 0; i < 10000; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(report.Add("count", i));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));
        EXPECT_EQ(report.GetCount(), 4);
        NNT_EXPECT_RESULT_SUCCESS(report.Save());
    }

    nn::fs::MountHost("prepo", "C:\\siglo\\prepo");
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("prepo");
    };

    nn::fs::DirectoryHandle dirHandle = {};
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dirHandle, "prepo:/data/", nn::fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(dirHandle);
    };

    static nn::fs::DirectoryEntry s_DirEntry[100];
    int64_t readNum = 0;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readNum, s_DirEntry, dirHandle, 2));
    for (int i = 0; i < readNum; i++)
    {
        static char s_Path[256];
        nn::util::SNPrintf(s_Path, sizeof(s_Path), "prepo:/data/%s", s_DirEntry[i].name);

        nn::Bit8 s_ReadBuffer[128 * 1024];

        size_t readSize;
        NNT_EXPECT_RESULT_SUCCESS(nnt::prepo::ReadFile(&readSize, s_Path, s_ReadBuffer, sizeof(s_ReadBuffer)));

        EXPECT_EQ(readSize, static_cast<size_t>(s_DirEntry[i].fileSize));

        NN_LOG("Verify %s\n", s_Path);
        //EXPECT_TRUE(nn::prepo::detail::VerifyReportFile(s_ReadBuffer, s_DirEntry[i].fileSize));
    }
}
#endif
