﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/fs.h>
#include <nn/os.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_Base64.h>

#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>

#include <nn/srepo/srepo_Types.h>

#include <nn/prepo/prepo_Api.h>
#include <nn/prepo/prepo_ApiDebug.h>

#include <nn/prepo/detail/prepo_PlayReportGenerator.h>

#include <nn/prepo/detail/msgpack/prepo_MessagePack.h>

#include <nn/prepo/detail/service/core/prepo_SystemInfo.h>
#include <nn/prepo/detail/service/core/prepo_SystemReport.h>

namespace nnt { namespace prepo {

template <typename T>
T GetRandom() NN_NOEXCEPT
{
    T random;
    nn::os::GenerateRandomBytes(&random, sizeof (T));
    return random;
}

bool FileExists(const char* path) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType = {};
    return true
        && nn::fs::GetEntryType(&entryType, path).IsSuccess()
        && entryType == nn::fs::DirectoryEntryType_File;
}

bool DirectoryExists(const char* path) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType = {};
    return true
        && nn::fs::GetEntryType(&entryType, path).IsSuccess()
        && entryType == nn::fs::DirectoryEntryType_Directory;
}

nn::Result WriteFile(const char* path, const void* buffer, size_t size) NN_NOEXCEPT
{
    bool isNew = false;

    nn::fs::FileHandle handle = {};

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_DO(nn::fs::CreateFile(path, static_cast<int64_t>(size)));
            NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

            isNew = true;
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    if (!isNew)
    {
        int64_t fileSize;

        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

        if (!(static_cast<size_t>(fileSize) == size))
        {
            NN_RESULT_DO(nn::fs::SetFileSize(handle, size));
        }
    }

    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, buffer, size,
        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

template <typename T>
nn::Result WriteFile(const char* path, const T& value) NN_NOEXCEPT
{
    return WriteFile(path, &value, sizeof (T));
}

namespace
{
    nn::Result ReadFileImpl(size_t* outValue, const char* path, void* buffer, size_t size, bool isJustSizeRequired) NN_NOEXCEPT
    {
        nn::fs::FileHandle handle = {};

        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t fileSize = 0;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

        if (isJustSizeRequired)
        {
            NN_ABORT_UNLESS_EQUAL(static_cast<size_t>(fileSize), size);
        }
        else
        {
            NN_ABORT_UNLESS_LESS_EQUAL(static_cast<size_t>(fileSize), size);
        }

        NN_RESULT_DO(nn::fs::ReadFile(outValue, handle, 0, buffer, size));

        NN_RESULT_SUCCESS;
    }
}

nn::Result ReadFile(size_t* outValue, const char* path, void* buffer, size_t size) NN_NOEXCEPT
{
    return ReadFileImpl(outValue, path, buffer, size, false);
}

template <typename T>
nn::Result ReadFile(T* out, const char* path) NN_NOEXCEPT
{
    size_t readSize;
    return ReadFileImpl(&readSize, path, out, sizeof (T), true);
}

nn::Result GetFileSize(int64_t* out, const char* path) NN_NOEXCEPT
{
    nn::fs::FileHandle handle = {};

    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::GetFileSize(out, handle));

    NN_RESULT_SUCCESS;
}

const char* GetTransmissionStatusString(nn::prepo::TransmissionStatus status) NN_NOEXCEPT
{
    switch (status)
    {
    case nn::prepo::TransmissionStatus_Idle:
        return "Idle";
    case nn::prepo::TransmissionStatus_Pending:
        return "Pending";
    case nn::prepo::TransmissionStatus_Processing:
        return "Processing";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool WaitForTransmissionStatusChanged(nn::prepo::TransmissionStatus status, nn::TimeSpan timeout) NN_NOEXCEPT
{
    const auto startTick = nn::os::GetSystemTick();

    do
    {
        nn::prepo::TransmissionStatus current;
        NNT_EXPECT_RESULT_SUCCESS(nn::prepo::GetTransmissionStatus(&current));

        if (current == nn::prepo::TransmissionStatus_Pending)
        {
            const auto error = nn::prepo::GetLastUploadError();
            NN_LOG("Current transmission status: %s (%03d-%04d)\n",
                GetTransmissionStatusString(current),
                error.GetModule(), error.GetDescription());
        }
        else
        {
            NN_LOG("Current transmission status: %s\n",
                GetTransmissionStatusString(current));
        }

        if (current == status)
        {
            return true;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

    } while ((nn::os::GetSystemTick() - startTick).ToTimeSpan() < timeout);

    return false;
}

class MessagePackVerifier
{
public:
    MessagePackVerifier(const nn::Bit8* data, size_t size) NN_NOEXCEPT
        : m_Stream({ data, size })
    {
    }
    bool VerifyMap(size_t expected) NN_NOEXCEPT
    {
        return true
            && nn::prepo::detail::msgpack::ReadCurrent(&m_Any, &m_Stream)
            && m_Any.type == nn::prepo::detail::msgpack::AnyDataType_Map
            && m_Any.num == expected;
    }
    bool VerifyString(const char* expected) NN_NOEXCEPT
    {
        return true
            && nn::prepo::detail::msgpack::ReadCurrent(&m_Any, &m_Stream)
            && m_Any.type == nn::prepo::detail::msgpack::AnyDataType_String
            && static_cast<size_t>(nn::util::Strnlen(expected, nn::srepo::KeyLengthMax)) == m_Any.string.length
            && nn::util::Strncmp(expected, reinterpret_cast<const char*>(&m_Stream.data[m_Any.string.position]),
                static_cast<int>(m_Any.string.length)) == 0;
    }
    bool VerifyDouble(double expected) NN_NOEXCEPT
    {
        return true
            && nn::prepo::detail::msgpack::ReadCurrent(&m_Any, &m_Stream)
            && m_Any.type == nn::prepo::detail::msgpack::AnyDataType_Float64
            && m_Any.f64 == expected;
    }
    bool VerifyBinary(const void* expected, size_t expectedSize) NN_NOEXCEPT
    {
        return true
            && nn::prepo::detail::msgpack::ReadCurrent(&m_Any, &m_Stream)
            && m_Any.type == nn::prepo::detail::msgpack::AnyDataType_Binary
            && m_Any.binary.length == expectedSize
            && std::memcmp(expected, &m_Stream.data[m_Any.string.position], expectedSize) == 0;
    }
    bool IsEnd() NN_NOEXCEPT
    {
        return m_Stream.position == m_Stream.size;
    }
private:
    nn::prepo::detail::msgpack::InputStreamParam m_Stream;
    nn::prepo::detail::msgpack::AnyData m_Any;
};

class ReportWriter
{
public:
    ReportWriter(nn::Bit8* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Buffer(buffer)
        , m_BufferSize(bufferSize)
    {
    }
    nn::Result Initialize(const char* eventId, const nn::ApplicationId& appId, const nn::account::Uid& user) NN_NOEXCEPT
    {
        nn::prepo::detail::msgpack::OutputStreamParam stream = { m_Buffer, m_BufferSize };

        NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::WriteMapAutoSize(&stream, 2), nn::prepo::ResultOutOfResource());

        NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::WriteString(&stream, "sys_info", sizeof("sys_info") - 1), nn::prepo::ResultOutOfResource());
        size_t sysInfoSize = 0;
        NN_RESULT_DO(nn::prepo::detail::service::core::SystemInfo::Collect(
            &sysInfoSize, stream.buffer + stream.position, stream.GetRemainSize(), appId, eventId, user));
        stream.position += sysInfoSize;

        NN_RESULT_THROW_UNLESS(nn::prepo::detail::msgpack::WriteString(&stream, "data", sizeof("data") - 1), nn::prepo::ResultOutOfResource());
        m_Data = stream.buffer + stream.position;
        m_DataSize = stream.GetRemainSize();
        nn::prepo::detail::PlayReportGenerator::Initialize(&m_DataPosition, m_Data, m_DataSize);

        NN_RESULT_SUCCESS;
    }
    nn::Result Initialize(const char* eventId, const nn::ApplicationId& appId) NN_NOEXCEPT
    {
        return Initialize(eventId, appId, nn::account::InvalidUid);
    }
    template <typename T>
    nn::Result Add(const char* key, T value) NN_NOEXCEPT
    {
        return nn::prepo::detail::PlayReportGenerator::AddKeyValue(
            &m_DataPosition, key, value, m_Data, m_DataSize, m_DataPosition);
    }
    size_t GetSize() NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(m_Data + m_DataPosition) - reinterpret_cast<uintptr_t>(m_Buffer);
    }
private:
    nn::Bit8* m_Buffer;
    size_t m_BufferSize;
    nn::Bit8* m_Data;
    size_t m_DataSize;
    size_t m_DataPosition;
};

nn::Result GetNetworkServiceAccountId(nn::account::NetworkServiceAccountId* out, const nn::account::Uid& uid) NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountManager manager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&manager, uid));

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(manager.GetNetworkServiceAccountId(&accountId));

    *out = accountId;

    NN_RESULT_SUCCESS;
}

}}
