﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <utility>  // std::pair
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/gpio/gpio.h>
#include <nn/psc/psc_Types.h>
#include <nn/psc/psc_PmControlTypes.h>
#include <nn/cec/cec_Api.h>

// テスト対象
#include <nn/spsm/server/spsm_UniqueMessageQueue.h>

namespace
{
    using namespace ::nn::spsm;

    struct TestMessage
    {
        int message;
        int identifier;

        TestMessage() NN_NOEXCEPT : message(0), identifier(0) {}
        TestMessage(int m, int i) NN_NOEXCEPT  : message(m), identifier(i) {}

        NN_IMPLICIT operator int() NN_NOEXCEPT { return message; }
        NN_IMPLICIT operator const int() const NN_NOEXCEPT { return message; }
    };
    const int TestMessageIdMin= 3;
    const int TestMessageIdMax = 20;
    typedef nn::spsm::server::UniqueMessageQueue<TestMessage, TestMessageIdMin, TestMessageIdMax> TestMessageQueue;
} // anonymous namespace

TEST(UniqueMessageQueueTest, Basic)
{
    TestMessageQueue q;
    TestMessage message;

    // 最初は空
    EXPECT_FALSE(q.TryDequeue(&message));

    q.Enqueue(TestMessage(TestMessageIdMin, 2));
    EXPECT_TRUE(q.TryDequeue(&message));
    EXPECT_EQ(TestMessageIdMin, message.message);
    EXPECT_FALSE(q.TryDequeue(&message));
}

TEST(UniqueMessageQueueTest, Reset)
{
    TestMessageQueue q;
    TestMessage message;

    for ( int m = TestMessageIdMin; m < TestMessageIdMax; ++m )
    {
        q.Enqueue(TestMessage(m, m));
    }
    q.ResetQueue();
    EXPECT_FALSE(q.TryDequeue(&message));
}

TEST(UniqueMessageQueueTest, RangeBoundary)
{
    TestMessageQueue q;

    q.Enqueue(TestMessage(TestMessageIdMin, INT_MIN));
    EXPECT_DEATH_IF_SUPPORTED(q.Enqueue(TestMessage(TestMessageIdMin  - 1, INT_MIN)), "");

    q.Enqueue(TestMessage(TestMessageIdMax - 1, INT_MAX));
    EXPECT_DEATH_IF_SUPPORTED(q.Enqueue(TestMessage(TestMessageIdMax, INT_MAX)), "");
}

// 同一とみなせるメッセージを複数キューイングすると、最後に投げたメッセージだけが取り出される
TEST(UniqueMessageQueueTest, OverwriteSameMessage)
{
    TestMessageQueue q;

    for ( int identifier = 0; identifier <= 10; ++identifier )
    {
        q.Enqueue(TestMessage(TestMessageIdMin, identifier));
    }

    TestMessage message;
    EXPECT_TRUE(q.TryDequeue(&message));
    EXPECT_EQ(TestMessageIdMin, message.message);
    EXPECT_EQ(10, message.identifier);
    EXPECT_FALSE(q.TryDequeue(&message));
}

// メッセージの値によって取り出し順序が整列される
TEST(UniqueMessageQueueTest, MessageOrder)
{
    TestMessageQueue q;
    TestMessage temp;

    q.Enqueue(TestMessage(TestMessageIdMin + 1, 1));
    q.Enqueue(TestMessage(TestMessageIdMin + 5, 5));
    q.Enqueue(TestMessage(TestMessageIdMin + 3, 3));
    q.Enqueue(TestMessage(TestMessageIdMin + 2, 2));
    q.Enqueue(TestMessage(TestMessageIdMax - 1, TestMessageIdMax - 1));
    q.Enqueue(TestMessage(TestMessageIdMin + 7, 7));
    q.Enqueue(TestMessage(TestMessageIdMin + 0, 0));
    q.Enqueue(TestMessage(TestMessageIdMin + 12, 12));
    q.Enqueue(TestMessage(TestMessageIdMin + 4, 4));
    const int TotalMessageCount = 9;

    TestMessage messages[TotalMessageCount];
    for ( auto& message : messages )
    {
        message = q.Dequeue();
    }
    EXPECT_FALSE(q.TryDequeue(&temp));

    EXPECT_EQ(TestMessageIdMin + 0, messages[0].message);
    EXPECT_EQ(TestMessageIdMin + 1, messages[1].message);
    EXPECT_EQ(TestMessageIdMin + 2, messages[2].message);
    EXPECT_EQ(TestMessageIdMin + 3, messages[3].message);
    EXPECT_EQ(TestMessageIdMin + 4, messages[4].message);
    EXPECT_EQ(TestMessageIdMin + 5, messages[5].message);
    EXPECT_EQ(TestMessageIdMin + 7, messages[6].message);
    EXPECT_EQ(TestMessageIdMin + 12, messages[7].message);
    EXPECT_EQ(TestMessageIdMax - 1, messages[8].message);
}
