﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/account.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/account/account_Result.h>
#include <nn/friends/friends_ApiAdmin.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_Utility.h>
#include <nn/ns/ns_UserResourceManagementApi.h>
#include <nn/oe.h>
#include <nn/os.h>
#include <nn/pdm/detail/pdm_Config.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/pdm/pdm_QueryApi.h>
#include <nn/pdm/pdm_QueryApiForDebug.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;

namespace {

    //! @brief      account::OpenUser/CloseUser の呼び出しレイテンシ。
    //!
    //! @details    ユーザ状態遷移イベントが pdm に通知される前に次の状態遷移要求を発生させないようにするための目安。
    //!             pdm のログ出力量やユーザアカウントイベント監視スレッドの処理負荷などに依存するため。
    //!             ちなみに要求競合で呼び出しが矛盾するケースをテストしたければ 5ms などで良い。
    //!             その場合はアカウントシステム側の整合性管理の検証が主題になるので、まぁ、このテストですべきことではない。
    static const TimeSpan LatencyForAccountOpenCloseNotification = TimeSpan::FromMilliSeconds(30);

    os::Mutex g_AccountLibraryMutex(true);

    Result OpenUserPrivate(account::UserHandle* pOutHandle, const account::Uid& user) NN_NOEXCEPT
    {
        g_AccountLibraryMutex.Lock();
        auto result = account::OpenUser(pOutHandle, user);
        if(result.IsFailure())
        {
            g_AccountLibraryMutex.Unlock();
            return result;
        }
        os::SleepThread(LatencyForAccountOpenCloseNotification);
        NN_RESULT_SUCCESS;
    }

    void CloseUserPrivate(const account::UserHandle& handle) NN_NOEXCEPT
    {
        account::CloseUser(handle);
        g_AccountLibraryMutex.Unlock();
        os::SleepThread(LatencyForAccountOpenCloseNotification);
    }

    Result DeleteUserPrivate(const account::Uid& user, bool withLatency) NN_NOEXCEPT
    {
        const TimeSpan latency = TimeSpan::FromMilliSeconds(50);
        const TimeSpan deleteTimeout = TimeSpan::FromMilliSeconds(10000);

        {
            std::lock_guard<decltype(g_AccountLibraryMutex)> lock(g_AccountLibraryMutex);
            account::Finalize();
            account::InitializeForAdministrator();
            NN_UTIL_SCOPE_EXIT
            {
                account::Finalize();
                account::Initialize();
            };

            nn::friends::DaemonSuspension friendsSuspension;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::friends::SuspendDaemon(&friendsSuspension));
            nn::ns::ProgressMonitorForDeleteUserSaveDataAll progress;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&progress, user));
            nn::os::SystemEvent deletedEvent;
            progress.GetSystemEvent(&deletedEvent);
            NN_ABORT_UNLESS(deletedEvent.TimedWait(deleteTimeout));
            NN_ABORT_UNLESS_RESULT_SUCCESS(progress.GetResult());
            NN_RESULT_DO(account::DeleteUser(user));
        }

        if(withLatency)
        {
            os::SleepThread(latency);
        }

        NN_RESULT_SUCCESS;
    }

    Result DeleteUserPrivate(const account::Uid& user) NN_NOEXCEPT
    {
        return DeleteUserPrivate(user, true);
    }

    void RegisterTestUsers()
    {
        account::Uid testUsers[] = {account::InvalidUid};

        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::GetUserCount(&count));
        if(NN_ARRAY_SIZE(testUsers) <= count)
        {
            return;
        }

        {
            std::lock_guard<decltype(g_AccountLibraryMutex)> lock(g_AccountLibraryMutex);
            account::Finalize();
            account::InitializeForAdministrator();
            NN_UTIL_SCOPE_EXIT
            {
                account::Finalize();
                account::Initialize();
            };

            for (int i = 0; i < NN_ARRAY_SIZE(testUsers); ++i)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::BeginUserRegistration(&testUsers[i]));
                NN_ABORT_UNLESS_RESULT_SUCCESS(account::CompleteUserRegistrationForcibly(testUsers[i]));
            }
        }
    }

    time::PosixTime GetPosixTimeFromElapsedMinutes(uint32_t elapsedMinSincePosixTimeMin)
    {
        return time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(static_cast<int64_t>(elapsedMinSincePosixTimeMin));
    }

    void GetDateTimeString(char* outBuffer, size_t outBufferSize, time::PosixTime posixTime)
    {
        if( posixTime < time::InputPosixTimeMin || posixTime > time::InputPosixTimeMax )
        {
            util::SNPrintf(outBuffer, outBufferSize, "(Invalid)");
            return;
        }

        time::CalendarTime calendarTime;
        time::CalendarAdditionalInfo calendarAdditionalInfo;
        auto result = time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
        if( result.IsSuccess() )
        {
            util::SNPrintf(outBuffer, outBufferSize, "%04d-%02d-%02d %02d:%02d:%02d (%s)",
                calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second,
                calendarAdditionalInfo.timeZone.standardTimeName);
        }
        else
        {
            util::SNPrintf(outBuffer, outBufferSize, "(GetDateTimeString failed : 0x%08x)", result.GetInnerValueForDebug());
        }
    }

    void DumpEventTimeData(const pdm::EventTimeData& eventTimeData)
    {
        const auto TimeStringSize = 64;

        char userTimeString[TimeStringSize];
        char networkTimeString[TimeStringSize];
        GetDateTimeString(userTimeString, TimeStringSize, GetPosixTimeFromElapsedMinutes(eventTimeData.userClockTime));
        GetDateTimeString(networkTimeString, TimeStringSize, GetPosixTimeFromElapsedMinutes(eventTimeData.networkClockTime));

        NN_LOG("Index         : %d\n", eventTimeData.eventIndex);
        NN_LOG("UserClockTime : %s\n", userTimeString);
        NN_LOG("NetClockTime  : %s\n", networkTimeString);
    }

    void DumpPlayStatistics(const pdm::PlayStatistics& playStatistics)
    {
        auto separator = "------------------------------\n";
        NN_LOG(separator);
        NN_LOG("ApplicationId : 0x%016llx\n", playStatistics.applicationId.value);
        NN_LOG("Play Count    : %u\n", playStatistics.totalPlayCount);
        NN_LOG("Play Time     : %u minutes\n", playStatistics.totalPlayTime);
        NN_LOG("[First Play]\n");
        DumpEventTimeData(playStatistics.firstEventTime);
        NN_LOG("[Latest Play]\n");
        DumpEventTimeData(playStatistics.latestEventTime);
        NN_LOG(separator);
    }

    const int PlayEventCount = 2048;
    pdm::PlayEvent g_PlayEvent[PlayEventCount];
    const ncm::ApplicationId SelfApplicationId{ 0x0005000C10000000 };
    const int StackSize = 8 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackForAccount[StackSize];
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStackForQuery[StackSize];
    os::Event g_AccountThreadStopEvent(os::EventClearMode_ManualClear);
    os::Event g_QueryThreadStopEvent(os::EventClearMode_ManualClear);

    void AccountOpenCloseThreadFunc(void* arg)
    {
        auto pUid = reinterpret_cast<account::Uid*>(arg);
        account::UserHandle userHandle;
        auto count = 0;
        while( NN_STATIC_CONDITION(1) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(OpenUserPrivate(&userHandle, *pUid));
            CloseUserPrivate(userHandle);
            count++;
            if( g_AccountThreadStopEvent.TryWait() )
            {
                g_AccountThreadStopEvent.Clear();
                break;
            }
        }
        NN_LOG("Exit from AccountOpenCloseThreadFunc. count = %d.\n", count);
    }

    void QueryPlayStatisticsThreadFunc(void* arg)
    {
        auto pUid = reinterpret_cast<account::Uid*>(arg);
        int count = 0;
        while( NN_STATIC_CONDITION(1) )
        {
            auto statistics = pdm::QueryPlayStatistics(SelfApplicationId, *pUid);
            NN_UNUSED(statistics);
            os::SleepThread(TimeSpan::FromMilliSeconds(50));
            count++;
            if( g_QueryThreadStopEvent.TryWait() )
            {
                break;
            }
        }
        NN_LOG("Exit from QueryPlayStatisticsThreadFunc. QueryCount = %d.\n", count);
    }

    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[StackSize];
    os::Event g_ThreadStopEvent(os::EventClearMode_ManualClear);

    void NotifyPlayEventThreadFunc(void* arg)
    {
        auto pUid = reinterpret_cast<account::Uid*>(arg);
        account::UserHandle userHandle;
        auto count = 0;

        while( NN_STATIC_CONDITION(1) )
        {
            // Launch
            pdm::NotifyAppletEvent(pdm::AppletEventType::Launch, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
            pdm::NotifyAppletEvent(pdm::AppletEventType::InFocus, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

            // Open / Close
            if(OpenUserPrivate(&userHandle, *pUid).IsSuccess())
            {
                CloseUserPrivate(userHandle);
            }

            // Exit
            pdm::NotifyAppletEvent(pdm::AppletEventType::Background, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
            pdm::NotifyAppletEvent(pdm::AppletEventType::Exit, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

            auto statistics = pdm::QueryPlayStatistics(SelfApplicationId, *pUid);
            NN_UNUSED(statistics);

            os::SleepThread(TimeSpan::FromMilliSeconds(50));
            count++;
            if( g_ThreadStopEvent.TryWait() )
            {
                g_ThreadStopEvent.Clear();
                break;
            }
        }
        NN_LOG("Exit from NotifyPlayEventThreadFunc. count = %d.\n", count);
    }

    void SuspendResumeThreadFunc(void* arg)
    {
        auto pUid = reinterpret_cast<account::Uid*>(arg);
        auto count = 0;

        while( NN_STATIC_CONDITION(1) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(*pUid));

            // ランダムな時間待つ
            os::SleepThread(TimeSpan::FromMilliSeconds(std::rand() % 1000));

            NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::ResumeUserAccountEventService(*pUid));

            // ランダムな時間待つ
            os::SleepThread(TimeSpan::FromMilliSeconds(std::rand() % 1000));

            count++;
            if( g_ThreadStopEvent.TryWait() )
            {
                g_ThreadStopEvent.Clear();
                break;
            }
        }
        NN_LOG("Exit from SuspendResumeThreadFunc. count = %d.\n", count);
    }
}

class StressTest : public testing::Test
{
protected:
    virtual void SetUp()
    {
        RegisterTestUsers();
    }
    virtual void TearDown()
    {
    }
    static void SetUpTestCase()
    {
        int64_t tick = nn::os::GetSystemTick().GetInt64Value();
        std::srand(static_cast<unsigned>((tick >> 32) ^ tick));

        oe::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
        account::Initialize();
        pdm::InitializeForQuery();
        pdm::InitializeForNotification();
    }
    static void TearDownTestCase()
    {
        time::Finalize();
        account::Finalize();
        pdm::FinalizeForQuery();
        pdm::FinalizeForNotification();
    }
};

// 所要時間 15 ~ 20 分。
TEST_F(StressTest, FillPlayEvent)
{
    int actualLength;
    account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    pdm::NotifyClearAllEvent();

    // 最大数のイベントを埋めるを 2 回繰り返す。
    const int RepeatNum = 2;
    for( int n = 0; n < RepeatNum; n++ )
    {
        for( uint32_t i = 0; i < pdm::detail::PlayEventCountMax; i++ )
        {
            pdm::NotifyAppletEvent(pdm::AppletEventType::Exit, SelfApplicationId, n * pdm::detail::PlayEventCountMax + i, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
        }

        int start;
        int last;
        int count = pdm::GetAvailablePlayEventRange(&start, &last);
        ASSERT_EQ(pdm::detail::PlayEventCountMax, static_cast<decltype(pdm::detail::PlayEventCountMax)>(count));
        ASSERT_EQ(n * pdm::detail::PlayEventCountMax, start);
        ASSERT_EQ((n + 1) * pdm::detail::PlayEventCountMax - 1, static_cast<decltype(pdm::detail::PlayEventCountMax)>(last));

        int offset = 0;
        while( NN_STATIC_CONDITION(true) )
        {
            auto readCount = pdm::QueryPlayEvent(g_PlayEvent, PlayEventCount, offset);
            if( readCount == 0 )
            {
                break;
            }
            for( int i = 0; i < readCount; i++ )
            {
                ASSERT_EQ(start + offset + i, g_PlayEvent[i].appletEventData.version);
            }
            offset += readCount;
        }
    }

    // リングバッファの先頭に少しのイベントを追加。
    const int AppendNum = 10;
    for( uint32_t i = 0; i < AppendNum; i++ )
    {
        pdm::NotifyAppletEvent(pdm::AppletEventType::Exit, SelfApplicationId, RepeatNum * pdm::detail::PlayEventCountMax + i, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    }
    int start;
    int last;
    int count = pdm::GetAvailablePlayEventRange(&start, &last);
    ASSERT_EQ(pdm::detail::PlayEventCountMax, static_cast<decltype(pdm::detail::PlayEventCountMax)>(count));
    ASSERT_EQ((RepeatNum - 1) * pdm::detail::PlayEventCountMax + AppendNum, start);
    ASSERT_EQ(RepeatNum * pdm::detail::PlayEventCountMax + AppendNum - 1, static_cast<decltype(pdm::detail::PlayEventCountMax)>(last));

    int offset = 0;
    while( NN_STATIC_CONDITION(true) )
    {
        auto readCount = pdm::QueryPlayEvent(g_PlayEvent, PlayEventCount, offset);
        if( readCount == 0 )
        {
            break;
        }
        for( int i = 0; i < readCount; i++ )
        {
            ASSERT_EQ(start + offset + i, g_PlayEvent[i].appletEventData.version);
        }
        offset += readCount;
    }
}

TEST_F(StressTest, SequentialAccountOpenCloseAndQuery)
{
    int actualLength;
    account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }
    pdm::NotifyClearAllEvent(); // AccountPlayEventFactory::Condition はクリアされない。
    // テストアプリケーションが起動状態なので AccountPlayEventFactory::Conditionは InFocus 状態だけども、改めて上書き。
    pdm::NotifyAppletEvent(pdm::AppletEventType::Launch, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(pdm::AppletEventType::InFocus, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

    account::UserHandle userHandle;
    // 書き込み総量がジャーナルサイズを十分に超過するようループ。
    // 1ループで Open　と Close の 2 イベント追加しているので、おおよそジャーナルサイズの 2 倍の書き込み。
    const auto LoopCount = 1 + pdm::detail::SystemSaveDataJournalSize / sizeof(pdm::PlayEvent);
    for(int i = 0; i < LoopCount; i++ )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(OpenUserPrivate(&userHandle, uid));
        CloseUserPrivate(userHandle);

        auto statistics = pdm::QueryPlayStatistics(SelfApplicationId, uid);
        NN_UNUSED(statistics);
        if( g_QueryThreadStopEvent.TryWait() )
        {
            break;
        }
    }
    auto statistics = pdm::QueryPlayStatistics(SelfApplicationId, uid);
    ASSERT_TRUE(statistics);
    ASSERT_GE((*statistics).totalPlayCount, LoopCount);
    DumpPlayStatistics((*statistics));
}

TEST_F(StressTest, PararellAccountOpenCloseAndQuery)
{
    int actualLength;
    account::Uid uid;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    // AccountPlayEventFactory::Conditionは、InFocus 状態だけども、改めて上書き。
    pdm::NotifyAppletEvent(pdm::AppletEventType::Launch, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(pdm::AppletEventType::InFocus, SelfApplicationId, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

    os::ThreadType accountOpenCloseThread;
    NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&accountOpenCloseThread, &AccountOpenCloseThreadFunc, &uid, g_ThreadStackForAccount, sizeof(g_ThreadStackForAccount), os::DefaultThreadPriority));
    os::SetThreadName(&accountOpenCloseThread, "Account");
    os::StartThread(&accountOpenCloseThread);

    os::ThreadType queryStatisticsThread;
    NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&queryStatisticsThread, &QueryPlayStatisticsThreadFunc, &uid, g_ThreadStackForQuery, sizeof(g_ThreadStackForQuery), os::DefaultThreadPriority));
    os::SetThreadName(&queryStatisticsThread, "Query");
    os::StartThread(&queryStatisticsThread);

    // 適当な期間回す。
    os::SleepThread(TimeSpan::FromMinutes(10));

    g_AccountThreadStopEvent.Signal();
    g_QueryThreadStopEvent.Signal();

    os::DestroyThread(&queryStatisticsThread);
    os::DestroyThread(&accountOpenCloseThread);

    NN_LOG("Exit from StressTest.AccountAndQuery\n");
}

TEST_F(StressTest, DeleteUserInSuspendAndResume)
{
    // Suspend -> ユーザ削除 -> Resume テスト

    int actualLength;
    account::Uid uid;

    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    pdm::NotifyClearAllEvent();

    // ユーザ削除後すぐに再開
    NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(uid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteUserPrivate(uid, false));
    auto result = pdm::ResumeUserAccountEventService(uid);
    ASSERT_TRUE(pdm::ResultServiceNotAvailable::Includes(result) || pdm::ResultUserNotExist::Includes(result));

    // ユーザを追加
    RegisterTestUsers();
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    ASSERT_TRUE(0 < actualLength);

    // ユーザ削除後レイテンシ考慮して再開
    NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(uid));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteUserPrivate(uid, true));
    ASSERT_TRUE(pdm::ResultServiceNotAvailable::Includes(pdm::ResumeUserAccountEventService(uid)));
}

TEST_F(StressTest, NotifyPlayEventAfterDeleteUser)
{
    // イベント記録 -> ユーザ削除 -> イベント記録 テスト

    int actualLength;
    account::Uid uid;
    account::UserHandle userHandle;
    int startIndex;
    int lastIndex;
    const nn::ncm::ApplicationId TestApplicationId1 = nn::ncm::ApplicationId{ 0x1111111111111111ull };
    const nn::ncm::ApplicationId TestApplicationId2 = nn::ncm::ApplicationId{ 0x2222222222222222ull };

    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    pdm::NotifyClearAllEvent();

    // Launch
    pdm::NotifyAppletEvent(pdm::AppletEventType::Launch, TestApplicationId1, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(pdm::AppletEventType::InFocus, TestApplicationId1, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

    // Open / Close
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenUserPrivate(&userHandle, uid));
    CloseUserPrivate(userHandle);

    // Exit
    pdm::NotifyAppletEvent(pdm::AppletEventType::Background, TestApplicationId1, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(pdm::AppletEventType::Exit, TestApplicationId1, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

    // Delete User
    NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteUserPrivate(uid));

    // Launch
    pdm::NotifyAppletEvent(pdm::AppletEventType::Launch, TestApplicationId2, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(pdm::AppletEventType::InFocus, TestApplicationId2, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

    // Open / Close
    ASSERT_TRUE(account::ResultUserNotExist::Includes(OpenUserPrivate(&userHandle, uid)));

    // Exit
    pdm::NotifyAppletEvent(pdm::AppletEventType::Background, TestApplicationId2, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);
    pdm::NotifyAppletEvent(pdm::AppletEventType::Exit, TestApplicationId2, 0, applet::AppletId_Application, ncm::StorageId::Card, ns::PlayLogPolicy::All);

    pdm::GetAvailablePlayEventRange(&startIndex, &lastIndex);

    auto statistics = pdm::QueryPlayStatistics(TestApplicationId1, uid);
    EXPECT_FALSE(statistics);
    statistics = pdm::QueryPlayStatistics(TestApplicationId2, uid);
    EXPECT_FALSE(statistics);
}

TEST_F(StressTest, PararellNotifyPlayEventAndDeleteUser)
{
    // イベント記録中のユーザ削除テスト

    int actualLength;
    account::Uid uid;
    const int RepeatNum = 100;

    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    pdm::NotifyClearAllEvent();

    // ひたすらイベントを記録し続けるスレッドの起動
    os::ThreadType thread;
    NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&thread, &NotifyPlayEventThreadFunc, &uid, g_ThreadStack, sizeof(g_ThreadStack), os::DefaultThreadPriority));
    os::SetThreadName(&thread, "NotifyPlayEvent");
    os::StartThread(&thread);

    for(int i = 0; i < RepeatNum; i++)
    {
        // ランダムな時間待つ
        os::SleepThread(TimeSpan::FromMilliSeconds(std::rand() % 1000));

        // ユーザを削除する
        DeleteUserPrivate(uid);

        // ユーザを追加
        RegisterTestUsers();
        NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
        ASSERT_TRUE(0 < actualLength);
    }

    // イベント記録を終了
    g_ThreadStopEvent.Signal();
    os::WaitThread(&thread);
    os::DestroyThread(&thread);
}

TEST_F(StressTest, PararellSuspendResume)
{
    // Suspend / Resume のマルチスレッドテスト

    int actualLength;
    account::Uid uid;
    const int RepeatNum = 100;

    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&actualLength, &uid, 1));
    if( actualLength == 0 )
    {
        NN_LOG("No account is registered. Skip this test.\n");
        return;
    }

    pdm::NotifyClearAllEvent();
    {
        // 非サスペンド中に pdm システムセーブデータがマウント出来ない事の確認。
        const fs::UserId userId = fs::ConvertAccountUidToFsUserId(uid);
        const auto result = fs::MountSystemSaveData("testPdm", fs::SystemSaveDataId{0x80000000000000f0llu}, userId);
        NN_LOG("[testPdm] Checks whether it can be mounted on the savedata of `pdm` in resumed => result[ 0x%lx ].\n", result.GetInnerValueForDebug());
        EXPECT_TRUE(result.IsFailure());
        if (result.IsSuccess())
        {
            fs::Unmount("testPdm");
        }
    }

    // 別のスレッドで Suspend / Resume
    os::ThreadType thread;
    NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&thread, &SuspendResumeThreadFunc, &uid, g_ThreadStack, sizeof(g_ThreadStack), os::DefaultThreadPriority));
    os::SetThreadName(&thread, "SuspendResume");
    os::StartThread(&thread);

    // メインスレッドで Suspend / Resume
    for(int i = 0; i < RepeatNum; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(uid));

        // サスペンド中に pdm システムセーブデータをマウント出来る事の確認。
        const fs::UserId userId = fs::ConvertAccountUidToFsUserId(uid);
        const auto result = fs::MountSystemSaveData("testPdm", fs::SystemSaveDataId{0x80000000000000f0llu}, userId);
        NN_LOG("[testPdm] Checks whether it can be mounted on the savedata of `pdm` in suspending => result[ 0x%lx ].\n", result.GetInnerValueForDebug());
        NNT_EXPECT_RESULT_SUCCESS(result);
        if (result.IsSuccess())
        {
            fs::Unmount("testPdm");
        }

        // ランダムな時間待つ
        os::SleepThread(TimeSpan::FromMilliSeconds(std::rand() % 1000));

        NN_ABORT_UNLESS_RESULT_SUCCESS(pdm::ResumeUserAccountEventService(uid));

        // ランダムな時間待つ
        os::SleepThread(TimeSpan::FromMilliSeconds(std::rand() % 1000));
    }

    // スレッドを終了
    g_ThreadStopEvent.Signal();
    os::WaitThread(&thread);
    os::DestroyThread(&thread);
}
