﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>

#include "testPdm_Common.h"

using namespace nn;
using namespace nn::pdm;

namespace nnt { namespace pdm {

namespace {

uint32_t Hi(uint64_t value) NN_NOEXCEPT
{
    return static_cast<uint32_t>((value >> 32) & 0xffffffff);
}

uint32_t Low(uint64_t value) NN_NOEXCEPT
{
    return static_cast<uint32_t>((value >> 0) & 0xffffffff);
}

PlayEvent MakePlayEventCommon(PlayEventCategory category, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent playEvent;
    memset(&playEvent, 0, sizeof(PlayEvent));
    playEvent.eventCategory = category;
    playEvent.userTime = userClockTime;
    playEvent.networkTime = networkClockTime;
    playEvent.steadyTime = steadyClockTime;
    return playEvent;
}

} // ~nnt::pdm::<anonymous>

PlayEvent MakeAppletEvent(AppletEventType eventType, const nn::ncm::ProgramId& programId, uint32_t version, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::Applet, userClockTime, networkClockTime, steadyClockTime);
    pe.appletEventData.eventType = eventType;
    pe.appletEventData.programIdHi = Hi(programId.value);
    pe.appletEventData.programIdLow = Low(programId.value);
    pe.appletEventData.version = version;
    pe.appletEventData.appletId = static_cast<uint8_t>(appletId);
    pe.appletEventData.storageId = storageId;
    pe.appletEventData.playLogPolicy = logPolicy;
    return pe;
}

PlayEvent MakeLibraryAppletEvent(AppletEventType eventType, const nn::ncm::ProgramId& mainProgramId, nn::applet::LibraryAppletMode libraryAppletMode, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::Applet, userClockTime, networkClockTime, steadyClockTime);
    pe.appletEventData.eventType = eventType;
    pe.appletEventData.programIdHi = Hi(mainProgramId.value);
    pe.appletEventData.programIdLow = Low(mainProgramId.value);
    pe.appletEventData.laInfo.version = 1;
    pe.appletEventData.laInfo.libraryAppletMode = static_cast<Bit8>(libraryAppletMode);
    pe.appletEventData.appletId = static_cast<uint8_t>(appletId);
    pe.appletEventData.storageId = storageId;
    pe.appletEventData.playLogPolicy = logPolicy;
    return pe;
}

PlayEvent MakeUserAccountEvent(UserAccountEventType eventType, const nn::account::Uid& user, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    NN_SDK_ASSERT(eventType != UserAccountEventType::NetworkServiceAccountAvailable);
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::UserAccount, userClockTime, networkClockTime, steadyClockTime);
    pe.userAccountEventData.eventType = eventType;
    pe.userAccountEventData.userId0Hi = Hi(user._data[0]);
    pe.userAccountEventData.userId0Low = Low(user._data[0]);
    pe.userAccountEventData.userId1Hi = Hi(user._data[1]);
    pe.userAccountEventData.userId1Low = Low(user._data[1]);
    return pe;
}

PlayEvent MakePowerStateChangeEvent(PowerStateChangeEventType eventType, const time::PosixTime& userClockTime, const time::PosixTime& networkClockTime, int64_t steadyClockTime) NN_NOEXCEPT
{
    PlayEvent pe = MakePlayEventCommon(PlayEventCategory::PowerStateChange, userClockTime, networkClockTime, steadyClockTime);
    pe.powerStateChangeEventData.eventType = eventType;
    return pe;
}

PlayEvent MakeAppletEventInMinutes(AppletEventType eventType, const nn::ncm::ProgramId& programId, uint32_t version, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy,
    int64_t userClockTimeValue, int64_t networkClockTimeValue, int64_t steadyClockTime) NN_NOEXCEPT
{
    return MakeAppletEvent(eventType, programId, version, appletId, storageId, logPolicy,
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(userClockTimeValue),
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(networkClockTimeValue),
        steadyClockTime * 60);
}

PlayEvent MakeLibraryAppletEventInMinutes(AppletEventType eventType, const nn::ncm::ProgramId& mainProgramId, nn::applet::LibraryAppletMode libraryAppletMode, nn::applet::AppletId appletId, nn::ncm::StorageId storageId, nn::ns::PlayLogPolicy logPolicy,
    int64_t userClockTimeValue, int64_t networkClockTimeValue, int64_t steadyClockTime) NN_NOEXCEPT
{
    return MakeLibraryAppletEvent(eventType, mainProgramId, libraryAppletMode, appletId, storageId, logPolicy,
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(userClockTimeValue),
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(networkClockTimeValue),
        steadyClockTime * 60);
}

PlayEvent MakeUserAccountEventInMinutes(UserAccountEventType eventType, const nn::account::Uid& user,
    int64_t userClockTimeValue, int64_t networkClockTimeValue, int64_t steadyClockTime) NN_NOEXCEPT
{
    return MakeUserAccountEvent(eventType, user,
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(userClockTimeValue),
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(networkClockTimeValue),
        steadyClockTime * 60);
}

PlayEvent MakePowerStateChangeEventInMinutes(PowerStateChangeEventType eventType,
    int64_t userClockTimeValue, int64_t networkClockTimeValue, int64_t steadyClockTime) NN_NOEXCEPT
{
    return MakePowerStateChangeEvent(eventType,
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(userClockTimeValue),
        time::InputPosixTimeMin + nn::TimeSpan::FromMinutes(networkClockTimeValue),
        steadyClockTime * 60);
}

}} // ~nnt::pdm
