﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/os.h>

#include <nn/init/init_Malloc.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/pcv/pcv.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>

#include <nv/nv_MemoryManagement.h>

int g_StressRuns = 10;

// Contains necessary initialization code, such as setting heap size and initializing PCV

extern "C" int NvOsDrvOpen(const char *pathname, int flags);

namespace {

const size_t TotalHeapSize = 32 * 1024 * 1024;

const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

// After testing, always set the clocks back to safe values
void SetSafeClocks() NN_NOEXCEPT
{
    nn::pcv::SetClockRate(nn::pcv::Module_Cpu, 1224 * 1000 * 1000);
    nn::pcv::SetClockRate(nn::pcv::Module_Gpu, 30720 * 10 * 1000);  // DrvOpen needed
    // nn::pcv::SetClockRate(nn::pcv::Module_Emc, 204 * 1000 * 1000);
}

class PcvEnvironment : public ::testing::Environment
{
public:
    virtual void SetUp() override
    {
        NN_LOG("Initializing Pcv...\n");
        nn::pcv::Initialize();

        NN_LOG("Initialized\n");
    }
    virtual void TearDown() override
    {
        NN_LOG("Finalizing Pcv...\n");
        SetSafeClocks();
        nn::pcv::Finalize();
    }
};

} // namespace

extern "C" void nninitStartup()
{
    nn::Result result;

    /* set heap size */
    result = nn::os::SetMemoryHeapSize(TotalHeapSize);
    if ( !result.IsSuccess() )
    {
        NN_LOG("Failed SetMemoryHeapSize\n");
        return;
    }
    NN_LOG("SetMemoryHeapSize 0x%x OK\n", TotalHeapSize);

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, TotalHeapSize / 2);
    NN_ASSERT(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), TotalHeapSize / 2);

    ::testing::AddGlobalTestEnvironment(new PcvEnvironment);
}

#define ADDITIONAL_SLEEP_FOR_AVOIDING_TIMEOUT

extern "C" void nnMain()
{
#if (defined ADDITIONAL_SLEEP_FOR_AVOIDING_TIMEOUT)
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000LL));
#endif

    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    // Power on GPU
    NvOsDrvOpen("/dev/nvhost-ctrl-gpu", 0);

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    static auto alloc = [](size_t size) -> void* {
        return std::malloc(size);
    };

    static auto dealloc = [](void* p, size_t size) {
        NN_UNUSED(size);
        std::free(p);
    };

    nn::fs::SetAllocator(alloc, dealloc);

    for (int i = 1; i < argc - 1; i++)
    {
        if (strcmp(argv[i], "--stress_runs") == 0)
        {
            g_StressRuns = atoi(argv[i + 1]);
        }
    }

    int result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
