﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include "../Common/ModuleNames.h"
#include "../Common/ModuleInfo.h"

using namespace nnt::pcv;

namespace {

class GetNonDiscreteRates : public ::testing::TestWithParam<nnt::pcv::ModuleInfo>{};
const nnt::pcv::ModuleInfo nonDiscreteRatesModules[] =
    {
        { nn::pcv::Module_I2s1,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2s2,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2s3,              NonDiscrete,      Floor },
        { nn::pcv::Module_Pwm,               NonDiscrete,      Floor },
        { nn::pcv::Module_I2c1,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c2,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c3,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c4,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c5,              NonDiscrete,      Floor },
        { nn::pcv::Module_I2c6,              NonDiscrete,      Floor },
        { nn::pcv::Module_Spi1,              NonDiscrete,      Floor },
        { nn::pcv::Module_Spi2,              NonDiscrete,      Floor },
        { nn::pcv::Module_Spi3,              NonDiscrete,      Floor },
        { nn::pcv::Module_Spi4,              NonDiscrete,      Floor },
        { nn::pcv::Module_Disp1,             NonDiscrete,      Exact },
        { nn::pcv::Module_Sdmmc1,            NonDiscrete,      Floor },
        { nn::pcv::Module_Sdmmc2,            NonDiscrete,      Floor },
        { nn::pcv::Module_Sdmmc3,            NonDiscrete,      Floor },
        { nn::pcv::Module_Sdmmc4,            NonDiscrete,      Floor },
        { nn::pcv::Module_Csite,             NonDiscrete,      Floor },
        { nn::pcv::Module_Hda2codec2x,       NonDiscrete,      Floor },
        { nn::pcv::Module_Actmon,            NonDiscrete,      Floor },
        { nn::pcv::Module_I2cSlow,           NonDiscrete,      Floor },
        { nn::pcv::Module_Sor1,              NonDiscrete,      Floor },
        { nn::pcv::Module_Hda,               NonDiscrete,      Floor },
        { nn::pcv::Module_XusbCoreHostSrc,   NonDiscrete,      Floor },
        { nn::pcv::Module_XusbFalconSrc,     NonDiscrete,      Floor },
        { nn::pcv::Module_XusbFsSrc,         NonDiscrete,      Floor },
        { nn::pcv::Module_XusbCoreDevSrc,    NonDiscrete,      Floor },
        { nn::pcv::Module_XusbSsSrc,         NonDiscrete,      Floor },
        { nn::pcv::Module_UartA,             NonDiscrete,      Floor },
        { nn::pcv::Module_UartB,             NonDiscrete,      Floor },
        { nn::pcv::Module_UartC,             NonDiscrete,      Floor },
        { nn::pcv::Module_UartD,             NonDiscrete,      Floor },
        { nn::pcv::Module_Entropy,           NonDiscrete,      Floor },
        { nn::pcv::Module_SocTherm,          NonDiscrete,      Floor },
        { nn::pcv::Module_AudioUart,         NonDiscrete,      Floor },

        { nn::pcv::Module_Dsi,               NonDiscrete,      Exact},
        { nn::pcv::Module_Maud,              NonDiscrete,      Exact},
        { nn::pcv::Module_UartFstMipiCal,    NonDiscrete,      Exact},
        { nn::pcv::Module_XusbSs,            NonDiscrete,      Exact},
        { nn::pcv::Module_XusbHost,          NonDiscrete,      Exact},
        { nn::pcv::Module_XusbDevice,        NonDiscrete,      Exact},

        { nn::pcv::Module_Extperiph1,        NonDiscrete,      Exact   },
        { nn::pcv::Module_Ahub,              NonDiscrete,      Exact   },
        { nn::pcv::Module_Hda2hdmicodec,     NonDiscrete,      Exact   },

    };

INSTANTIATE_TEST_CASE_P(GetNonDiscreteModules, GetNonDiscreteRates, testing::ValuesIn(nonDiscreteRatesModules));

}

TEST_P(GetNonDiscreteRates, GetPossibleClockRates)
{
    // Pcv library should already be initialized
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::pcv::Module module = GetParam().name;

    nn::pcv::ClockRatesListType listType = nn::pcv::ClockRatesListType_Invalid;
    nn::pcv::ClockHz rates[nn::pcv::MaxNumClockRates];
    memset(rates, 0, sizeof(rates) / sizeof(rates[0]));

    int numRates = 0;

    nn::Result queryResult = nn::pcv::GetPossibleClockRates(
        &listType,
        rates,
        &numRates,
        module,
        nn::pcv::MaxNumClockRates);

    // Expectations:
    // QueryPossibleRates should succeed
    // There should 2 items in rates, [0. MaxRate]
    EXPECT_TRUE(queryResult.IsSuccess()) << "Failed to QueryPossibleRates with module: " << GetModuleName(module);
    EXPECT_LT(0, numRates);
    EXPECT_GE(nn::pcv::MaxNumClockRates, numRates);
    EXPECT_EQ(nn::pcv::ClockRatesListType_Range, listType);

    NN_LOG("Module: %s\n", GetModuleName(module));
    for (int i=0; i < numRates; ++i)
    {
        NN_LOG("\t%.2f MHz\n", rates[i] / (1000.0 * 1000.0));
    }

    ASSERT_EQ(NonDiscrete, GetParam().ratesListType);
    EXPECT_EQ(2, numRates);
    EXPECT_EQ(0, rates[0]);
    EXPECT_LT(0, rates[1]);
}
