﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>

#include <nn/time.h>
#include <nn/time/time_ApiForSystem.h>
#include <nn/time/time_ApiForMenu.h>
#include <nn/time/time_StandardNetworkSystemClock.h>


#include <nnt.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/pcie_LoggedState.h>

namespace {
const bool IsOutputVerbose = true;
nn::fs::FileHandle g_FileHandle;
const ::std::string SdcardMountName = "sd";
int g_WriteOffset = 0;
bool g_IsFileOpen = false;
}


void GetTimeString(::std::string& timeString, bool isFilenameSafe)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;

    const char* colon = isFilenameSafe ? "-" : ":";

    // get time
    ::nn::time::PosixTime posixTime;
    if(::nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime).IsFailure())
    {
        posixTime = nn::time::PosixTime();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
    timeString = ::std::to_string(calendarTime.year)   + colon +
                 ::std::to_string(calendarTime.month)  + colon +
                 ::std::to_string(calendarTime.day)    + "_"   +
                 ::std::to_string(calendarTime.hour)   + colon +
                 ::std::to_string(calendarTime.minute) + colon +
                 ::std::to_string(calendarTime.second);
}


void OpenOutputFile()
{
    ::std::string timeStr;
    nn::Result result;

    g_WriteOffset = 0;

    // resolve filename and path
    GetTimeString(timeStr, true);
    const ::std::string OutFileName = "PcieLoggedState-" + timeStr + ".txt";
    const ::std::string filePath = SdcardMountName + ":/" + OutFileName;
    NN_LOG("FilePath: %s\n", filePath.c_str());

    // mount SD card
    if(nn::fs::MountSdCardForDebug(SdcardMountName.c_str()).IsSuccess())
    {
        ::nn::fs::DirectoryEntryType type;
        result = ::nn::fs::GetEntryType(&type, filePath.c_str());
        if (::nn::fs::ResultPathNotFound().Includes(result))
        {
            result = ::nn::fs::CreateFile(filePath.c_str(), 0);
            NN_ASSERT(result.IsSuccess(), "Failed to create file");
        }
        else
        {
            NN_ASSERT(result.IsSuccess(), "Failed to get entry type");
        }

        // open the file
        result = ::nn::fs::OpenFile(&g_FileHandle, filePath.c_str(), ::nn::fs::OpenMode_Write | ::nn::fs::OpenMode_AllowAppend);
        NN_ASSERT(result.IsSuccess(), "Failed to open log");
        g_IsFileOpen = true;
    }
    else
    {
        g_IsFileOpen = false;
        NN_LOG("SD Card Logging is NOT available.\n");
    }
}

void CloseOutputFile()
{
    g_IsFileOpen = false;

    // close file
    ::nn::fs::CloseFile(g_FileHandle);

    // unmount SD card
    ::nn::fs::Unmount(SdcardMountName.c_str());
}

void MyPrintfCallback(const char *format, ...)
{
    int size;
    char buffer[512] = {0};
    va_list args;
    va_start(args, format);
    size = nn::util::VSNPrintf(buffer, sizeof(buffer), format, args);
    NN_LOG("%s", buffer);
    if(g_IsFileOpen)
    {
        nn::fs::WriteFile(g_FileHandle, g_WriteOffset, buffer, size, ::nn::fs::WriteOption::MakeValue(0));
        nn::fs::FlushFile(g_FileHandle);
        g_WriteOffset += size;
    }
    va_end(args);
}

TEST( PcieServer_Full, Initialize)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::InitializeLoggedState());
    ::nn::time::Initialize();
    OpenOutputFile();

    // Write startup banner to the log
    ::std::string timeString;
    GetTimeString(timeString, false);
    MyPrintfCallback("\n[pcie MonitorLoggedState]\n__________________________ START OF LOG @ %s _____________________\n", timeString.c_str());
}

TEST( PcieServer_Full, MonitorLoggedState)
{
    ::std::string timeString;
    nn::os::SystemEvent loggedStateEvent;
    static uint32_t errorCount = 0;

    // Initialize logged state event
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetLoggedStateEvent(loggedStateEvent.GetBase(),
                                                            nn::os::EventClearMode_ManualClear,
                                                            IsOutputVerbose));

    // forever monitoring and reporting
    do
    {
        bool wasLoggedErrorPending = false;
        nn::pcie::LoggedState loggedState;

        // wait for next event
        if(loggedStateEvent.TimedWait(nn::TimeSpan::FromSeconds(5)))
        {
            GetTimeString(timeString, false);
            loggedStateEvent.Clear();

            // get logged state data
            NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetLoggedState(&loggedState, &wasLoggedErrorPending, false));
            if(wasLoggedErrorPending)
            {
                MyPrintfCallback("\n[pcie MonitorLoggedState]\n!!!!!!!!!!!!!!!!!!!!!!!!! ERROR DETECTED @ %s !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n", timeString.c_str());
                errorCount++;
            }
            else
            {
                MyPrintfCallback("\n[pcie MonitorLoggedState]\n_________________________ PERIODIC STATUS @ %s _____________________________\n", timeString.c_str());
            }

            // print logged state to SD and console
            nn::pcie::PrintLoggedState(loggedState, MyPrintfCallback);
        }
        else
        {
            GetTimeString(timeString, false);
            NN_LOG("[pcie MonitorLoggedState] %d error events detected @ %s\n", errorCount, timeString.c_str());
        }

    }while(NN_STATIC_CONDITION(true));
}

TEST( PcieServer_Full, Finalize)
{
    if(g_IsFileOpen)
    {
        CloseOutputFile();
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::FinalizeLoggedState());
}

