﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nnt.h>

#include <nn/pcie/pcie.h>
#include <nn/pcie/pcie_LoggedState.h>

nn::pcie::ClassDriverHandle g_ClassDriverHandle = nn::pcie::InvalidClassDriverHandle;
nn::os::SystemEventType g_RegistrationEvent;
nn::os::SystemEventType g_LoggedStateEvent;
nn::os::SystemEventType g_PmEvent;
nn::os::SystemEventType g_IrqEvent;
nn::pcie::FunctionState g_FuncStates[nn::pcie::MaxEnumeratedDeviceFunctions];
nn::pcie::FunctionState* g_pFuncUnderTest = NULL;

void MyPrintfCallback(const char *format, ...)
{
    va_list args;
    va_start(args, format);
    NN_VLOG(format, args);
    va_end(args);
}

TEST( PcieServer_Full, Initialize)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::Initialize());
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::InitializeLoggedState());
}

TEST( PcieServer_Full, RegisterClassDriver)
{
    // accept any device for this test
    nn::pcie::ClassDriverConfig cfg = {0};
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::RegisterClassDriver(&g_ClassDriverHandle, &g_RegistrationEvent, &cfg, nn::os::EventClearMode_ManualClear));
}

TEST( PcieServer_Full, IdentifyAndAcquireDeviceFunction)
{
    NN_LOG("Waiting for device function availability...\n");
    nn::os::WaitSystemEvent(&g_RegistrationEvent);
    NN_LOG("GOT EVENT.\n");
    nn::os::ClearSystemEvent(&g_RegistrationEvent);

    int32_t numReturned = 0;
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::QueryFunctions(g_FuncStates, &numReturned, g_ClassDriverHandle,
                                                       sizeof(g_FuncStates)));
    for(int32_t i = 0; i < numReturned; i++)
    {
        nn::pcie::FunctionState *pF = g_FuncStates + i;
        NN_LOG("Query out %d\n",i);
        NN_LOG("  functionHandle = 0x%llx\n", static_cast<uint64_t>(pF->functionHandle));
        NN_LOG("  vendorId       = 0x%x\n", pF->vendorId);
        NN_LOG("  deviceId       = 0x%x\n", pF->deviceId);
        if(pF->isAcquired)
        {
            NN_LOG("There is a problem. Registered QueryFunctions() API should never return info about acquired function!\n");
            ADD_FAILURE();
            return;
        }
        g_pFuncUnderTest = pF;
    }
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    NN_LOG("Trying to acquire functionHandle 0x%llx...\n",g_pFuncUnderTest->functionHandle);
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::AcquireFunction(&g_PmEvent, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,
                                                        nn::os::EventClearMode_ManualClear));
}


TEST( PcieServer_Full, BarProfile)
{
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}
    for(uint8_t bar=0; bar < nn::pcie::StandardConfigMaxBaseAddresses; bar++)
    {
        nn::pcie::BarProfile barProfile;
        if(nn::pcie::GetBarProfile(&barProfile, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, bar).IsSuccess())
        {
            NN_LOG("BAR %d\n",bar);
            NN_LOG("  flags = 0x%x\n", barProfile.flags);
            NN_LOG("  size  = 0x%x\n", barProfile.size);
        }
    }
}

TEST( PcieServer_LoggedState, BarAccessReadFault)
{
    nn::pcie::LoggedState loggedState;
    bool wasLoggedErrorPending = false;
    memset(&loggedState, 0, sizeof(loggedState));
    if(g_pFuncUnderTest==NULL){NN_LOG("No available function found, test skipped.\n");return;}

    // Clear stats
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetLoggedState(&loggedState, &wasLoggedErrorPending, true));

    // Get logged state event, ignoring statistics
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetLoggedStateEvent(&g_LoggedStateEvent, nn::os::EventClearMode_ManualClear, false));

    // Sanity - the tally should be zero now
    nn::pcie::GetLoggedState(&loggedState, &wasLoggedErrorPending, false);
    NN_ABORT_UNLESS(loggedState.rootComplex.globalCounts.afiTargetDecodeReadErrorCount == 0);

    // For now, turn off IO, MEM and bus master capability
    uint32_t originalCommandValue;
    if(nn::pcie::ReadConfig32(&originalCommandValue, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,  nn::pcie::StandardConfigOffset_Command).IsSuccess())
    {
       nn::pcie::WriteConfig32(originalCommandValue & ~(nn::pcie::StandardConfigValue_CommandIo | nn::pcie::StandardConfigValue_CommandMemory),
                               g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,  nn::pcie::StandardConfigOffset_Command);
    }

    for(uint8_t bar=0; bar < nn::pcie::StandardConfigMaxBaseAddresses; bar++)
    {
        nn::pcie::BarProfile barProfile;
        if(nn::pcie::GetBarProfile(&barProfile, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, bar).IsSuccess())
        {
            volatile uint8_t* pBase = nullptr;
            NN_LOG("BAR %d\n",bar);
            NN_LOG("  flags = 0x%x\n", barProfile.flags);
            NN_LOG("  size  = 0x%x\n", barProfile.size);

            if(nn::pcie::GetMappedBar(&pBase, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle, bar).IsSuccess())
            {
                // For a bunch of test iterations
                for(int iteration = 0; iteration < 25; iteration++)
                {
                    // Cause the error
                    *((volatile uint32_t*)pBase);

                    // Wait on the logged state event
                    NN_LOG("Iteration %d...\n",iteration);
                    nn::os::WaitSystemEvent(&g_LoggedStateEvent);
                    nn::os::ClearSystemEvent(&g_LoggedStateEvent);
                }
                break;
            }
        }
    }

    // Restore command
    nn::pcie::WriteConfig32(originalCommandValue, g_ClassDriverHandle, g_pFuncUnderTest->functionHandle,  nn::pcie::StandardConfigOffset_Command);

    // Verify that expected statistics are reported
    nn::pcie::GetLoggedState(&loggedState, &wasLoggedErrorPending, false);
    NN_ABORT_UNLESS(loggedState.rootComplex.globalCounts.afiTargetDecodeReadErrorCount > 0);

    // Print
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::GetLoggedState(&loggedState, &wasLoggedErrorPending, true));
    nn::pcie::PrintLoggedState(loggedState, MyPrintfCallback);
}

TEST( PcieServer_LoggedState, Unregister)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::UnregisterClassDriver(&g_RegistrationEvent, g_ClassDriverHandle));
}

TEST( PcieServer_LoggedState, Finalize)
{
    nn::os::DestroySystemEvent(&g_LoggedStateEvent);
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::FinalizeLoggedState());
    NNT_EXPECT_RESULT_SUCCESS(nn::pcie::Finalize());
}
