﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// Application Control を読み取り、ログに出力するアプリケーション
// 外でそのログを読み取り、期待する値が書かれているかを確認するのが目的
// 文字コード依存は嫌なので、出力はハッシュ

// args: Application Control を読み取りたい対象の ApplicationID

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/crypto.h>

#include <nnt/nnt_Argument.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/settings/system/settings_Language.h>

namespace
{
const int bufferSize = 1 << 20; // 1MB
char s_buffer[bufferSize];

nn::crypto::Sha256Generator s_sha256;
char s_sha256Buffer[nn::crypto::Sha256Generator::HashSize * 2 + 1];

const int logoBufferSize = 4 << 20; // 4MB
char s_logoBuffer[logoBufferSize];

// データを渡すと、その sha256 値の文字列を返す
const char* GetSha256(const void* data, size_t size)
{
    char hash[nn::crypto::Sha256Generator::HashSize];
    s_sha256.Initialize();
    s_sha256.Update(data, size);
    s_sha256.GetHash(hash, sizeof(hash));

    for (int i = 0; i < nn::crypto::Sha256Generator::HashSize; ++i)
    {
        std::snprintf(&(s_sha256Buffer[i * 2]), 3, "%02x", hash[i]);
    }
    s_sha256Buffer[nn::crypto::Sha256Generator::HashSize * 2] = '\0';

    return s_sha256Buffer;
}
const char* GetSha256FromString(const char* str)
{
    auto length = std::strlen(str);
    return GetSha256(reinterpret_cast<const void*>(str), length);
}

void OutputParameter(const char* parameterName, const void* data, size_t size)
{
    NN_LOG("[application control reader] %s: %s\n", parameterName, GetSha256(data, size));
}

void OutputStringParameter(const char* parameterName, const char* value)
{
    NN_LOG("[application control reader] %s: %s\n", parameterName, GetSha256FromString(value));
    NN_LOG("[application control reader] %s (plain): %s\n", parameterName, value);
}

nn::Result ReadApplicationControl(nn::ncm::ApplicationId id)
{
    size_t size;

    nn::ns::InvalidateAllApplicationControlCache();

    {
        const char logoPath[] = "logo.png"; // 現状の logo API は、ロゴのパスを指定させる
        NN_RESULT_DO(
            nn::ns::GetApplicationLogoData(&size, s_logoBuffer, sizeof(s_logoBuffer), id, logoPath)
        );
        OutputParameter("Logo", s_logoBuffer, size);
    }
    {
        nn::settings::system::SetLanguageCode(nn::settings::LanguageCode::Make(nn::settings::Language_AmericanEnglish));
        NN_RESULT_DO(
            nn::ns::GetApplicationControlData(&size, s_buffer, sizeof(s_buffer),
                                              nn::ns::ApplicationControlSource::Storage, id)
        );

        nn::ns::ApplicationControlDataAccessor accessor(s_buffer, size);

        auto& property = accessor.GetProperty();
        auto& title = property.GetDefaultTitle();
        OutputStringParameter("AmericanEnglish::Title", title.name);
        OutputStringParameter("displayVersion", property.displayVersion);
        OutputParameter("AmericanEnglish::Icon", accessor.GetIconData(), accessor.GetIconSize());
    }
    {
        nn::settings::system::SetLanguageCode(nn::settings::LanguageCode::Make(nn::settings::Language_Japanese));
        NN_RESULT_DO(
            nn::ns::GetApplicationControlData(&size, s_buffer, sizeof(s_buffer),
                                              nn::ns::ApplicationControlSource::Storage, id)
        );

        nn::ns::ApplicationControlDataAccessor accessor(s_buffer, size);

        auto& property = accessor.GetProperty();
        auto& title = property.GetDefaultTitle();
        OutputStringParameter("Japanese::Title", title.name);
        OutputStringParameter("displayVersion", property.displayVersion);
        OutputParameter("Japanese::Icon", accessor.GetIconData(), accessor.GetIconSize());
    }

    NN_RESULT_SUCCESS;
}

}

extern "C" void nnMain()
{
    NN_ASSERT(nnt::GetHostArgc() == 2);

    nn::ncm::ApplicationId id;
    auto buffer = nnt::GetHostArgv()[1];
    id.value = std::strtoull(buffer, nullptr, 16);

    nn::ns::Initialize();
    nn::settings::LanguageCode originalLanguage;
    nn::settings::GetLanguageCode(&originalLanguage);

    auto result = ReadApplicationControl(id);
    if(nn::ns::ResultApplicationControlDataNotFound::Includes(result))
    {
        NN_LOG("Control not found: %016llx\n", id.value);
    }
    else if(result.IsFailure())
    {
        NN_LOG("read failed: %08x\n", result.GetInnerValueForDebug());
    }

    // 一応もとに戻そうとする
    // ただし途中で死んだら戻らないのに注意
    nn::settings::system::SetLanguageCode(originalLanguage);

    nn::ns::Finalize();
}

